/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.StreamSupport;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.ResourceNotFoundException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.excalibur.source.Source;

import org.ametys.core.resources.ImageResourceHandler;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Resource handler for pdf resources files 
 */
public class PdfCoverResourceHandler extends ImageResourceHandler implements Contextualizable
{
    private static final Pattern _RESOURCE_PATH_PATTERN = Pattern.compile("^(.*resources(/.+))/([^/]+?)\\.pdf\\.([^/.]+)$");

    /** Attachment component */
    protected ConvertExternalResource2ImagesComponent _externalResourceComponent;

    /** Ametys object resolver */
    protected AmetysObjectResolver _ametysObjectResolver;
    
    private Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _externalResourceComponent = (ConvertExternalResource2ImagesComponent) serviceManager.lookup(ConvertExternalResource2ImagesComponent.ROLE);
        _ametysObjectResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public Source setup(String location, Map objectModel, Parameters par, boolean readForDownload) throws IOException, ProcessingException
    {
        try
        {
            return super.setup(location, objectModel, par, readForDownload);
        }
        catch (ResourceNotFoundException e)
        {
            // The src provided does not point to a direct file, we can process it has a .pdf.* resource
            
            Matcher matcher = _RESOURCE_PATH_PATTERN.matcher(location);
            
            if (!matcher.matches())
            {
                throw new ProcessingException("Invalid resource path when generating resource images", e);
            }
            
            String name = matcher.group(3) + ".pdf";
            String path = matcher.group(2) + "/" + name;
            String uri = matcher.group(1) + "/" + name;
            
            try
            {
                Source resource = _resolver.resolveURI(uri);
                
                Request request = ContextHelper.getRequest(_context);
                String cachePath = _externalResourceComponent.doCache(resource, path, "external-resources", name);
                request.setAttribute(ImagesGenerator.IMAGES_DIRECTORY_PATH_REQUEST_ATTR, cachePath);
                
                // lookup the first page, its actual name depends on the total number of pages
                Path firstPage = null;
                try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(Paths.get(cachePath, "pages"), "page*.png"))
                {
                    firstPage = StreamSupport.stream(directoryStream.spliterator(), false)
                                             .sorted(Path::compareTo)
                                             .findFirst()
                                             .orElseThrow();
                }
                        
                Source src =  _resolver.resolveURI(firstPage.toString());
                
                if (!src.exists())
                {
                    _resolver.release(src);
                    throw new ResourceNotFoundException("Resource not found for URI : " + src.getURI());
                }
                
                _source = src;
                return src;
            }
            catch (Exception ex)
            {
                throw new ProcessingException("Failed to get resource for URI " + location, ex);
            }
        }
    }
}
