/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook.pdfbox;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import javax.imageio.ImageIO;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPageTree;
import org.apache.pdfbox.rendering.ImageType;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.apache.pdfbox.tools.imageio.ImageIOUtil;

import org.ametys.plugins.flipbook.Document2ImagesConvertorPolicy;
import org.ametys.plugins.flipbook.FlipbookException;

import net.coobird.thumbnailator.makers.FixedSizeThumbnailMaker;
import net.coobird.thumbnailator.resizers.DefaultResizerFactory;

/**
 * PDF to PNG convertor which makes use of the pdfbox library.
 * Based on pdfbox's {@link PDFRenderer} utility class, adding the possibility to specify the file name pattern.
 */
public class PdfboxConvertor extends AbstractLogEnabled implements Document2ImagesConvertorPolicy
{
    @Override
    public void convert(File pdfFile, File folder) throws IOException, FlipbookException
    {
        String outputPrefix = "page";
        String imageFormat = "png";
        
        try (PDDocument document = PDDocument.load(pdfFile))
        {
            if (document.isEncrypted())
            {
                throw new IOException("The PDF file is encrypted, cannot read it.");
            }
            
            long start = System.currentTimeMillis();
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("Converting PDF to PNG images using pdfbox.");
            }
            
            writeImages(document, folder, imageFormat, outputPrefix, 120);
            
            // Generate preview
            writePreview(folder, outputPrefix);
            
            long end = System.currentTimeMillis();
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("PDF converted to PNG in " + (end - start) + "ms.");
            }
        }
    }
    
    /**
     * Converts a given page range of a PDF document to bitmap images.
     * @param document the PDF document
     * @param folder the folder where to write
     * @param imageFormat the target format (ex. "png")
     * @param outputPrefix used to construct the filename for the individual images
     * @return true if the images were produced, false if there was an error
     * @throws IOException if an I/O error occurs
     * @throws FlipbookException if an error occurs when manipulating the flipbook
     */
    protected List<String> writeImages(PDDocument document, File folder, String imageFormat, String outputPrefix) throws IOException, FlipbookException
    {
        return writeImages(document, folder, imageFormat, outputPrefix, 96);
    }
    
    /**
     * Converts a given page range of a PDF document to bitmap images.
     * @param document the PDF document
     * @param folder the folder where to write
     * @param imageFormat the target format (ex. "png")
     * @param outputPrefix used to construct the filename for the individual images
     * @param resolution the resolution in dpi (dots per inch)
     * @return true if the images were produced, false if there was an error
     * @throws IOException if an I/O error occurs
     * @throws FlipbookException if an error occurs when manipulating the flipbook
     */
    protected List<String> writeImages(PDDocument document, File folder, String imageFormat, String outputPrefix, int resolution) throws IOException, FlipbookException
    {
        return writeImages(document, folder, imageFormat, "", 1, Integer.MAX_VALUE, outputPrefix, ImageType.RGB, resolution, 1.0f);
    }
    
    /**
     * Converts a given page range of a PDF document to bitmap images.
     * @param document the PDF document
     * @param folder the folder where to write
     * @param imageFormat the target format (ex. "png")
     * @param password the password (needed if the PDF is encrypted)
     * @param startPage the start page (1 is the first page)
     * @param endPage the end page (set to Integer.MAX_VALUE for all pages)
     * @param outputPrefix used to construct the filename for the individual images
     * @param imageType the image type (see {@link BufferedImage}.TYPE_*)
     * @param resolution the resolution in dpi (dots per inch)
     * @param quality the image compression quality (0 &lt; quality &lt; 1.0f).
     * @return true if the images were produced, false if there was an error
     * @throws IOException if an I/O error occurs
     * @throws FlipbookException if an error occurs when manipulating the flipbook
     */
    protected List<String> writeImages(PDDocument document, File folder, String imageFormat, String password, int startPage, int endPage, String outputPrefix, ImageType imageType, int resolution, float quality) throws IOException, FlipbookException
    {
        List<String> fileNames = new ArrayList<>();
        
        PDPageTree pageTree = document.getPages();
        int pageCount = pageTree.getCount();
        int digitCount = Integer.toString(pageCount).length();
        
        // %03d.png
        String format = "%0" + digitCount + "d." + imageFormat;
        for (int i = startPage - 1; i < endPage && i < pageCount; i++)
        {
            PDFRenderer pdfRenderer = new PDFRenderer(document);
                    
            BufferedImage image = pdfRenderer.renderImageWithDPI(i, resolution, imageType);
            String fileName = outputPrefix + String.format(format, i + 1, imageFormat);
            
            fileNames.add(fileName);
            
            File imageFile = new File(folder, fileName + ".part");
            
            try (OutputStream os = new FileOutputStream(imageFile))
            {
                if (!ImageIOUtil.writeImage(image, imageFormat, os, resolution, quality))
                {
                    throw new FlipbookException("Unable to write PDF page " + i + " to " + imageFile.getAbsolutePath());
                }
            }

            imageFile.renameTo(new File(folder, fileName));
        }
        
        return fileNames;
    }

    public List<String> getSupportedMimeTypes()
    {
        List<String> mimeTypeSupported = new ArrayList<>();
        mimeTypeSupported.add("application/pdf");
        
        return mimeTypeSupported;
    }
    
    /**
     * Generate the preview image
     * @param folder The folder with the PDF images
     * @param outputPrefix The prefix of PDF images
     * @throws IOException if an I/O error occurs
     */
    protected void writePreview(File folder, String outputPrefix) throws IOException
    {
        File[] images = folder.listFiles(new FileFilter() 
        {
            public boolean accept(File pathname)
            {
                return pathname.getName().startsWith(outputPrefix);
            }
        });
        
        // Compute preview dimension from first image
        File firstImage = images[0];
        BufferedImage bfi = ImageIO.read(firstImage);
        double imgRatio = (double) bfi.getHeight() / bfi.getWidth();
        
        int rows = (int) Math.ceil(1 + images.length / 2);
        
        int imgWidthInPreview = 56;
        int imgHeightInPreview = (int) Math.round(imgWidthInPreview * imgRatio);
        
        // Create a image on 2 columns 
        BufferedImage result = new BufferedImage(imgWidthInPreview * 2, imgHeightInPreview * rows, BufferedImage.TYPE_INT_RGB);
        Graphics g = result.getGraphics();

        int x = 0;
        int y = 0;
        int count = 0;
        for (File image : images)
        {
            BufferedImage bi = ImageIO.read(image);
            BufferedImage ri = _resizeImage(bi, imgHeightInPreview, imgWidthInPreview);
            g.drawImage(ri, x, y, null);
            x += imgWidthInPreview;
            if (count == 0 || x >= result.getWidth())
            {
                x = 0;
                y += ri.getHeight();
            }
            count++;
        }
        
        ImageIO.write(result, "jpg" , new File(folder, "preview.jpg"));
    }
    
    private static BufferedImage _resizeImage(BufferedImage src, int maxHeight, int maxWidth)
    {
        int srcHeight = src.getHeight();
        int srcWidth = src.getWidth();
        
        int destHeight = 0;
        int destWidth = 0;
        
        boolean keepAspectRatio = true;
        
        if (srcHeight <= maxHeight && srcWidth <= maxWidth || destHeight == srcHeight && destWidth == srcWidth)
        {
            // the source image is already smaller than the destination box or already the good format, don't change anything
            return src;
        }
        
        destWidth = maxWidth;
        destHeight = maxHeight;
        
        Dimension srcDimension = new Dimension(srcWidth, srcHeight);
        Dimension thumbnailDimension = new Dimension(destWidth, destHeight);
        
        BufferedImage thumbImage = new FixedSizeThumbnailMaker(destWidth, destHeight, keepAspectRatio, true)
                                   .resizer(DefaultResizerFactory.getInstance().getResizer(srcDimension, thumbnailDimension))
                                   .imageType(src.getColorModel().hasAlpha() ? BufferedImage.TYPE_INT_ARGB : BufferedImage.TYPE_INT_RGB)
                                   .make(src); 
        
        return thumbImage;
    }
}
