/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

var AmetysFormsHelper = {
    
    /** The error css class name */
    ERROR_CSS_CLASS_NAME: "invalid",
    
    /** The refreshing css class name */
    FORM_REFRESHING_CSS_CLASS_NAME: "refreshing",
    
    /** The global errors */
    globalErrors: [],
    
    /** The form errors */
    formErrors: [],
    
    /** The form precede pages */
    formPrecedePages: {},
    
    /**
     * Init the form js
     * @param {String} id the unique id 
     */
    initForm: function(id)
    {
        AmetysFormsHelper.formPrecedePages[id] = [];
        AmetysFormsHelper.handleSubmitButtons(id, 1);
        
        FormListenersHelper.addListenerForInputs(id, AmetysFormsHelper.handleSubmitButtons);
    },
    
    /**
     * End the form
     * @param {String} id the unique id 
     * @param {Number} pos the page position
     */
    endForm: function(id, pos)
    {
        AmetysFormsHelper.formErrors[id] = [];
        
        eval('checkForm_' + id + '_' + pos + '()');
        if (AmetysFormsHelper.formErrors[id].length > 0)
        {
            AmetysFormsHelper._showFormErrors(id, pos);
            return;
        }
        
        var iframe = $j("#iframe-" + id);
        if (iframe.length == 0)
        {
            iframe = $j('<iframe name="iframe-' + id + '" frameborder="0" scrolling="no" style="display:none" id="iframe-' + id + '"></iframe>').appendTo('body');
            iframe.on('load', function() {
                if (window.event && window.event.srcElement.readyState && !/loaded|complete/.test(window.event.srcElement.readyState))
                {
                    return;
                }
                
                var doc = this.contentWindow.document;
                if (doc.title != 'Ametys-form-iframe-submit')
                {
                    alert("{{i18n PLUGINS_FORMS_IFRAME_ERROR}}");
                }
            });
        }
        
        var form = $j("#form-" + id + "-form");
        form.prop('target', 'iframe-' + id);
        
        AmetysFormsHelper.refreshing(id, pos);
        
        // Submit form
        form.submit();
    },
    
    /**
     * Function call before form submission, set form in refreshing state
     * @param {String} id The unique di
     * @param {Number} pos the page position
     */
    refreshing: function(id, pos)
    {
        this._disableSubmitButton(id, pos, true);
        
        $j("#form-" + id + "-form").addClass(AmetysFormsHelper.FORM_REFRESHING_CSS_CLASS_NAME);
    },
    
    /**
     * Function call when form was submitted. Stop the refreshing state of the form
     * @param {String} id The unique di
     * @param {Number} pos the page position
     */
    stopRefreshing: function(id, pos)
    {
        this._disableSubmitButton(id, pos, false);
        
        $j("#form-" + id + "-form").removeClass(AmetysFormsHelper.FORM_REFRESHING_CSS_CLASS_NAME);
    },
    
    /**
     * Disable or enable the submit button
     * @param {String} id the unique id 
     * @param {Number} pos the page position
     * @param {Boolean} disable true to disable the submit button. Otherwise, enable the submit button.
     */
    _disableSubmitButton: function(id, pos, disable)
    {
        var submitButton = $j("#submit-" + pos + "-" + id);
        submitButton.prop( "disabled", disable);
        submitButton.css('opacity', disable ? '0.5' : '1');
    },
    
    /**
     * Handle submit buttons
     * @param {String} id the unique id 
     * @param {Number} pos the page position
     */
    handleSubmitButtons: function(id, pos)
    {
        // Branches 
        var branch = eval('branches_' + id + '_' + pos + '()');
        if (branch == 0) // End page
        {
            $j("#submit-" + pos + "-" + id).parent().show();
            $j("#form-next-" + pos + "-" + id).parent().hide();
            $j("#buttons-" + pos + "-" + id).show();
            $j("#captcha-div-" + pos + "-" + id).append($j("#captcha-" + id))
            $j("#captcha-div-" + pos + "-" + id).show();
        }
        else
        {
            $j("#submit-" + pos + "-" + id).parent().hide();
            $j("#form-next-" + pos + "-" + id).parent().show();
            $j("#buttons-" + pos + "-" + id).show();
            $j("#captcha-div-" + pos + "-" + id).hide();
        }
    },
    
    
    /**
     * Go to previous page
     * @param {String} id the unique id 
     * @param {Number} pos the page position
     */
    goToPreviousPage: function(id, pos)
    {
        AmetysFormsHelper.resetFormErrors(id);
    
        var precedePage;
        if (AmetysFormsHelper.formPrecedePages[id].length > 0)
        {
            precedePage = AmetysFormsHelper.formPrecedePages[id][AmetysFormsHelper.formPrecedePages[id].length -1];
            // Remove last page
            AmetysFormsHelper.formPrecedePages[id].splice(AmetysFormsHelper.formPrecedePages[id].length -1, 1);
        }
        else
        {
            precedePage = pos - 1;
        }
        
        $j("#form-" + id + "-page-" + pos).hide();
        $j("#form-" + id + "-page-" + precedePage).show();
        $j("#submit-" + pos + "-" + id).parent().hide();
        $j("#captcha-div-" + pos + "-" + id).hide();
        
        $j(window).scrollTop($j("#form-" + id + "-page-" + precedePage).position().top);
    },
    
    /**
     * Go to next page
     * @param {String} id the unique id 
     * @param {Number} pos the page position
     * @param {Number} last the last page position
     */
    goToNextPage: function(id, pos, last)
    {
        AmetysFormsHelper.formErrors[id] = [];
        
        eval('checkForm_' + id + '_' + pos + '()');
        if (AmetysFormsHelper.formErrors[id].length > 0)
        {
            AmetysFormsHelper._showFormErrors(id, pos);
            return;
        }
        
        // Branches 
        var branch = eval('branches_' + id + '_' + pos + '()');
        if (branch == -1)
        {
            // No branch
            next = pos + 1;
        }
        else if (branch == 0)
        {
            // Finish form
            next = last;
        }
        else
        {
            next = branch;
        }
        
        $j("#form-" + id + "-page-" + pos).hide();
        $j("#form-" + id + "-page-" + next).show();
        $j("#buttons-" + next + "-" + id).show();
        
        if (next == last)
        {
            $j("#submit-" + next + "-" + id).parent().show();
            $j("#captcha-div-" + next + "-" + id).append($j("#captcha-" + id))
            $j("#captcha-div-" + next + "-" + id).show();
        }
        else
        {
            $j("#submit-" + next + "-" + id).parent().hide();
            $j("#captcha-div-" + next + "-" + id).hide();
        }
        
        $j(window).scrollTop($j("#form-" + id + "-page-" + next).position().top);
        
        AmetysFormsHelper.formPrecedePages[id].push(pos);
        AmetysFormsHelper.handleSubmitButtons(id, next);
    },
    
    /**
     * Add global form error
     * @param {String} id the unique id 
     * @param {String} message the message of the error
     */
    addGlobalFormError: function(id, message)
    {
        AmetysFormsHelper.globalErrors[id].push({message: message});
    },
    
    /**
     * Add form error
     * @param {String} id the unique id 
     * @param {String} fieldId the field id on error
     * @param {String} label the label of the error
     * @param {String} message the message of the error
     */
    addFormError: function(id, fieldId, label, message)
    {
        var parentField = $j('#' + fieldId).parents('div.field');
        parentField.addClass(AmetysFormsHelper.ERROR_CSS_CLASS_NAME);
        parentField.find('> div.label').addClass(AmetysFormsHelper.ERROR_CSS_CLASS_NAME);
        parentField.find('> div.input').addClass(AmetysFormsHelper.ERROR_CSS_CLASS_NAME);
        $j('#' + fieldId).attr("aria-invalid", true);
        
        AmetysFormsHelper.formErrors[id].push({label: label, message: message});
    },
    
    /**
     * Reset the form errors
     * @param {String} id the unique id 
     */
    resetFormErrors: function(id)
    {
        $j('#form-' + id).find('.error').remove();
        $j('#form-' + id).find('div.' + AmetysFormsHelper.ERROR_CSS_CLASS_NAME).removeClass(AmetysFormsHelper.ERROR_CSS_CLASS_NAME);
        $j('#form-' + id).find('fieldset.' + AmetysFormsHelper.ERROR_CSS_CLASS_NAME).removeClass(AmetysFormsHelper.ERROR_CSS_CLASS_NAME);
        $j('#form-' + id).find('*[aria-invalid]').attr("aria-invalid", true);   
        
        AmetysFormsHelper.formErrors[id] = [];
        AmetysFormsHelper.globalErrors[id] = [];
    },
    
    /**
     * Get regexp error msg
     * @param {String} type the regexp type of simple text
     */
    getRegexpErrorMsg: function(type)
    {
        var msg = "{{i18n PLUGINS_FORMS_FORM_ERROR_TEXT_REGEXP}}";
        if (type == 'email')
        {
            msg = "{{i18n PLUGINS_FORMS_FORM_ERROR_TEXT_EMAIL}}";
        }
        else if (type == 'phone')
        {
            msg = "{{i18n PLUGINS_FORMS_FORM_ERROR_TEXT_PHONE}}";
        }
        return msg;
    },
    
    /**
     * Listener on matrix change
     * @param {String} fieldId the matrix field id
     * @param {String} rowName the row name
     * @param {String} rowValue the row value
     * @param {String} colValue the column value
     * @param {Boolean} mandatory true if the field is mandatory
     */
    onFormMatrixChange: function(fieldId, rowName, rowValue, colValue, mandatory)
    {
        var jsonVal = {};
        
        // First get matrix old values
        var oldVal = $j('#' + fieldId).val();
        if (oldVal != null && oldVal != '')
        {
            jsonVal = JSON.parse(oldVal);
        }    
    
        var colValues = [];
        
        // Then, get all matrix new values
        $j("input[name = '" + rowName + "']").each(function() {
            if ($j(this).is(":checked"))
            {
            	var value = $j(this).val()
                // If the case is checked and the old value is already checked, we want to uncheck it only if it is not mandatory
		        if (value == colValue && !mandatory && jsonVal[rowValue] && (jsonVal[rowValue].indexOf(colValue) != -1))
		        {
		        	$j(this).prop('checked', false);
		        }
		        else // else just add the value
		        {
		        	colValues.push(value);
		        }
            }
        });
    
        // And update the matrix values with new values
        jsonVal[rowValue] = colValues;
        
        var newVal = JSON.stringify(jsonVal);
        $j('#' + fieldId).val(newVal);
    },
    
    /**
     * Listener on select other option
     * @param {Object} select the select DOM or its id for jquery selection (starting with #)
     */
    onFormOtheroptSelect: function(select)
    {
        var otherOptVal = "__internal_other";
        
        var val = $j(select).val();
        if ((Array.isArray(val) && val.indexOf(otherOptVal) != -1) || !Array.isArray(val) && val == otherOptVal)
        {
            $j(select).closest(".field").find(".field-other").show();
        }
        else
        {
            $j(select).closest(".field").find(".input-other").val("");
            $j(select).closest(".field").find(".field-other").hide()
        }
    },
    
    /**
     * Listener on key up for other option
     * @param {String} input the other input
     * @param {String} id the unique id 
     */
    onFormOtheroptKeyup: function(input, id)
    {
        var checked = input.value != '';
        $j('#' + id).get(0).checked = checked;
    },
    
    /**
     * Hide/show the rule target depending on current source value
     * @param {Object} uniqueId id of the form
     * @param {Object} ruleTargetName name of the target
     * @param {Object} ruleAction the rule action (hide or show)
     * @param {Boolean} isRuleTriggered true if source value equals rule value
     * @param {Boolean} canRead true if the question can be read
     * @param {Boolean} isPreview true if it's the preview mode
     */
    onFormRuleActionChange: function(uniqueId, ruleTargetName, ruleAction, isRuleTriggered, canRead, isPreview)
    {
        var target = $j('#form-' + uniqueId + '-q-' + ruleTargetName);
        
        if (!canRead)
        {
            AmetysFormsHelper._hide(target, isPreview);
        }
        else if (isRuleTriggered)
        {
            if (ruleAction == 'HIDE')
            {
                AmetysFormsHelper._hide(target, isPreview);
            }
            else
            {
                AmetysFormsHelper._show(target, isPreview);
            }
        }
        else
        {
            if (ruleAction == 'HIDE')
            {
                AmetysFormsHelper._show(target, isPreview);
            }
            else
            {
                AmetysFormsHelper._hide(target, isPreview);
            }
        }
    },
    
    /**
     * Hide/show the restriction target depending on current source value
     * @param {Object} uniqueId id of the form
     * @param {Object} targetName name of the target
     * @param {Boolean} canRead true if the question can be read
     * @param {Boolean} canWrite true if the question can be writed
     * @param {Boolean} isPreview true if it's the preview mode
     */
    handleRestrictingReading: function(uniqueId, targetName, canRead, canWrite, isPreview)
    {
        var div = $j('#form-' + uniqueId + '-q-' + targetName);
        if (!canRead)
        {
            AmetysFormsHelper._hide(div, isPreview);
        }
        if (!canWrite)
        {
            var input = $j('#' + targetName + '_' + uniqueId);
            AmetysFormsHelper._disable(div, input, isPreview);
        }
    }, 
    
    /**
     * Hide the div 
     * @param {Object} div the div to hide 
     * @param {Boolean} isPreview true if is preview
     */
    _hide: function(div, isPreview)
    {
        if (isPreview)
        {
            div.addClass("preview-hide");
        }
        else
        {
            div.hide();
        }
    },
    
    /**
     * Show the div 
     * @param {Object} div the div to show 
     * @param {Boolean} isPreview true if is preview
     */
    _show: function(div, isPreview)
    {
        if (isPreview)
        {
            div.removeClass("preview-hide");
        }
        else
        {
            div.show();
        }
    },
    
    /**
     * Hide the div 
     * @param {Object} div the div to hide 
     * @param {Boolean} isPreview true if is preview
     */
    _disable: function(div, input, isPreview)
    {
        if (isPreview)
        {
            div.addClass("preview-disable");
        }
        else
        {
            input.prop('disabled', true);
        }
    },
    
    /**
     * Show the div 
     * @param {String} id the unique id
     * @param {Number} pos the page position
     */
    _showFormErrors: function(id, pos)
    {
        $j('#form-' + id + '-page-' + pos).before('<div class="error"></div>');
        var $errors = $j("#form-" + id + " .error");
        
        if (AmetysFormsHelper.globalErrors[id].length)
        {
            for (var i=0; i < AmetysFormsHelper.globalErrors[id].length; i++)
            {
                $errors.append(`<p>${AmetysFormsHelper.globalErrors[id][i].message}</p>`);
            }
        }
        
        if (AmetysFormsHelper.formErrors[id].length)
        {
            $errors.append(`<p>{{i18n PLUGINS_FORMS_PAGE_FORM_ERRORS}}</p><ul></ul>`);
            for (var i=0; i < AmetysFormsHelper.formErrors[id].length; i++)
            {
                var label = AmetysFormsHelper.formErrors[id][i].label;
                if (label != null)
                {
                    $errors.find("ul").append('<li>' + label + " : " + AmetysFormsHelper.formErrors[id][i].message + '</li>');
                }
                else
                {
                    $errors.find("ul").append('<li>' + AmetysFormsHelper.formErrors[id][i].message + '</li>');
                }
            }
        }
        
        $j(window).scrollTop($errors.position().top);
        $j("#form-" + id + " .error").attr("tabindex", "0");
        $j("#form-" + id + " .error").focus();
        
        if (window.parent != window)
        {
            // Form is in a iframe (in popup), post message to parent to scroll to top             
            window.parent.postMessage({type: "ametys-forms", action: "scrollTop", formId: id})
        }
    },
    
    /**
     * Remove the file value 
     * @param {Object} button the button to remove file
     * @param {String} name the name of the input file
     */
    removeFile: function(button, name)
    {
        $j("#" + name + "-info").val("");
        
        var parent = $j(button).closest(".input-file");
        parent.find(".input-field-value").hide();
        parent.find(".input-field").show();
    }
}
