/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.autofill;

import java.util.List;
import java.util.Map;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItem;

/**
 * Interface for creating a new source for autofill in text fields
 */
public interface AutofillSource
{
    /** 
     * Get id of the autofill source
     * @return the id
     */
    public String getId();
    
    /** 
     * Get label of autofill source
     * @return the label
     */
    public I18nizableText getLabel();
    
    /** 
     * Get description of autofill source
     * @return the description 
     */
    public I18nizableText getDescription();
    
    /**
     * Get autofill value
     * @param question the question
     * @return the autofill value.
     */
    public String getAutofillValue(FormQuestion question);

    /**
     * Get the type model items
     * If the return value is null there won't be any processing on the server side
     * @return a list of the model items
     */
    public Map<String, ModelItem> getModelItems();
    
    /**
     * Get the view items for the main tab
     * @return a list of view items
     */
    public List<ViewItem> getViewElements();
    
    /**
     * Sax additional infos
     * @param contentHandler the content handler
     * @param question the question
     * @throws SAXException if an error occurred
     */
    public void saxAdditionalInfos(ContentHandler contentHandler, FormQuestion question) throws SAXException;
    
    /**
     * Return <code>true</code> if the question can be cacheable with this source.
     * @param question the question
     * @return <code>true</code> if the question can be cacheable.
     */
    public boolean isCacheable(FormQuestion question);

}
