/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Class handling the receipt dialog box 
 */
Ext.define('Ametys.plugins.forms.content.ReceiptBox', {
	
	singleton: true,
	
	/**
	 * @private
	 * @property {Function} _cnFn the callback function invoked when the dialog box is closed
	 */
	
	/**
	 * @private
	 * @property {Ametys.window.DialogBox} _box the dialog box
	 */
	
	/**
	 * @private
	 * @property {String} _removeButtonId the id of the remove button
	 */
	
	/**
	 * @private
	 * @property {Ext.form.field.Text} _fieldFrom the 'from' field
	 */
	
	/**
	 * @private
	 * @property {Ext.form.field.ComboBox} _fieldTo the 'to' field
	 */
	
	/**
	 * @private
	 * @property {Ext.form.field.Text} _fieldSubject the 'subject' field
	 */
	
	/**
	 * @private
	 * @property {Ext.form.field.TextArea} _fieldBody the 'body' field
	 */
	
	/**
	 * Create the receipt dialog box
	 * @param {String} from the sender
	 * @param {String} to the receiver
	 * @param {String} subject the subject
	 * @param {String} body the body
	 * @param {Object[]} emailIds the email ids
	 * @param {Function} cb the function to invoke when the dialog box is validated
	 */
	create: function(from, to, subject, body, emailIds, cb)
	{
		this._cbFn = cb;
		
		if (this._box != null)
		{
			// FIXME : Hack because the first time loadData is called, it does not work as expected
			var fieldToStore = this._fieldTo.getStore();
			fieldToStore.removeAll();
			fieldToStore.loadData(emailIds);
		}

		this._initialize(emailIds);
		
		this._fieldFrom.setValue(from != null ? from : '');
		this._fieldSubject.setValue(subject != null ? subject : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_SUBJECT_DEFAULT}}");
		this._fieldBody.setValue(body != null ? body : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_BODY_DEFAULT}}");
		this._fieldTo.setValue(to || emailIds[0][0]);

		Ext.getCmp(this._removeButtonId).setDisabled(to == null);
		
		this._box.show();
	},
	
	/**
	 * Initialize the dialog box
	 * @param {Object[]} emailIds the email ids
	 */
	_initialize: function(emailIds)
	{
		if (this._box != null)
		{
			return;
		}
		
		this._fieldFrom = Ext.create('Ext.form.TextField', {
			 fieldLabel: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_FROM}}",
			 ametysDescription: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_FROM_DESC}}",
			 value: '',
			 validator: function (v)
			 			{
			 				if (/^(([a-zA-Z0-9_\.\-])+\@(([a-zA-Z0-9\-])+\.)+([a-zA-Z0-9]{2,4})+)?$/.test(v))
			 				{
			 					return true;
			 				}
			 				else
			 				{
			 					return "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_FROM_VALID}}";
			 				}
			 			}
		});
		this._fieldTo = Ext.create('Ext.form.field.ComboBox', {
			 fieldLabel: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_TO}}",
			 ametysDescription: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_TO_DESC}}",
			 mode: 'local',
			 forceSelection : true,
			 editable: false,
			 valueField: 'id',
	 		 displayField: 'text',
			 
			 store: Ext.create('Ext.data.Store', 
			 {
		        fields: [
		            'id',
		            'text'
		        ],
		        data: emailIds
			 }),
			 value: ''
		});
		this._fieldSubject = Ext.create('Ext.form.TextField',{
			 fieldLabel: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_SUBJECT}}",
			 ametysDescription: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_SUBJECT_DESC}}",
			 allowBlank: false,
			 value: ''
		});
		this._fieldBody = Ext.create('Ext.form.field.TextArea', {
			 fieldLabel: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_BODY}}",
			 ametysDescription: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_FIELD_BODY_DESC}}",
			 allowBlank: false,
			 height: 240,
			 value: ''
		});
		
		var formPanel = Ext.create('Ext.form.Panel', {
			border: false,
			cls: 'ametys',
			scrollable: true,
			defaults : {
				cls: 'ametys',
				labelWidth: 90,
				labelSeparator: '',
				labelAlign: 'right',
				width: 340,
				anchor:'90%',
				xtype: 'textfield'
			},
			items: [ this._fieldFrom, this._fieldTo, this._fieldSubject, this._fieldBody ]
		});
		
		this._removeButtonId = Ext.id();
		
		this._box = Ext.create('Ametys.window.DialogBox', {
			title: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_TITLE}}",
			iconCls: 'ametysicon-envelope14',
			
			bodyCls: 'ametys',
			scrollable: true,
			maxHeight: 500,
			layout: "fit",
			width: 500,
			
			items: [ formPanel ],
				
			closeAction: 'hide',
			defaultFocus: this._fieldFrom,
			
			referenceHolder: true,
			defaultButton: 'validate',
			
			buttons: [{
				reference: 'validate',
				text : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_OKBUTTON}}",
				handler: function() 
				{ 
		    		if (formPanel.getForm().isValid() && this._cbFn(this._fieldFrom.getValue(), this._fieldTo.getValue(), this._fieldSubject.getValue(), this._fieldBody.getValue()))
		    		{
		    			this._box.hide();
		    		}
				},
				scope: this
		    }, 
		    {
				text : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_CANCELBUTTON}}",
				handler: function() 
				{ 
					this._box.hide(); 
				},
				scope: this
		    }, 
		    {
		    	id: this._removeButtonId,
				text : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_MAILRECEIPT_DIALOG_REMOVEBUTTON}}",
				handler: function() 
				{ 
					this._cbFn(null, null, null, null);
					this._box.hide(); 
				},
				scope: this
		    }]
		});
	}
});
