/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.SimpleMenu;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * This element creates a content button allowing the export of the form data.
 */
public class FormsContentClientSideElement extends SimpleMenu
{
    /** The form properties manager. */
    private FormPropertiesManager _formPropertiesManager;
    
    /** The Ametys object resolver. */
    private AmetysObjectResolver _resolver;

    private ContentHelper _contentHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _formPropertiesManager = (FormPropertiesManager) smanager.lookup(FormPropertiesManager.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
    }
    
    /**
     * Get the forms status of contents
     * @param contentIds the ids of the contents
     * @return the map of form properties of the contents
     */
    @SuppressWarnings("unchecked")
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getStatus(List<String> contentIds)
    {
        Map<String, Object> results = new HashMap<>();
        
        results.put("forms", new ArrayList<>());
        results.put("noright-contents", new ArrayList<>());
        
        for (String contentId : contentIds)
        {
            Content content = _resolver.resolveById(contentId);
            if (_rightManager.currentUserHasReadAccess(content))
            {
                try
                {
                    List<Form> contentForms = _formPropertiesManager.getForms(content);
                    for (Form form : contentForms)
                    {
                        // If the ID or label is blank, the form won't be in the database.
                        if (StringUtils.isNotEmpty(form.getId()) && StringUtils.isNotEmpty(form.getLabel()))
                        {
                            Map<String, String> formParams = new HashMap<> ();
                            formParams.put("id", form.getId());
                            formParams.put("label", form.getLabel());
                            formParams.put("content-id", contentId);
                            formParams.put("content-title", _contentHelper.getTitle(content));
                            
                            List<Map<String, String>> forms = (List<Map<String, String>>) results.get("forms");
                            forms.add(formParams);
                        }
                    }
                }
                catch (FormsException e)
                {
                    getLogger().error("Error getting forms for a content.", e);
                }
            }
            else
            {
                List<String> noRightContents = (List<String>) results.get("noright-contents");
                noRightContents.add(contentId);
            }
        }
        
        return results;
    }
}
