/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.computing;

import java.util.List;
import java.util.Map;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItem;

/**
 * Interface for computing types
 */
public interface ComputingType
{
    /**
     * Get id of the computing type
     * @return the id
     */
    public String getId();
    
    /**
     * Get label of computing type
     * @return the label
     */
    public I18nizableText getLabel();
    
    /**
     * Get description of computing type
     * @return the description
     */
    public I18nizableText getDescription();
    
    /**
     * Get the XSLT file for computing questions.
     * If the return value is null there won't be any processing on the client side
     * @return the address of the XSLT file
     */
    public String getXSLT();
    
    /**
     * <code>true</code> if the computed field has a computed value for server
     * @return <code>true</code> if the computed field has a computed value for server
     */
    public boolean hasComputedValue();
    
    /**
     * Get the computed value for the field
     * @param question the question
     * @param entry the entry
     * @return the computed value
     */
    public Object getComputedValue(FormQuestion question, FormEntry entry);
    
    /**
     * Get the type model items
     * If the return value is null there won't be any processing on the server side
     * @return a map of the model items
     */
    public Map<String, ModelItem> getModelItems();
    
    /**
     * Get the storage type of the question type
     * @param question the question
     * @return the storage type
     */
    public String getStorageType(FormQuestion question);

    /**
     * Get the view items for the main tab
     * @return a list of view items
     */
    public List<ViewItem> getViewElements();

    /**
     * Get field to disable if form is published
     * @return the list of field name
     */
    public List<String> getFieldToDisableIfFormPublished();

    /**
     * Sax additional values
     * @param contentHandler the content handler
     * @param question the question
     * @param entry the entry
     * @throws SAXException if an error occurred
     */
    public void saxAdditionalValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException;

    /**
     * Get the renderer for js
     * @return the js renderer
     */
    public String getJSRenderer();

    /**
     * <code>true</code> if the question can be edited after submission
     * @return <code>true</code> if the question can be edited after submission
     */
    public boolean canEdit();
    
    /**
     * Convert the entry value into a JSON object to use client side
     * @param value the value to convert
     * @param question the form question
     * @param entry the form entry
     * @param modelItem the model item
     * @return The value as JSON
     * @throws Exception if an error occurred
     */
    public Object valueToJSONForClient(Object value, FormQuestion question, FormEntry entry, ModelItem modelItem) throws Exception;

    /**
     * Return <code>true</code> if the computing type can be cacheable.
     * @return <code>true</code> if the computing type can be cacheable.
     */
    public boolean isCacheable();

    /**
     * Get generic data to use to replace the answer of a confidential question
     * @param question the question
     * @return a mapping of the new values and their data path
     */
    public Map<String, Object> getAnonymizedData(FormQuestion question);
}
