/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.types;

import org.ametys.plugins.forms.question.FormQuestionType;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewElement;

/**
 * Provide all the necessary methods to add a autocomplete attribute to a {@link FormQuestionType}
 */
public interface AutocompleteAwareQuestionType
{
    /** Constant for autocomplete attribute. */
    public static final String ATTRIBUTE_AUTOCOMPLETE = "autocomplete";

    /**
     * The type of the input
     */
    public enum InputType
    {
        /** A simple text input */
        TEXT,
        /** A textarea input */
        TEXTAREA,
        /** A date input */
        DATE,
        /** A select input */
        SELECT
    }
    
    /**
     * Get a model item to include in the model return by FormQuestionType.getModel
     * @param inputType the input type
     * @return the item
     */
    public ModelItem getAutocompleteModelItem(InputType inputType);
    
    /**
     * Get a view element to include in the view returned by FormQuestionType.getView
     * @param model the question type model
     * @return the element
     */
    public ViewElement getAutocompleteViewElement(Model model);
    
    /**
     * Get the question description
     * @param question the question
     * @return the description
     */
    public String getAutocomplete(FormQuestion question);
    
    /**
     * Record representing an autocomplete option
     * @param value the value of the option
     * @param label the label of the option
     */
    record AutocompleteOption(String value, I18nizableText label) { /** empty */ }
}
