/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.schedulable;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobExecutionContext;

import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.Form.ExpirationPolicy;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;

/**
 * Schedular to delete or anonymize all expired form entries
 */
public class FormEntriesExpirationSchedulable extends AbstractStaticSchedulable
{
    private AmetysObjectResolver _resolver;
    private FormEntryDAO _formEntryDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _formEntryDAO = (FormEntryDAO) manager.lookup(FormEntryDAO.ROLE);
    }
    
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        String query = QueryHelper.getXPathQuery(null, FormFactory.FORM_NODETYPE, new BooleanExpression(Form.EXPIRATION_ENABLED, true));
        try (AmetysObjectIterable<Form> queryResult = _resolver.query(query))
        {
            for (Form form: queryResult)
            {
                ExpirationPolicy policy = form.getExpirationPolicy();
                
                ZonedDateTime expirationLimit = ZonedDateTime.now().minus(Period.ofMonths((int) form.getExpirationPeriod()));
                Expression dateExpr = new DateExpression(FormEntry.ATTRIBUTE_SUBMIT_DATE, Operator.LT, expirationLimit);
                // Filter already anonymized entries when the policy is to anonymize
                Expression expiredExpr = policy == ExpirationPolicy.ANONYMIZE ? new AndExpression(dateExpr, new NotExpression(new MetadataExpression(FormEntry.ATTRIBUTE_ANONYMIZATION_DATE))) : dateExpr;
                
                List<FormEntry> entries = _formEntryDAO.getFormEntries(form, false, expiredExpr, List.of());
                if (!entries.isEmpty())
                {
                    for (FormEntry entry: entries)
                    {
                        switch (policy)
                        {
                            case DELETE:
                                entry.remove();
                                break;
                            case ANONYMIZE:
                                entry.anonymize();
                                break;
                            default:
                                // not possible
                                getLogger().error("Unsupported expiration policy '{}' for form '{}'. Action is ignored", policy, form.getId());
                                break;
                        }
                        entry.saveChanges();
                    }
                    String category = switch (policy)
                    {
                        case DELETE -> "data.policy.gdpr.remove.form.submissions.expiration";
                        case ANONYMIZE -> "data.policy.gdpr.anonymize.form.submissions.expiration";
                        default -> "data.policy.gdpr.form.submissions.expiration";
                    };
                    ForensicLogger.info(category, Map.of("handled", Long.toString(entries.size()), "form", form.getId(), "formTitle", form.getTitle()), UserPopulationDAO.SYSTEM_USER_IDENTITY);
                }
            }
        }
    }
}
