/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.commons.collections.ListUtils;

import org.ametys.cms.clientsideelement.WorkflowStepsClientSideElement;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;

/**
 * This class creates multiple toggle buttons each representing a workflow step on form entries
 */
public abstract class AbstractFormEntriesWorkflowStepsClientSideElement extends WorkflowStepsClientSideElement
{
    @Override
    protected void _configureWorkflowStep(WorkflowDescriptor workflowDescriptor, Integer stepId, Map<String, Object> stepParameters, Configuration stepConfiguration, List<Integer> allowedActionIds) throws ConfigurationException
    {
        super._configureWorkflowStep(workflowDescriptor, stepId, stepParameters, stepConfiguration, allowedActionIds);
        
        List<Integer> sendMailIds = new ArrayList<>();
        
        List<Integer> currentStepActions = workflowDescriptor.getStep(stepId)
                .getActions()
                .stream()
                .mapToInt(action -> ((ActionDescriptor) action).getId())
                .boxed()
                .collect(Collectors.toList());
        List<Integer> allowedStepActions = ListUtils.intersection(currentStepActions, allowedActionIds);

        if (stepConfiguration != null)
        {
            Configuration stepWorkflowSendMail = stepConfiguration.getChild("send-mail", false);
            if (stepWorkflowSendMail != null)
            {
                sendMailIds.addAll(_configureWorkflowStepActions(workflowDescriptor.getName(), stepId, currentStepActions, allowedStepActions, stepWorkflowSendMail));
            }
            else
            {
                sendMailIds.addAll(allowedStepActions);
            }
        }
        else
        {
            sendMailIds.addAll(allowedStepActions);
        }
        
        stepParameters.put("workflow-send-mail-ids", sendMailIds);
    }
    
    @Override
    protected Script _configureScript(Configuration configuration) throws ConfigurationException
    {
        List<ScriptFile> scriptsImports = _configureImports(configuration.getChild("scripts"));
        List<ScriptFile> cssImports = _configureImports(configuration.getChild("css"));
        
        Script script = new Script(this.getId(), "", scriptsImports, cssImports, new HashMap<>());
        List<ScriptFile> scriptFiles = _getAdditionalScriptFiles();
        script.getScriptFiles().addAll(scriptFiles);
        
        return script;
    }

    /**
     * Get the additional script files to import
     * @return the list of script
     */
    protected abstract List<ScriptFile> _getAdditionalScriptFiles();
    
    @Override
    protected String _getDefaultPluginName()
    {
        return "forms";
    }
    
    @Override
    protected void _additionalMenuItemConfiguration(Configuration itemConf, DefaultConfiguration classConf, int actionId, Map<String, Object> parameters)
    {
        super._additionalMenuItemConfiguration(itemConf, classConf, actionId, parameters);
        
        // Multiselection enabled ?
        DefaultConfiguration multiselectConf = new DefaultConfiguration("selection-enable-multiselection");
        multiselectConf.setValue(itemConf.getChild("menu-" + actionId + "-enable-multiselection").getValueAsBoolean(true));
        classConf.addChild(multiselectConf);

        // Send mail
        _addActionSendMailConfiguration(itemConf, classConf, actionId, parameters);
    }
    
    /**
     * Add the send mail configuration
     * @param itemConf The item configuration
     * @param classConf The class configuration
     * @param actionId The workflow action id
     * @param parameters The script parameters
     */
    @SuppressWarnings("unchecked")
    protected void _addActionSendMailConfiguration(Configuration itemConf, DefaultConfiguration classConf, int actionId, Map<String, Object> parameters)
    {
     // Send mail
        DefaultConfiguration sendMailConf = new DefaultConfiguration("send-mail");
        sendMailConf.setValue(itemConf.getChild("menu-" + actionId + "-send-mail").getValueAsBoolean(((List<Integer>) parameters.get("workflow-send-mail-ids")).contains(actionId)));
        classConf.addChild(sendMailConf);
        
        // Dialog title for the mails
        if (itemConf.getChild("menu-" + actionId + "-dialog-title", false) != null)
        {
            DefaultConfiguration dialogTitleConf = new DefaultConfiguration("dialog-title");
            dialogTitleConf.setAttribute("i18n", itemConf.getChild("menu-" + actionId + "-dialog-title").getAttribute("i18n", "true"));
            dialogTitleConf.setValue(itemConf.getChild("menu-" + actionId + "-dialog-title").getValue(null));
            classConf.addChild(dialogTitleConf);
        }
        
        // i18n key for the subject  
        if (itemConf.getChild("menu-" + actionId + "-subject-key", false) != null)
        {
            DefaultConfiguration subjectKeyConf = new DefaultConfiguration("subject-key");
            subjectKeyConf.setAttribute("i18n", itemConf.getChild("menu-" + actionId + "-subject-key").getAttribute("i18n", "true"));
            subjectKeyConf.setValue(itemConf.getChild("menu-" + actionId + "-subject-key").getValue(null));
            classConf.addChild(subjectKeyConf);
        }
        
        // i18n key for the body
        if (itemConf.getChild("menu-" + actionId + "-body-key", false) != null)
        {
            DefaultConfiguration bodyKeyConf = new DefaultConfiguration("body-key");
            bodyKeyConf.setAttribute("i18n", itemConf.getChild("menu-" + actionId + "-body-key").getAttribute("i18n", "true"));
            bodyKeyConf.setValue(itemConf.getChild("menu-" + actionId + "-body-key").getValue(null));
            classConf.addChild(bodyKeyConf);
        }
    }

    @Override
    protected void _configureDefaultDescriptions(Map<String, Object> parameters)
    {
        // One entry selected
        if (!parameters.containsKey("entryselected-description"))
        {
            parameters.put("entryselected-description", new I18nizableText("plugin.forms", "ENTRY_WORKFLOW_DESCRIPTION"));
        }
        
        // Several entries selected
        if (!parameters.containsKey("several-entryselected-description"))
        {
            parameters.put("several-entryselected-description", new I18nizableText("plugin.forms", "ENTRIES_WORKFLOW_DESCRIPTION"));
        }
        
        // Empty selection
        if (!parameters.containsKey("selection-description-empty"))
        {
            parameters.put("selection-description-empty", new I18nizableText("plugin.forms", "ENTRY_WORKFLOW_NOSELECTION_DESCRIPTION"));
        }
        
        // No match selection
        if (!parameters.containsKey("selection-description-nomatch"))
        {
            parameters.put("selection-description-nomatch", new I18nizableText("plugin.forms", "ENTRY_WORKFLOW_NOMATCH_DESCRIPTION"));
        }
        
        // Multiselection forbidden
        if (!parameters.containsKey("selection-description-multiselectionforbidden"))
        {
            parameters.put("selection-description-multiselectionforbidden", new I18nizableText("plugin.forms", "ENTRY_WORKFLOW_NOMULTISELECT_DESCRIPTION"));
        }
        
        // No available action
        if (!parameters.containsKey("noaction-available-description"))
        {
            parameters.put("noaction-available-description", new I18nizableText("plugin.forms", "ENTRY_WORKFLOW_NOACTIONAVAILABLE_DESCRIPTION"));
        }
        
        // Refreshing
        if (!parameters.containsKey("refreshing-description"))
        {
            parameters.put("refreshing-description", new I18nizableText("plugin.forms", "ENTRY_WORKFLOW_REFRESH_DESCRIPTION"));
        }
    }

    /**
     * Get the workflow state of contents
     * @param entryIds The ids of the selected entries
     * @param formId The id of the form
     * @param scriptId The id of the script
     * @return The workflow state as a map
     */
    @Callable (rights = FormEntryDAO.HANDLE_FORMS_ENTRIES_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 1)
    public abstract Map<String, Object> getWorkflowState(List<Object> entryIds, String formId, String scriptId);
    
}
