/*
 * Copyright 2016 Anyware Services
 * 
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.rights.FormsDirectoryRightAssignmentContext;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper component allowing to retrieve information in order to send emails
 */
public abstract class AbstractMailInformationHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user manager */
    protected UserManager _userManager;
    
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
    }
    
    /**
     * Get information on the given entry of the given form
     * @param formId the id of the form
     * @param entryId the id of the entry
     * @return a map of information on the current user and on the email fields filled in the entry
     */
    @Callable (rights = FormEntryDAO.HANDLE_FORMS_ENTRIES_RIGHT_ID, rightContext = FormsDirectoryRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getMailInfo(String formId, Object entryId)
    {
        Map<String, Object> result = new HashMap<> ();
        Map<String, Object> sendMail = new HashMap<> ();
        List<Map<String, Object>> emails = new ArrayList<> ();
        
        // Retrieve the current user name and email
        UserIdentity userIdentity = _currentUserProvider.getUser();
        User user = _userManager.getUser(userIdentity.getPopulationId(), userIdentity.getLogin());
        if (user != null)
        {
            Map<String, Object> currentUser = new HashMap<> ();
            currentUser.put("email", user.getEmail());
            currentUser.put("fullname", user.getFullName());
            
            sendMail.put("current-user", currentUser);
            
            for (MailInfo mailInfo : _getMailInfos(formId, entryId))
            {
                emails.add(mailInfo.toJson());
            }
        }
        
        sendMail.put("emails", emails);
        result.put("send-mail", sendMail);
        return result;
    }
    
    /**
     * Get mail infos
     * @param formId the form id
     * @param entryId the entry id
     * @return the list of mail info
     */
    protected abstract List<MailInfo> _getMailInfos(String formId, Object entryId);
    
    /**
     * Class representing mail info
     */
    public static class MailInfo
    {
        private String _email;
        private String _displayValue;
    
        /**
         * The constructor
         * @param email the email
         * @param displayValue the display value
         */
        public MailInfo(String email, String displayValue)
        {
            this._email = email;
            this._displayValue = displayValue;
        }
        
        /**
         * Get the email
         * @return the email
         */
        public String getEmail()
        {
            return this._email;
        }
        
        /**
         * Get display value
         * @return the display value
         */
        public String getDisplayValue()
        {
            return this._displayValue;
        }
        
        /**
         * The mail info to json
         * @return the mail info to json
         */
        public Map<String, Object> toJson()
        {
            Map<String, Object> email = new HashMap<> ();
            
            email.put("displayValue", this._displayValue);
            email.put("value", this._email);
            
            return email;
        }
    }
}
