/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Class handling the limit dialog box 
 */
Ext.define('Ametys.plugins.forms.content.LimitBox', {
	
	singleton: true,
	
	/**
	 * @private
	 * @property {Function} _cnFn the callback function invoked when the dialog box is closed
	 */
	
	/**
	 * @private
	 * @property {Ametys.window.DialogBox} _box the dialog box
	 */
	
	/**
	 * @private
	 * @property {String} _removeButtonId the id of the remove button
	 */
    
    /**
     * @private
     * @property {Ext.form.field.Number} _fieldLimit the 'limit' field
     */
    
    /**
     * @private
     * @property {Ext.form.field.Text} _fieldPlaces the 'places' field
     */
	
	/**
	 * @private
	 * @property {Ext.form.field.Text} _fieldNoPlaces the 'no places' field
	 */
	
	/**
	 * @private
	 * @property {Ext.form.field.TextArea} _fieldBody the 'body' field
	 */
	
	/**
	 * Create the limit dialog box
     * @param {Number} limit the limit of entries
     * @param {String} places the message to display if there are remaining places.
     * @param {String} noPlaces the message to display if there are no remaining places left.
	 * @param {Function} cb the function to invoke when the dialog box is validated
	 */
	create: function(limit, places, noPlaces, cb)
	{
		this._cbFn = cb;
		
		this._initialize();
		
		this._fieldLimit.setValue(limit != null ? limit : 10);
		this._fieldPlaces.setValue(places != null ? places : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_REMAINING_PLACES_DEFAULT}}");
		this._fieldNoPlaces.setValue(noPlaces != null ? noPlaces : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_NO_REMAINING_PLACES_DEFAULT}}");

		Ext.getCmp(this._removeButtonId).setDisabled(limit == null);
		
		this._box.show();
	},
	
	/**
	 * Initialize the dialog box
	 */
	_initialize: function()
	{
		if (this._box != null)
		{
			return;
		}

        this._fieldLimit = Ext.create('Ext.form.NumberField', {
             xtype: 'numberfield',
             fieldLabel: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_LIMIT}} *",
             ametysDescription: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_LIMIT_DESC}}",
             allowBlank: false,
             minValue: 1
        });

        this._fieldPlaces = Ext.create('Ext.form.TextField', {
             fieldLabel: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_REMAINING_PLACES}}",
             ametysDescription: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_REMAINING_PLACES_DESC}}"
        });
        
        this._fieldNoPlaces = Ext.create('Ext.form.TextField', {
            fieldLabel: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_NO_REMAINING_PLACES}}",
            ametysDescription: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_FIELD_NO_REMAINING_PLACES_DESC}}"
       });
		
		var formPanel = Ext.create('Ext.form.Panel', {
			border: false,
			cls: 'ametys',
            layout: 'anchor',
			defaults : {
				cls: 'ametys',
				labelWidth: 190,
                labelSeparator: '',
                labelAlign: 'top',
                anchor:'100%',
				xtype: 'textfield'
			},
			items: [ this._fieldLimit, this._fieldPlaces, this._fieldNoPlaces]
		});
		
		this._removeButtonId = Ext.id();
		
		this._box = Ext.create('Ametys.window.DialogBox', {
			title: "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_TITLE}}",
			iconCls: 'ametysicon-body-idcard-badge',
			
			bodyCls: 'ametys',
			scrollable: true,
			maxHeight: 500,
			layout: "fit",
			width: 500,
			
			items: [ formPanel ],
				
			closeAction: 'hide',
			defaultFocus: this._fieldFrom,
			
			referenceHolder: true,
			defaultButton: 'validate',
			
			buttons: [{
				reference: 'validate',
				text : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_OKBUTTON}}",
				handler: function() 
				{ 
		    		if (formPanel.getForm().isValid() && this._cbFn(this._fieldLimit.getValue(), this._fieldPlaces.getValue(), this._fieldNoPlaces.getValue()))
		    		{
		    			this._box.hide();
		    		}
				},
				scope: this
		    }, 
		    {
				text : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_CANCELBUTTON}}",
				handler: function() 
				{ 
					this._box.hide(); 
				},
				scope: this
		    }, 
		    {
		    	id: this._removeButtonId,
				text : "{{i18n PLUGINS_FORMS_FORMS_EDITOR_LIMIT_DIALOG_REMOVEBUTTON}}",
				handler: function() 
				{ 
					this._cbFn(null, null, null);
					this._box.hide(); 
				},
				scope: this
		    }]
		});
	}
});
