/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class controls a ribbon button that is able to check for conditions on a form element
 * @private
 */
Ext.define('Ametys.plugins.forms.controllers.FormController', {
    extend: 'Ametys.ribbon.element.ui.ButtonController',
    
	/**
	 * @cfg {Object} togglestate-target-parameter Use this configuration in addition to #toggle-enabled in order to be more specific. This allow to check a target parameter.
	 * @cfg {String} togglestate-target-parameter.name The name of the parameter to check for toggle state
	 * @cfg {String} togglestate-target-parameter.value The value of the parameter to check for toggle state
	 */
			
    constructor: function (config)
    {
        this.callParent(arguments);
		
		// Has an associated target-parameter check for toggle state ?
		var targetParameter = this.getInitialConfig("togglestate-target-parameter");
		if (targetParameter)
		{
			this._toggleStateTargetParameterName = targetParameter.name;
			this._toggleStateTargetParameterValue = new RegExp(targetParameter.value);;
		}
        
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
    },
    
    /**
     * Listener on modified message.
     * Update the state of the controller accordingly.
     * @param {Ametys.message.Message} message the message of type modified.
     * @private
     */
    _onModified: function(message)
    {
        if (this.updateTargetsInCurrentSelectionTargets(message))
        {
            this.refresh();
        }
    },
    
    updateState: function ()
    {
        this.refreshing();
        this.disable();
            
        this._getStatus(this.getMatchingTargets());
    },
    
    /**
     * Refresh the controller from the form informations state of given targets
     * @param {Ametys.message.MessageTarget[]} targets The form targets
     * @protected
     */
    _getStatus: function (targets)
    {
        var description = this.getInitialConfig("description") || this.getInitialConfig("default-description") || '';
        
        var enabledOnPrivateOnly = this.getConfig("enable-on-private-only") == "true";
        var enabledOnPublishedOnly = this.getConfig("enable-on-published-only") == "true";
        var enabledOnUnpublishedOnly = this.getConfig("enable-on-unpublished-only") == "true";
        var enabledOnNoEntryOnly = this.getConfig("enable-on-noentry-only") == "true";
        var enabledOnEntryOnly = this.getConfig("enable-on-entry-only") == "true";
        
        // Multiselection is currently not supported
        var matchingTarget = targets[0];
        
		this._hasError = false;
        description = this._updateTooltipDescription(description, matchingTarget.getParameters());
        
		this._updateToggleState(matchingTarget.getParameters());
		
		if (this._hasError)
		{
			this.disable();
		}
		else
		{
			this.enable();
		}
        
        this.stopRefreshing();
        this.setDescription (description);
    },
    
	/**
	 * @protected
	 * Update the tooltip description according state of the current selection
	 * @param description The initial description. Can be empty.
	 * @param params The target parameters
	 */
	_updateTooltipDescription: function(description, params)
	{
		var enabledOnPrivateOnly = this.getInitialConfig("enable-on-private-only") == "true";
        var enabledOnPublishedOnly = this.getInitialConfig("enable-on-published-only") == "true";
        var enabledOnUnpublishedOnly = this.getInitialConfig("enable-on-unpublished-only") == "true";
        var enabledOnNoEntryOnly = this.getInitialConfig("enable-on-noentry-only") == "true";
        var enabledOnEntryOnly = this.getInitialConfig("enable-on-entry-only") == "true";
		
		var errorDescription = "";
		        
        if (enabledOnPrivateOnly && params.isAnonymous)
        {
            description = this._addTooltipDescription(description, "noprivate");
			this._hasError = true;
        }
        if (enabledOnPublishedOnly && !params.isPublished)
        {
			description = this._addTooltipDescription(description, "unpublished");
            this._hasError = true;
        }
        if (enabledOnUnpublishedOnly && params.isPublished)
        {
			description = this._addTooltipDescription(description, "published");
			this._hasError = true;
        }
        if (enabledOnNoEntryOnly && params.hasEntries)
        {
			description = this._addTooltipDescription(description, "has-entries");
			this._hasError = true;
        }
        if (enabledOnEntryOnly && !params.hasEntries)
        {
			description = this._addTooltipDescription(description, "noentry");
			this._hasError = true;
        }
		
		return description;
	},
	
	/**
	 * @private
	 */
	_addTooltipDescription: function(description, prefix)
	{
		if (description != "")
		{
			description += "<br/><br/>";
		}
		description += this.getInitialConfig(prefix + "-description");
		return description;
	},
	
	/**
	 * @protected
	 * Update the toggle state according state of the current selection
	 * @param params The target parameters
	 */
	_updateToggleState: function(params)
	{
		if (this._toggleEnabled && this._toggleStateTargetParameterName)
		{
			var isToggled = this._toggleStateTargetParameterValue.test(params[this._toggleStateTargetParameterName]);
			
			this.toggle(isToggled);
			
			if (isToggled)
			{
				this.setAdditionalDescription(this.getConfig("toggle-on-description") || '');
				this.setIconDecorator(this.getConfig("toggle-on-decorator") || '');
			}
			else
			{
				this.setAdditionalDescription(this.getConfig("toggle-off-description") || '');
				this.setIconDecorator(this.getConfig("toggle-off-decorator") || '');
			}
		}
	},
	
    /**
     * Get an additional description to display when the button is enabled
     * @param {Ametys.message.MessageTarget} matchingTarget The matching target
     * @return {String} teh additional description
     */
    getAdditionnalDescription: function(matchingTarget)
    {
        return "";
    },
    
    /**
     * @protected
     * Get an additional error not handled by default form controller
     * @param {Ametys.message.MessageTarget} matchingTarget The matching target
     * @return {String} additional error or empty is there is no error for current selection
     */
    getErrorDescription: function(matchingTarget)
    {
        return "";
    }
});
