/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The form directory DAO
 */
Ext.define('Ametys.plugins.forms.dao.FormDirectoryDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDirectoryDAO
         * @method getFormDirectoryProperties 
         * Gets form directory information.
         * This calls the method 'getFormDirectoryProperties' of the server DAO 'FormDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String[]} parameters.id The id of forms to retrieve
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object[]} callback.forms The form's properties. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object[]} callback.notAllowedForm The unauthorized forms
         * @param {String[]} callback.unknownForms The unknown forms
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDirectoryDAO",
                methodName: "getFormDirectoryProperties",
                convertor: this._convertFormDirectoryProperties,
                waitMessage: false,
                errorMessage: {
                    msg: "{{i18n DAOS_FORM_REQUEST_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDirectoryDAO
         * @method createFormDirectory 
         * Creates a new form directory
         * This calls the method 'createFormDirectory' of the server DAO 'FormDirectoryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.parentId The id of the parent
         * @param {String} parameters.name The desired name for the new form directory
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDirectoryDAO",
                methodName: "createFormDirectory",
                callback: {
                    handler: this._createFormDirectoryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_DIRECTORY_CREATION_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDirectoryDAO
         * @method renameFormDirectory 
         * Renames a form directory
         * This calls the method 'renameFormDirectory' of the server DAO 'FormDirectoryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form directory
         * @param {String} parameters.newName The new name of the directory
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDirectoryDAO",
                methodName: "renameFormDirectory",
                callback: {
                    handler: this._renameFormDirectoryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_RENAMING_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDirectoryDAO
         * @method mustWarnBeforeDeletion 
         * Determines if application must warn before deleting the given form directories
         * This calls the method 'mustWarnBeforeDeletion' of the server DAO 'FormDirectoryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.ids The ids of the form directories
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Boolean} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDirectoryDAO",
                methodName: "mustWarnBeforeDeletion",
                callback: {
                    handler: Ext.emptyFn
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDirectoryDAO
         * @method canDeleteFormDirectories 
         * Determines if the current user can delete the given form directories
         * This calls the method 'canDeleteFormDirectories' of the server DAO 'FormDirectoryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.ids The ids of the form directories
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDirectoryDAO",
                methodName: "canDeleteFormDirectories",
                callback: {
                    handler: Ext.emptyFn
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDirectoryDAO
         * @method deleteFormDirectory 
         * Deletes form directories
         * This calls the method 'deleteFormDirectory' of the server DAO 'FormDirectoryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.ids The ids of the form directories to delete
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDirectoryDAO",
                methodName: "deleteFormDirectory",
                callback: {
                    handler: this._deleteDirectoryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_DELETE_DIRECTORY_ERROR}}",
                    category:Ext.getClassName(this)
                },
                localParamsIndex: 1
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormDirectoryDAO
         * @method moveFormDirectory 
         * Moves a form directory
         * This calls the method 'moveFormDirectory' of the server DAO 'FormDirectoryDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the form directory
         * @param {String} parameters.newParentId The id of the new parent directory of the form directory
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormDirectoryDAO",
                methodName: "moveFormDirectory",
                callback: {
                    handler: this._moveFormDirectoryCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_MOVE_FAILED}}",
                    category:Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * Callback function called after {@link #createFormDirectory} has been processed.
     * @param {Object} response The server response
     * @param {String} response.id The form directory id
     * @param {String} response.name The form directory name
     * @private
     */
    _createFormDirectoryCb: function(response)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: {
                id: Ametys.message.MessageTarget.FORM_DIRECTORY,
                parameters: {
                    id: response.id
                }
            }
        });
    },
    
    /**
     * Callback function called after {@link #renameFormDirectory} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _renameFormDirectoryCb: function(response)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_FORMS_NOT_ALLOWED_TITLE}}",
                msg: "{{i18n PLUGINS_FORMS_NOT_ALLOWED_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else if (response.message == 'cannot-rename')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_FORMS_CANNOT_RENAME_TITLE}}",
                msg: "{{i18n PLUGINS_FORMS_CANNOT_RENAME_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING
            });
        }
        else
        {   
            var formId = response.id;
                    
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_DIRECTORY,
                    parameters: {
                        id: formId
                    }
                }
            });
        }
    },
    
    /**
     * Callback function called after {@link #deleteFormDirectory} has been processed.
     * @param {Object} response The server response
     * @param {Object} args The callback arguments
     * @param {Object[]} params The callback parameters (server-side and client-side)
     * @private
     */
    _deleteDirectoryCb: function(response, args, params)
    {
        if (response.message == 'not-allowed')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_FORMS_DELETE_DIRECTORY_TITLE}}",
                msg: "{{i18n PLUGINS_FORMS_DELETE_DIRECTORY_NOT_ALLOWED}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.ERROR
            });
        }
        else
        {
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.DELETED,
                targets: params[1] // the deleted form directory targets
            });
        }
    },
    
    /**
     * Callback function called after {@link #moveFormDirectory} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _moveFormDirectoryCb: function(response)
    {
        if (response.message == 'cannot-move')
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_FORMS_CANNOT_MOVE_TITLE}}",
                msg: "{{i18n PLUGINS_FORMS_CANNOT_MOVE_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING
            });
        }
        else
        {   
            var formDirectoryId = response.id;
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MOVED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_DIRECTORY,
                    parameters: {
                        id: formDirectoryId
                    }
                }
            });
        }
    },
    
    /**
     * @private
     * Convert function called after #getFormDirectoryProperties is processed
     * @param {Object} pageProperties The server response
     */
    _convertFormDirectoryProperties: function (formDirectoryProperties)
    {
        return Ext.create ('Ametys.plugins.forms.FormDirectory', formDirectoryProperties);
    }
 });