/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The form question DAO.
 */
 Ext.define('Ametys.plugins.forms.dao.FormQuestionDAO', {
    singleton: true,
    
    constructor: function()
    {
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method getQuestionProperties 
         * Gets properties of a form question
         * This calls the method 'getQuestionProperties' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the question to retrieve
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "getQuestionProperties",
                convertor: this._convertQuestionProperties,
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_QUESTIONS_GET_PROPERTIES_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method createQuestion 
         * Creates a form question.
         * This calls the method 'createQuestion' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.pageId The id of the parent page
         * @param {String} parameters.type The type of question
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "createQuestion",
                callback: {
                    handler: this._createQuestionCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_QUESTIONS_CREATE_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method renameQuestion 
         * Rename a question.
         * This calls the method 'renameQuestion' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the question to rename
         * @param {String} parameters.newName The new title for the question
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "renameQuestion",
                callback: {
                    handler: this._renameQuestionCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_RENAME_PAGE_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        ); 
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method editQuestion 
         * Edits a form question.
         * This calls the method 'editQuestion' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the question to edit
         * @param {String} parameters.type The type of question
         * @param {String} parameters.label The label
         * @param {String} parameters.title The title
         * @param {String} parameters.regexp The regexp type
         * @param {String} parameters.mandatory "true" if the question is mandatory
         * @param {String} parameters.columns The columns
         * @param {String} parameters.options The options
         * @param {String} parameters.otherOption "true" if the question has an "other" option
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "editQuestion",
                callback: {
                    handler: this._editQuestionCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_QUESTIONS_EDIT_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method deleteQuestion 
         * Deletes a form question.
         * This calls the method 'deleteQuestion' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the question to delete
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "deleteQuestion",
                callback: {
                    handler: this._deleteQuestionCb,
                    scope: this
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_QUESTIONS_DELETE_ERROR}}",
                    category: Ext.getClassName(this)
                },
                localParamsIndex: 1
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method copyQuestion 
         * Copies and pastes a form question.
         * This calls the method 'copyQuestion' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.pageId The id of the page, target of the copy
         * @param {String} parameters.questionId The id of the question to copy
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "copyQuestion",
                callback: {
                    handler: this._copyQuestionCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_COPY_QUESTION_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method addPageRule 
         * Adds a new rule to a question.
         * This calls the method 'addPageRule' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the question
         * @param {String} parameters.option The option
         * @param {String} parameters.rule The rule type
         * @param {String} parameters.page The page to jump or skip
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "addPageRule",
                callback: {
                    handler: this._addRuleCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_BRANCHES_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method deletePageRule
         * Deletes a rule to a question. 
         * This calls the method 'deletePageRule' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the question
         * @param {String} parameters.option The option to delete
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "deletePageRule",
                callback: {
                    handler: Ext.emptyFn
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_DELETE_RULE_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.plugins.forms.dao.FormQuestionDAO
         * @method getRules 
         * Gets the rules for a form question.
         * This calls the method 'getRules' of the server DAO 'FormQuestionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the question
         * @param {String} parameters.number The question number
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.forms.dao.FormQuestionDAO",
                methodName: "getRules",
                callback: {
                    handler: Ext.emptyFn
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_FORMS_BRANCHES_ERROR}}",
                    category: Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * Callback function called after {@link #createQuestion} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _createQuestionCb: function(response)
    {
        if (response && !response['errors'])
        {
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.CREATED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_QUESTION,
                    parameters: {
                        id: response.id,
                        pageId: response.pageId,
                        formId: response.formId,
                        type: response.type
                    }
                }
            });
        }
    },
    
    /**
     * Callback function called after {@link #renameQuestion} has been processed.
     * @param {Object} response The server response
     * @private
     */
    _renameQuestionCb: function(response)
    {
        var questionId = response.id,
            questionNewName = response.newName;
                
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.MODIFIED,
            targets: {
                id: Ametys.message.MessageTarget.FORM_QUESTION,
                parameters: {
                    id: questionId,
                    name: questionNewName,
                    pageId: response.pageId,
                    formId: response.formId
                }
            }
        });
    },
    
    /**
     * Callback function called after {@link #editQuestion} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _editQuestionCb: function(response)
    {
        if (response && !response['errors'])
        {
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_QUESTION,
                    parameters: {
                        id: response.id,
                        pageId: response.pageId,
                        formId: response.formId,
                        type: response.type
                    }
                }
            });
            
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_PAGE,
                    parameters: {
                        id: response.pageId,
                        formId: response.formId
                    }
                }
            });
        }
    },
    
    /**
     * Callback function called after {@link #deleteQuestion} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @param {Object[]} params The parameters (server-side and client-side)
     * @param {Object} params[0] the question id
     * @param {Object} params[1] the target to delete
     * @private
     */
    _deleteQuestionCb: function(response, args, params)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.DELETED,
            targets: params[1]
        });
     },
     
     /**
     * Callback function called after {@link #copyQuestion} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
    _copyQuestionCb: function(response)
    {
        Ext.create("Ametys.message.Message", {
            type: Ametys.message.Message.CREATED,
            targets: {
                id: Ametys.message.MessageTarget.FORM_QUESTION,
                parameters: {
                    id: response.id,
                    pageId: response.pageId,
                    formId: response.formId,
                    type: response.type
                }
            }
        });
    },
    
    /**
     * Callback function called after {@link #addRule} has been processed.
     * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
     * @private
     */
     _addRuleCb: function(response)
     {
        if (!response['error'])
        {
            Ext.create("Ametys.message.Message", {
                type: Ametys.message.Message.MODIFIED,
                targets: {
                    id: Ametys.message.MessageTarget.FORM_QUESTION,
                    parameters: {
                        id: response.id,
                        pageId: response.pageId,
                        formId: response.formId,
                        type: response.type
                    }
                }
            });
        }
        else
        {
            var error = response['error'];
            if (error == 'already-exists')
            {
                Ametys.Msg.show({
                    title: "{{i18n PLUGINS_FORMS_ADD_RULE_LABEL}}",
                    msg: "{{i18n PLUGINS_FORMS_ADD_RULE_ALREADY_EXISTS}}",
                    buttons: Ext.Msg.OK,
                    icon: Ext.MessageBox.ERROR
                });
            }
        }
     },
     
     /**
     * @private
     * Convert function called after #getQuestionProperties is processed
     * @param {Object} questionProperties The server response
     */
    _convertQuestionProperties: function (questionProperties)
    {
        return Ext.create ('Ametys.plugins.forms.Question', questionProperties);
    }
 });