/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.actions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.forms.dao.FormEntryDAO;
import org.ametys.plugins.forms.dao.FormQuestionDAO.FormEntryValues;
import org.ametys.plugins.forms.helper.FormWorkflowHelper;
import org.ametys.plugins.forms.question.types.impl.FileQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.holder.values.UntouchedValue;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

/**
 * Action to edit the given form entry
 */
public class EditFormEntryAction extends AbstractProcessFormAction
{
    /** The form workflow helper */
    protected FormWorkflowHelper _formWorkflowHelper;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _formWorkflowHelper = (FormWorkflowHelper) smanager.lookup(FormWorkflowHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        try
        {
            // Force the workspace.
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            String entryId = request.getParameter("entryId");
            if (entryId == null)
            {
                throw new IllegalStateException("Impossible to edit entry. No id provided.");
            }
            
            Multimap<String, I18nizableText> formErrors = ArrayListMultimap.create();
            FormEntry entry = _resolver.resolveById(entryId);
            boolean canSubmitterEdit = _canSubmitterEdit(entry);
            if (!canSubmitterEdit && !_canManagerEdit(entry))
            {
                formErrors.put("form-access", new I18nizableText("plugin.forms", "PLUGINS_FORMS_EDIT_FORM_ENTRY_ERRORS"));
                request.setAttribute("form", entry.getForm());
                request.setAttribute("form-errors", formErrors);
                return null;
            }
            
            Optional<Long> currentStepId = Optional.of(_entryDAO.getCurrentStepId(entry));
            
            View view = View.of(entry.getModel());
            
            View filteredEntryView = _getRuleFilteredEntryView(request, entry.getForm(), view, new FormEntryValues(null, entry), currentStepId);
            
            Map<String, Object> values = _foAmetysObjectCreationHelper.getFormValues(request, filteredEntryView, "", formErrors);
            _adaptFormValuesForChoiceList(entry.getForm(), values);
            _adaptFormValuesForFile(request, entry.getForm(), values);
            
            Map<String, Object> additionalParameters = new HashMap<>();
            additionalParameters.put("ignoreWriteRestriction", canSubmitterEdit);

            formErrors.putAll(_foAmetysObjectCreationHelper.validateValues(values, filteredEntryView));
            for (FormQuestion question : entry.getForm().getQuestions())
            {
                if (filteredEntryView.hasModelViewItem(question.getNameForForm()))
                {
                    question.getType().validateEntryValues(question, values, formErrors, currentStepId, additionalParameters);
                }
            }
            
            if (!formErrors.isEmpty())
            {
                request.setAttribute("form", entry.getForm());
                request.setAttribute("form-errors", formErrors);
                return null;
            }
            
            entry.synchronizeValues(filteredEntryView, values);
            _handleComputedValues(entry.getForm().getQuestions(), entry, true);
            
            entry.saveChanges();
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
        
        return EMPTY_MAP;
    }

    @Override
    protected List<FormQuestion> _getRuleFilteredQuestions(Request request, Form form, FormEntryValues entryValues, Optional<Long> currentStepId)
    {
        String isSubmitterAsString = request.getParameter("isSubmitter");
        boolean isSubmitter = StringUtils.isNotBlank(isSubmitterAsString) ? Boolean.valueOf(isSubmitterAsString) : false;
        
        boolean canSubmitterEdit = isSubmitter && _canSubmitterEdit(entryValues.entry());
        boolean onlyWritableQuestion = !canSubmitterEdit; // Get only writable questions if it's not the submitter
        boolean onlyReadableQuestion = canSubmitterEdit; // Get only readable questions if it's the submitter
        
        return _formQuestionDAO.getRuleFilteredQuestions(form, entryValues, currentStepId, onlyWritableQuestion, onlyReadableQuestion)
                .stream()
                .filter(q -> !canSubmitterEdit || _formQuestionDAO.canSubmitterEditSubmission(q)) // If the submitter is editing the entry, check is the question can be edited by the submitter
                .toList();
    }
    
    private void _adaptFormValuesForFile(Request request, Form form, Map<String, Object> values)
    {
        List<FormQuestion> fileQuestions = form.getQuestions()
                .stream()
                .filter(q -> q.getType() instanceof FileQuestionType)
                .toList();
            
        for (FormQuestion question : fileQuestions)
        {
            String nameForForm = question.getNameForForm();
            String fileInfo = request.getParameter(nameForForm + "-info");
            if ("untouched".equals(fileInfo))
            {
                values.put(nameForForm, new UntouchedValue());
            }
        }
    }
    
    private boolean _canManagerEdit(FormEntry entry)
    {
        // Manager can edit if the edit action is available and he has the right to handle the entry
        return _rightManager.currentUserHasRight(FormEntryDAO.HANDLE_FORMS_ENTRIES_RIGHT_ID, entry) == RightResult.RIGHT_ALLOW
                && !_formWorkflowHelper.getAvailableActionsRestrictedToTypes(entry, List.of(FormWorkflowHelper.EDIT_ACTION)).isEmpty();
    }
    
    private boolean _canSubmitterEdit(FormEntry entry)
    {
        // Submitter can edit if the edit-by-submitter action is available and he is the entry submitter
        return _currentUserProvider.getUser().equals(entry.getUser())
                && !_formWorkflowHelper.getAvailableActionsRestrictedToTypes(entry, List.of(FormWorkflowHelper.EDIT_BY_SUBMITTER_ACTION)).isEmpty();
    }
}
