/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.content.data;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.forms.FormsException;
import org.ametys.plugins.forms.content.Field.FieldType;
import org.ametys.plugins.forms.content.Form;
import org.ametys.plugins.forms.content.jcr.FormPropertiesManager;
import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workflow.store.JdbcWorkflowStore;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.spi.Step;

/**
 * Searches the list of forms contained in a given set of contents.
 */
public class FormEntriesGenerator extends ServiceableGenerator
{
    /** The date format. */
    protected static final DateFormat _DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
    
    /** The internationalizable text symbolizing the absence of workflow step */
    protected static final I18nizableText __MESSAGE_NO_STEP = new I18nizableText("plugin.forms", "PLUGINS_FORMS_UITOOL_ENTRY_WORKFLOW_NO_WORKFLOW");
    
    /** The form properties manager. */
    protected FormPropertiesManager _formPropertiesManager;
    
    /** The form data manager. */
    protected FormTableManager _formTableManager;
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
        
    /** Component gathering utility methods for internationalizable text */
    protected I18nUtils _i18nUtils;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _formPropertiesManager = (FormPropertiesManager) serviceManager.lookup(FormPropertiesManager.ROLE);
        _formTableManager = (FormTableManager) serviceManager.lookup(FormTableManager.ROLE);
        _workflowProvider = (WorkflowProvider) serviceManager.lookup(WorkflowProvider.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        String siteName = parameters.getParameter("site", "");
        String formId = parameters.getParameter("id", "");
        int start = parameters.getParameterAsInteger("start", 0);
        int limit = parameters.getParameterAsInteger("limit", Integer.MAX_VALUE);
        
        if (StringUtils.isEmpty(siteName) || StringUtils.isEmpty(formId))
        {
            throw new ProcessingException("The site name and form ID must be provided.");
        }
        
        try
        {
            Form form = _formPropertiesManager.getForm(siteName, formId);
            
            if (form == null)
            {
                throw new ProcessingException("The form of ID '" + formId + " can't be found in the site '" + siteName + "'.");
            }
            
            Map<String, FieldValue> columns = _formTableManager.getColumns(form);
            List<UserEntry> entries = _formTableManager.getSubmissions(form, columns, start, limit, null);
            int totalSubmissions = _formTableManager.getTotalSubmissions(form.getId());
            
            contentHandler.startDocument();
            
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("id", form.getId());
            atts.addCDATAAttribute("label", form.getLabel());
            atts.addCDATAAttribute("total", String.valueOf(totalSubmissions));
            
            XMLUtils.startElement(contentHandler, "form", atts);
            
            XMLUtils.startElement(contentHandler, "columns");
            for (FieldValue column : columns.values())
            {
                atts.clear();
                atts.addCDATAAttribute("id", column.getColumnName());
                atts.addCDATAAttribute("label", column.getField().getLabel());
                atts.addCDATAAttribute("type", column.getField().getType().toString());
                atts.addCDATAAttribute("field-name", column.getField().getName());
                
                Map<String, String> properties = column.getField().getProperties();
                if (properties.containsKey("regexptype"))
                {
                    atts.addCDATAAttribute("regexptype", properties.get("regexptype"));
                }
                
                XMLUtils.createElement(contentHandler, "column", atts);
            }
            XMLUtils.endElement(contentHandler, "columns");
            
            Workflow workflow = _workflowProvider.getExternalWorkflow(JdbcWorkflowStore.ROLE);
            for (UserEntry entry : entries)
            {
                _saxUserEntry(entry, workflow);
            }
            
            XMLUtils.endElement(contentHandler, "form");
            contentHandler.endDocument();
        }
        catch (FormsException e)
        {
            throw new ProcessingException("An error occurred while getting the results of a form.", e);
        }
    }
    
    /**
     * Sax a user entry.
     * @param entry the user entry.
     * @param workflow the workflow component for form entries
     * @throws IOException if an I/O error occurs.
     * @throws SAXException if an error occurs generating the XML.
     * @throws ProcessingException if a processing error occurs.
     */
    protected void _saxUserEntry(UserEntry entry, Workflow workflow) throws IOException, SAXException, ProcessingException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("id", Integer.toString(entry.getId()));
        atts.addCDATAAttribute("submission-date", _DATE_FORMAT.format(entry.getCreationDate()));
        
        // Workflow step
        Integer workflowId = entry.getWorkflowId();
        if (workflowId != null)
        {
            _saxWorkflowStep (atts, workflow, workflowId);
        }
        else
        {
            atts.addCDATAAttribute("workflowStep", _i18nUtils.translate(__MESSAGE_NO_STEP));
        }
        
        XMLUtils.startElement(contentHandler, "entry", atts);
        
        for (FieldValue fdValue : entry.getValues())
        {
            atts.clear();
            atts.addCDATAAttribute("id", fdValue.getColumnName());

            String defaultValue = fdValue.getField().getProperties().get("value");
            if (StringUtils.trimToNull(fdValue.getField().getProperties().get("value")) != null)
            {
                atts.addCDATAAttribute("defaultvalue", defaultValue);
            }
            
            String displayValue = "";
            Object rawValue = fdValue.getValue();
            if (rawValue == null)
            {
                atts.addCDATAAttribute("null", "true");
            }
            else
            {
                displayValue = String.valueOf(rawValue);
                if (fdValue.getField().getType().equals(FieldType.SELECT) || fdValue.getField().getType().equals(FieldType.RADIO))
                {
                    atts.addCDATAAttribute("value", String.valueOf(rawValue));
                    displayValue = _formPropertiesManager.getDisplayValue(fdValue.getField(), String.valueOf(rawValue));
                }
            }
            
            XMLUtils.createElement(contentHandler, "field", atts, displayValue);
        }
        
        XMLUtils.endElement(contentHandler, "entry");
    }
    
    /**
     * SAX the workflow step
     * @param attrs The XML attributes
     * @param workflow the workflow component for form entries
     * @param workflowId The workflow id
     */
    protected void _saxWorkflowStep (AttributesImpl attrs, Workflow workflow, int workflowId)
    {
        String workflowName = workflow.getWorkflowName(workflowId);
        Step step = (Step) workflow.getCurrentSteps(workflowId).iterator().next();
        
        WorkflowDescriptor workflowDescriptor = workflow.getWorkflowDescriptor(workflowName);
        int stepId = step.getStepId();
        attrs.addCDATAAttribute("workflowStepId", String.valueOf(stepId));
        
        StepDescriptor stepDescriptor = workflowDescriptor.getStep(stepId);
        
        I18nizableText workflowStepName = stepDescriptor == null ? __MESSAGE_NO_STEP : new I18nizableText("application", stepDescriptor.getName());
        attrs.addCDATAAttribute("workflowStep", _i18nUtils.translate(workflowStepName));
        
        if ("application".equals(workflowStepName.getCatalogue()))
        {
            attrs.addCDATAAttribute("workflowStepIcon", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-small.png");
        }
        else
        {
            String pluginName = workflowStepName.getCatalogue().substring("plugin.".length());
            attrs.addCDATAAttribute("workflowStepIcon", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-small.png");
        }
    }
    
    

}
