/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.data;

import java.util.Date;
import java.util.Optional;

import org.ametys.core.user.UserIdentity;

/**
 * Object representing an answer with common attribute for each forms
 */
public class Answer
{
    private String _id;
    private Long _number;
    private String _formId;
    private String _formLabel;
    private String _workflowName;
    private Date _creationDate;
    private Integer _workflowId;
    private Boolean _inQueue;
    private UserIdentity _user;
    
    /**
     * The constructor
     * @param id the id of the answer
     * @param number the number of the answer
     * @param formId the form id
     * @param formLabel the form label
     * @param creationDate the creation date
     * @param workflowName the workflow name
     * @param workflowId the workflow id
     * @param user the user
     */
    public Answer(String id, Long number, String formId, String formLabel, Date creationDate, String workflowName, Integer workflowId, UserIdentity user)
    {
        this(id, number, formId, formLabel, creationDate, workflowName, workflowId, null, user);
    }
    
    /**
     * The constructor
     * @param id the id of the answer
     * @param number the number of the answer
     * @param formId the form id
     * @param formLabel the form label
     * @param creationDate the creation date
     * @param workflowName the workflow name
     * @param workflowId the workflow id
     * @param inQueue is the answer is in queue
     * @param user the user
     */
    public Answer(String id, Long number, String formId, String formLabel, Date creationDate, String workflowName, Integer workflowId, Boolean inQueue, UserIdentity user)
    {
        this._id = id;
        this._number = number;
        this._formId = formId;
        this._formLabel = formLabel;
        this._creationDate = creationDate;
        this._workflowName = workflowName;
        this._workflowId = workflowId;
        this._inQueue = inQueue;
        this._user = user;
    }

    /**
     * Get the id of the answer
     * @return the id
     */
    public String getId()
    {
        return this._id;
    }
    
    /**
     * Get the number of the answer
     * @return the number
     */
    public Long getNumber()
    {
        return this._number;
    }
    
    /**
     * Get the form id
     * @return the form id
     */
    public String getFormId()
    {
        return this._formId;
    }
    
    /**
     * Get the form label
     * @return the form label
     */
    public String getFormLabel()
    {
        return this._formLabel;
    }
    
    /**
     * Get the creation date
     * @return the creation date
     */
    public Date getCreationDate()
    {
        return this._creationDate;
    }
    
    /**
     * Get the workflow name
     * @return the workflow workflow
     */
    public String getWorkflowName()
    {
        return this._workflowName;
    }
    
    /**
     * Get the workflow id. Can be null.
     * @return the workflow id. Can be null.
     */
    public Integer getWorkflowId()
    {
        return this._workflowId;
    }
    
    /**
     * <code>true</code> is the answer is in queue
     * @return <code>true</code> is the answer is in queue. Return empty is the answer belong to a form with no queue
     */
    public Optional<Boolean> isInQueue()
    {
        return Optional.ofNullable(this._inQueue);
    }
    
    /**
     * Get the user of the answer
     * @return the user
     */
    public UserIdentity getUser()
    {
        return _user;
    }
}
