/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.generators;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.forms.helper.FormMailHelper;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;

/**
 * Generate the display of an answer to a question type in a mail body
 */
public class FormMailQuestionGenerator extends FormEntryInformationGenerator
{
    @Override
    protected void _checkRights(FormEntry entry)
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Boolean ignoreRight = (Boolean) request.getAttribute(FormMailHelper.IGNORE_RIGHT_KEY);
        
        if (ignoreRight == null || !ignoreRight)
        {
            super._checkRights(entry);
        }
    }
    
    @Override
    protected List<FormQuestion> _getQuestions(FormEntry entry, Optional<Long> currentStepId)
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> params = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        String questionName = (String) params.get("questionName");
        
        return List.of(entry.getForm().getQuestion(questionName));
    }
    
    @Override
    protected void _saxQuestion(FormQuestion question, FormEntry entry) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("type", question.getType().getId());
        attrs.addCDATAAttribute("format", question.getType().getStorageType(question));
        attrs.addCDATAAttribute("label", question.getTitle());
        attrs.addCDATAAttribute("name", question.getNameForForm());
        
        XMLUtils.startElement(contentHandler, "question", attrs);
        question.getType().saxEntryValue(contentHandler, question, entry);
        XMLUtils.startElement(contentHandler, "question-infos");
        question.getType().saxAdditionalInfos(contentHandler, question);
        XMLUtils.endElement(contentHandler, "question-infos");
        XMLUtils.endElement(contentHandler, "question");
    }
}
