/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.question.computing;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ametys.plugins.forms.question.sources.AbstractSourceType;
import org.ametys.plugins.forms.question.sources.ChoiceOption;
import org.ametys.plugins.forms.question.sources.ChoiceOptionWithCost;
import org.ametys.plugins.forms.question.sources.ChoiceSourceType;
import org.ametys.plugins.forms.question.sources.ManualWithCostsSourceType;
import org.ametys.plugins.forms.question.types.impl.ChoicesListQuestionType;
import org.ametys.plugins.forms.repository.Form;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.StaticEnumerator;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Class for creating cost computed field
 */
public class CostComputingType extends AbstractStaticComputingType
{
    /** The currency attribute */
    public static final String ATTRIBUTE_CURRENCY = "currency";
    /** Name of euro currencyStaticEnumerator entry  */
    public static final String EURO_CURRENCY_VALUE = "euro";
    /** Name of pound sterling currencyStaticEnumerator entry  */
    public static final String POUND_CURRENCY_VALUE = "pound";
    /** Name of us dollar currencyStaticEnumerator entry  */
    public static final String DOLLAR_CURRENCY_VALUE = "dollar";
    
    /** Map of ModelItems specific to ManualSourceType */
    protected Map<String, ModelItem> _costComputingItems;

    public Map<String, ModelItem> getModelItems()
    {
        _costComputingItems = new HashMap<>();
        
        ElementDefinition<String> currency = _formElementDefinitionHelper.getElementDefinition(ATTRIBUTE_CURRENCY, ModelItemTypeConstants.STRING_TYPE_ID, "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTING_COST_CURRENCY", "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTING_COST_CURRENCY_DESC", null);
        StaticEnumerator<String> currencyStaticEnumerator = new StaticEnumerator<>();
        currencyStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTING_COST_CURRENCY_EURO"), EURO_CURRENCY_VALUE);
        currencyStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTING_COST_CURRENCY_POUND"), POUND_CURRENCY_VALUE);
        currencyStaticEnumerator.add(new I18nizableText("plugin.forms", "PLUGINS_FORMS_QUESTIONS_DIALOG_COMPUTING_COST_CURRENCY_DOLLAR"), DOLLAR_CURRENCY_VALUE);
        currency.setEnumerator(currencyStaticEnumerator);
        currency.setDefaultValue(EURO_CURRENCY_VALUE);
        _costComputingItems.put(currency.getName(), currency);
       
        return _costComputingItems;
    }

    public List<ViewItem> getViewElements()
    {
        List<ViewItem> viewElements = new ArrayList<>();
        
        ViewElement grid = new ViewElement();
        grid.setDefinition((ElementDefinition< ? >) _costComputingItems.get(ATTRIBUTE_CURRENCY));
        viewElements.add(grid);
        
        return viewElements;
    }

    public Object getComputedValue(FormQuestion computedQuestion, FormEntry entry)
    {
        Form form = entry.getForm();
        Double cost = 0.0;
        for (FormQuestion question : form.getQuestions())
        {
            if (question.getType() instanceof ChoicesListQuestionType)
            {
                String nameForForm = question.getNameForForm();
                ChoiceSourceType sourceType = _choiceSourceTypeExtensionPoint.getExtension(question.getValue(ChoicesListQuestionType.ATTRIBUTE_SOURCE_TYPE));
                if (sourceType instanceof ManualWithCostsSourceType && entry.hasValue(nameForForm))
                {
                    List<String> values = new ArrayList<>();
                    values = (entry.isMultiple(nameForForm))
                            ? List.of(entry.getValue(nameForForm))
                            : Collections.singletonList(entry.getValue(nameForForm));
                    try
                    {
                        Map<String, Object> enumParam = new HashMap<>();
                        enumParam.put(AbstractSourceType.QUESTION_PARAM_KEY, question);
                        Set<ChoiceOption> options = sourceType.getTypedEntries(enumParam).keySet();
                        for (ChoiceOption option : options)
                        {
                            if (values.contains(option.getValue()))
                            {
                                cost += ((ChoiceOptionWithCost) option).getCost();
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        getLogger().error("An error occured while getting choices options", e);
                    }
                }
            }
        }
        return cost;
    }

    public String getStorageType(FormQuestion question)
    {
        return ModelItemTypeConstants.DOUBLE_TYPE_ID;
    }
    
    public boolean canEdit()
    {
        return true;
    }
    
    public Map<String, Object> getAnonymizedData(FormQuestion question)
    {
        return Map.of(question.getNameForForm(), 0.0);
    }
}
