/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.statistics;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.IntStream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Repository;
import javax.jcr.Session;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.jackrabbit.JcrConstants;
import org.slf4j.Logger;

import org.ametys.plugins.forms.content.table.FormTableManager;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugins.admin.statistics.Statistics;
import org.ametys.runtime.plugins.admin.statistics.StatisticsNode;
import org.ametys.runtime.plugins.admin.statistics.StatisticsProvider;
import org.ametys.runtime.plugins.admin.statistics.StatisticsValue;

/**
 * Stats for the "old" content forms
 */
public class ContentFormsStatisticsProvider extends AbstractLogEnabled implements StatisticsProvider, Serviceable, PluginAware
{
    private String _id;
    private FormTableManager _formTableManager;
    private Repository _repository;

    public void service(ServiceManager manager) throws ServiceException
    {
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _formTableManager = (FormTableManager) manager.lookup(FormTableManager.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public Statistics getStatistics()
    {
        List<Integer> formsSubmissions = _getFormsSubmissions();
        int submissions = formsSubmissions.stream().flatMapToInt(IntStream::of).sum();
        
        return new StatisticsNode(
                _id,
                new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_CONTENTFORMS_LABEL"),
                "ametysicon-code-html-form",
                formsSubmissions.size(),
                List.of(
                    new StatisticsNode(
                        "submissions",
                        new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_CONTENTFORMS_SUBMISSIONS_LABEL"),
                        "ametysicon-desktop-school-tool",
                        submissions,
                        List.of(
                            new StatisticsValue(
                                "max",
                                new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_CONTENTFORMS_SUBMISSIONS_MAX_LABEL"),
                                "ametysicon-sort51",
                                formsSubmissions.size() > 0 ? formsSubmissions.get(formsSubmissions.size() - 1) : 0
                            ),
                            new StatisticsValue(
                                "median",
                                new I18nizableText("plugin.forms", "PLUGINS_FORMS_STATISTICS_CONTENTFORMS_SUBMISSIONS_MEDIAN_LABEL"),
                                "ametysicon-maths-window-symbol-x",
                                formsSubmissions.size() > 0 ? formsSubmissions.get(formsSubmissions.size() / 2) : 0
                            )
                        ),
                        false
                    )
                ),
                true
            );
    }
    
    private List<String> _getFormsIds()
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            QueryManager queryManager = session.getWorkspace().getQueryManager();
            @SuppressWarnings("deprecation")
            Query query = queryManager.createQuery("//element(ametys-internal:forms, " + JcrConstants.NT_UNSTRUCTURED + ")/*", Query.XPATH);
            NodeIterator nodeIterator = query.execute().getNodes();
            
            List<String> formsIds = new ArrayList<>();
            while (nodeIterator.hasNext())
            {
                Node node = nodeIterator.nextNode();
                
                formsIds.add(node.getName());
            }
            
            return formsIds;
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Cannot get forms", e);
        }
        finally
        {
            if (session != null)
            {
                session.logout();
            }
        }
    }
    
    private List<Integer> _getFormsSubmissions()
    {
        final Logger logger = getLogger();
        try
        {
            return _getFormsIds().stream()
                .map(id -> {
                    try
                    {
                        return _formTableManager.getTotalSubmissions(id);
                    }
                    catch (Exception e)
                    {
                        
                        logger.warn("An error occurred while retrieving the total submissions for the form " + id + ". It will be ignored.", e);
                        /* ignore */
                        return 0;
                    }
                })
                .sorted()
                .toList();
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Cannot compute content forms statistics", e);
        }
    }
}
