/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.workflow.AmetysObjectCheckRightsCondition;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.workflow.EnhancedCondition;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * Worklow condition for checking if current user is the same user as the entry submmiter
 */
public class IsSubmitterCondition extends AbstractLogEnabled implements EnhancedCondition, Serviceable
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }

    @Override
    public boolean passesCondition(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        FormEntry formEntry = (FormEntry) transientVars.get(AmetysObjectCheckRightsCondition.AMETYS_OBJECT_KEY);
        UserIdentity user = formEntry.getUser();
        
        return user != null && user.equals(_currentUserProvider.getUser());
    }
    
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_IS_SUBMITTER_CONDITION_LABEL");
    }
    
    @Override
    public List<WorkflowVisibility> getVisibilities()
    {
        List<WorkflowVisibility> visibilities = EnhancedCondition.super.getVisibilities();
        visibilities.add(WorkflowVisibility.USER);
        return visibilities;
    }
}
