/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.forms.workflow;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;

import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.plugins.workflow.support.WorkflowHelper.WorkflowVisibility;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

import jakarta.mail.MessagingException;

/**
 * OS workflow function to send mail after an action is triggered.
 * The author of a form entry is also notified if the receipt is set
 */
public class SendMailFunction extends AbstractLogEnabled implements Component, EnhancedFunction
{
    /** Actually send the email ? */
    public static final String SEND_MAIL = "send-request-information-mail";
    
    /** Sender of the email */
    public static final String SENDER = "sender";
    
    /** Recipient of the email */
    public static final String RECIPIENT = "recipient";

    /** Subject of the email */
    public static final String SUBJECT = "subject";
    
    /** Body of the email */
    public static final String BODY = "body";
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        // If "send-mail" is set to true or is not present, send the mail.
        boolean dontSendMail = "false".equals(transientVars.get(SEND_MAIL));
        if (dontSendMail)
        {
            return;
        }
        
        String sender = (String) transientVars.get(SENDER);
        String recipient = (String) transientVars.get(RECIPIENT);
        String subject = (String) transientVars.get(SUBJECT);
        String body = (String) transientVars.get(BODY);
        
        try
        {
            String htmlBody = StandardMailBodyHelper.newHTMLBody()
                .withTitle(subject)
                .withMessage(body.replaceAll("\n\r", "<br/>"))
                .build();
            
            SendMailHelper.newMail()
                          .withAsync(true)
                          .withSubject(subject)
                          .withHTMLBody(htmlBody)
                          .withSender(sender)
                          .withRecipient(recipient)
                          .sendMail();
        }
        catch (MessagingException | IOException e)
        {
            getLogger().warn("Could not send a workflow notification mail to " + recipient, e);
        }
    }
    
    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }
    

    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_SEND_MAIL_FUNCTION_LABEL");
    }
    
    @Override
    public I18nizableText getFullLabel(Map<String, String> argumentsValues)
    {
        return new I18nizableText("plugin.forms", "PLUGINS_FORMS_SEND_MAIL_FUNCTION_DESCRIPTION");
    }
    
    @Override
    public List<WorkflowVisibility> getVisibilities()
    {
        List<WorkflowVisibility> visibilities = EnhancedFunction.super.getVisibilities();
        visibilities.add(WorkflowVisibility.USER);
        return visibilities;
    }
}
