/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
AmetysFrontEditionToolbarHelper = {
    
    KEYCODE: {
      TAB: 9,
      LEFT: 37,
      UP: 38,
      RIGHT: 39,
      DOWN: 40
    },
    
    /**
     * Create a toolbar button
     * @param {Object} config the button configuration
     * @param {String} config.className the button's class name
     * @param {String} config.label the label of button
     * @param {Boolean} config.hideLabel true to hide label
     * @param {String} [config.iconCls] the CSS class for icon
     * @param {String} [config.id] the id of button
     * @param {Function} [config.handler] the function to call on click
     * @param {Boolean} [config.openDialog] set to true if the handler opens a dialog
     * @param {Boolean} [config.pressed] set to true if the button is pressed
     * @param {Boolean} [config.disabled] set to true if the button is disabled
     * @return the button
     */
    createToolbarButton: function(config)
    {
        var icon = config.iconCls ? '<span aria-hidden="true" class="icon ' + config.iconCls + '"></span>' : '';
        var cls = config.className || "";
        if (config.hideLabel || !config.label)
        {
            cls += ' no-label';
        }
        
        var $btn = $j('<button type="button"' + (config.id ? ' id="' + config.id + '"' : '') + ' title="' + config.label + '" class="' + cls + '">' + icon + (config.hideLabel ? '' : '<span>' + config.label + '</span>') + '</button>');
        if (typeof config.handler == 'function')
        {
            $btn.on('click', config.handler);
            if (config.openDialog)
            {
                $btn.attr("aria-haspopup", "dialog"); // Indicates the button opens a dialog
            }
        }
        
        if (config.pressed)
        {
            $btn.attr("aria-pressed", true);
        }
        
        if (config.disabled)
        {
            $btn.attr("aria-disabled", true);
            $btn.attr("tabindex", -1);
        }
        
        return $btn;
    },

    /**
     * Create and attach context menu
     * @param {Object} config the menu configuration
     * @param {String} config.selector (required The jQuery selector matching the elements to trigger on
     * @param {String} [config.id] The id of menu
     * @param {String} config.className The CSS class of menu
     * @param {String} [config.trigger=left] event that triggers the contextmenu.
     * @param {Object[]} config.menuItems the menu items
     * @param {Boolean} [config.triggerActive] to trigger active
     * @param {Object} [config.position] the menu position
     * @param {Object} [config.position.my] the x and y position
     * @param {Object} [config.position.at] the x and y position
     */
    attachContextMenu: function(config)
    {
        var id = config.id || 'context-menu-list-' + Math.random().toString(16).slice(-8);
        
        var menuCfg = {
            id: id,
            selector: config.selector, 
            //appendTo: $j(selector).parent().get(0), 
            
            className: config.className,
            
            // open menu on left click ------------------------                    
            trigger: config.trigger || 'left',
            
            // menu position at left top of bouton
            position: function(opt, x, y) {
              opt.$menu.position({
                my: config.position && config.position.my || 'left top',
                at: config.position && config.position.at || 'left bottom',
                of: opt.$trigger
              });
            },
            zIndex: 1000,
            
            items: config.menuItems,
            
            events: {
               show : function(options){
                  this.attr("aria-expanded", true);
                  if (config.triggerActive)
                  {
                     this.addClass('trigger-active'); // for menu with chevron pattern as icon
                  }
               },
               
               hide: function(options) {
                  this.attr("aria-expanded", false);
                  if (config.triggerActive)
                  {
                     this.removeClass('trigger-active'); // for menu with chevron pattern as icon
                  }
               }
            }
        }
        
        // attach menu to button
        $j.contextMenu(menuCfg);
        
        // Set aria attributes
        $j(config.selector).attr("aria-haspopup", true); // Indicates the button opens a menu
        $j(config.selector).attr("aria-expanded", false);
        $j(config.selector).attr("aria-controls", id);
    },
    
    /**
     * Managing focus within toolbar Using a roving tabindex
     * @param {DOMElement} toolbar the toolbar
     */
    handleKeyBoardNavigation: function(toolbar) 
    {
        // Set all of the buttons to tabindex -1 except first
        toolbar.querySelectorAll("button:not([aria-disabled])").forEach((btn, i) => (btn.tabIndex = i == 0 ? 0 : -1));
        
        // Handle navigation by arrow in toolbar
        if ($j(toolbar).attr("aria-orientation") == 'vertical')
        {
            toolbar.addEventListener("keydown", this._onUpDownKeyDown);
        }
        else
        {
            toolbar.addEventListener("keydown", this._onLeftRightKeyDown);
        }
        
        toolbar.addEventListener("click", this._onClick);
	},
    
    _onLeftRightKeyDown: function(e) 
    {
        if ($j(e.target).attr("aria-haspopup") == 'true' && $j(e.target).attr("aria-expanded") == 'true')
        {
            // target button is a context menu button and menu is open => don't capture this key
            e.preventDefault();
            return;
        }
        
        switch (e.keyCode) 
        {
            case AmetysFrontEditionToolbarHelper.KEYCODE.RIGHT:
                // on key right, go to next focusable button
                e.preventDefault();
                AmetysFrontEditionToolbarHelper._focusNextItem(this);
                break;
            case AmetysFrontEditionToolbarHelper.KEYCODE.LEFT:
                // on key left, go to previous focusable button
                e.preventDefault();
                AmetysFrontEditionToolbarHelper._focusPreviousItem(this);
            break;
        }
    },
    
    _onUpDownKeyDown: function(e) 
    {
        if ($j(e.target).attr("aria-haspopup") == 'true' && $j(e.target).attr("aria-expanded") == 'true')
        {
            // target button is a context menu button and menu is open => don't capture this key
	        e.preventDefault();
	        return;
        }
        
        switch (e.keyCode) 
        {
            case AmetysFrontEditionToolbarHelper.KEYCODE.DOWN:
                // on key down, go to next focusable button
                e.preventDefault();
                AmetysFrontEditionToolbarHelper._focusNextItem(this);
                break;
            case AmetysFrontEditionToolbarHelper.KEYCODE.UP:
                // on key up, go to previous focusable button
                e.preventDefault();
                AmetysFrontEditionToolbarHelper._focusPreviousItem(this);
            break;
        }
    },

    _onClick: function(e) 
    {
        // Make sure the clicked item is one of the buttons and
        // not something random :)
        const buttons = Array.from(this.querySelectorAll("button"));
        if (buttons.indexOf(e.target) == -1) {
            return;
        }
        AmetysFrontEditionToolbarHelper._activate(this, e.target);
    },

    /**
     * @private
     * Focus the next focusable button in toolbar
     * @param {DOMElement} toolbar the toolbar
     */
    _focusNextItem: function (toolbar) 
    {
        const item = document.activeElement;
        const nextBtn = $j(item).nextAll("button:not([aria-disabled])"); // next focusable buttons
      
        if (nextBtn.length > 0) {
            AmetysFrontEditionToolbarHelper._activate(toolbar, nextBtn.get(0));
        }
    },
    
    /**
     * @private
     * Focus the previous focusable button in toolbar
     * @param {DOMElement} toolbar the toolbar
     */
    _focusPreviousItem: function (toolbar) 
    {
        const item = document.activeElement;
        const previousBtn = $j(item).prevAll("button:not([aria-disabled])"); // previous focusable buttons
        if (previousBtn.length > 0) 
        {
            AmetysFrontEditionToolbarHelper._activate(toolbar, previousBtn.get(0));
        }
    },

    /**
     * @private
     * Activate and focus the item of the toolbar
     * @param {DOMElement} toolbar the toolbar
     */
    _activate: function (toolbar, item) {
        // Set all of the buttons to tabindex -1
        toolbar.querySelectorAll("button").forEach((btn) => (btn.tabIndex = -1));
        
        // Make the current button "active"
        item.tabIndex = 0;
        item.focus();
    }
}