/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The static class for edition from front office
 */
Ext.define("AmetysFrontEdition.widget", {
    singleton: true,
    
    /**
     * Create the widget
     * @param {HTMLElement} The parent node where to render
     * @param {String} contentId The content beeing edited
     * @param {String} metadataPath The path of the metadata beeing edited
     * @param {String} type The metadata type such as STRING, DATE...
     * @param {String} label The metadata label
     * @param {String} description The metadata description
     * @param {Boolean} multiple Is the value multiple?
     * @param {Object} value The value to set
     * @param {String} widget The backoffice widget to use
     * @param {Object} widgetParams Params for the widget
     * @param {Object[]} enumeration The possible values
     * @param {Object} validation Validation informations such as mandatory or regexp...
     * @param {Object} rawData The server side data
     */
    createWidget: function(parentNode, contentId, metadataPath, label, description, type, multiple, value, widget, widgetParams, enumeration, validation, rawData)
    {
        var widgetCfg = Ext.apply(widgetParams || {}, {
            name: metadataPath,
            shortName: name,
            type: type,
            
            // ametysDescription: description || '', // no description in FO mode
            ametysDescriptionUrl: '',
            
            mandatory: validation ? validation.mandatory : false,
            value: value,
            
            multiple: multiple === true || multiple === 'true', // some components may send a string instead of the awaited boolean
            widget: widget,
            
            disabled: false,
            disableCondition: null,
            
            contentInfo: {
                contentId: contentId,
                metadataPath: metadataPath
            },
            contentType: rawData.contentType,
            
            // widget specific config for front edition
            floating: true,
            hideLabel: true,
            renderTo: parentNode,
            ametysOffset: [0, 0.9, 0, 0],
            minWidth: 100,
            standaloneEdition: true,
            
            listeners: {
                'specialkey': function(input, e) {
                    if (e.getKey() === Ext.event.Event.RETURN)
                    {
                        AmetysFrontEdition.closeEdition(true);
                    }
                    else if (e.getKey() === Ext.event.Event.ESC)
                    {
                        AmetysFrontEdition.closeEdition(false);
                    }
                },
                scope: this
            }
        });
        
        if (enumeration !== undefined)
        {
            var widgetEnumeration = null; 
            if (enumeration)
            {
                widgetEnumeration = [];
                for (var j=0; j < enumeration.length; j++)
                {
                    widgetEnumeration.push([enumeration[j].value, enumeration[j].label]);
                }
            }
            widgetCfg.enumeration = widgetEnumeration;
        }
        
        AmetysFrontEdition.widget._applySpecificWidgetConf(widget, type, widgetCfg);
        
        var input = Ametys.form.WidgetManager.getWidget(widget, type, widgetCfg);
        // Apply common stuffs
        input.addCls(AmetysFrontEdition.CLASSNAME_EDITORS);
        return input;
    },
    
    /**
     * @private
     * Apply specific configuration depending on widget or type
     * @param {String} type the input type
     * @param {String} widget the widget
     * @param {Object} widgetCfg the widget configuration
     */
    _applySpecificWidgetConf: function(widget, type, widgetCfg)
    {
        if (type == 'rich-text')
        {
            widgetCfg = Ext.apply(widgetCfg, {
                ametysOffset: [36, 21, 0, 2],
                minHeight: 350,
                minWidth: 160
            });
            
            if (AmetysFrontEdition.enableSimpleMode)
            {
                // add save and cancel button to toolbar
                widgetCfg.additionalToolbar = 'ametysSaveButton ametysCancelButton';
                widgetCfg.additionalTinyMCEPlugins = 'ametyssave';
            }
        }
        else if (type == 'datetime')
        {
            widgetCfg = Ext.apply(widgetCfg, {
                minWidth: 280
            });
        }
        else if (type == 'boolean')
        {
            widgetCfg = Ext.apply(widgetCfg, {
                minWidth: 50
            });
        }
        else if (widget == 'edition.textarea')
        {
            widgetCfg = Ext.apply(widgetCfg, {
                minHeight: 80
            });
        }
    }
});


