/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The static class for front edition floating toolbar for page
 */
AmetysFrontEditionPageToolbar = {
    
    /**
     * @readonly
     * @property {String} MENU_CLASSNAME The css classname for toolbar's menus
     */
    MENU_CLASSNAME: 'ametys-front-edition-page-toolbar-menu',
    
    /**
     * @readonly
     * @property {String} BUTTON_CLASSNAMES The css classnames separated by coma for toolbar's buttons
     */
    BUTTON_CLASSNAMES: 'ametys-front-edition-page-toolbar-btn',
    
    /**
     * Create the page toolbar
     * @param {String} selector The jQuery selector for toolbar
     * @param {Object} config The toolbar configuration
     * @param {Object[]} config.items The toolbar items and their organization (ex: ['edition-mode', 'add-content', 'add-page', {type: 'menu', items: ['tag', 'rename', 'delete', 'schedule-publication', 'move']}])
     * @param {Object} config.actionStatus The active status of each action depending on user rights, FO edition mode, page position (ex: rights = {'add-page': true, 'add-content': true', 'tag': false, 'move: 'true'})
     * @param {String} config.pageId The page id
     * @param {String} config.parentId The page parent id
     * @param {Number} config.position The page position into its parent
     */
    createPageToolbar: function(selector, config)
    {
        var $toolbar = $j(selector);
        $toolbar.attr("role", "toolbar");
        $toolbar.attr("aria-orientation", "vertical");
        if (config.label)
        {
            $toolbar.attr("aria-label", config.label);
        }
        
         var items = config.items || [];  // ex: items = ['edition-mode', 'add-content', 'add-page', {type: 'menu', items: ['tag', 'rename', 'delete', 'schedule-publication', 'move']}]
         var actionStatus = config.actionStatus || {}; // ex: actionStatus = {'add-page': true, 'add-content': true', 'tag': false, 'move: 'true'}
         
         for (var i=0; i < items.length; i++)
         {
	           if (typeof items[i] == 'string')
	           {
	               items[i] = {type: items[i]};
	           }
               
               var type = items[i].type;
               if (type == 'menu')
               {
                    var menuItems = AmetysFrontEditionPageToolbar.getPageMenuItems(config, items[i].items);
                    if (menuItems && Object.keys(menuItems).length > 0)
                    {
                        if (Object.keys(menuItems).length == 1)
                        {
                            // transform unique menu item into button
                            var itemType = Object.keys(menuItems)[0];
                            var menuItem = menuItems[itemType];
                            var id = 'ametys-front-edition-page-toolbar-btn-' + type + '-' + Math.random().toString(16).slice(-8);
                            var $btn = AmetysFrontEditionPageToolbar._getToolbarButton({iconCls: menuItem.icon, label: menuItem.name, handler: menuItem.callback, id: id});
                            
                            $btn.appendTo($toolbar);
                            
                            if (menuItem.items)
                            {
                                // attach menu to button
                                AmetysFrontEditionPageToolbar._createMenu('#' + id, menuItem.items);
                            }
                        }
                        else
                        {
                            // create menu button
                            var id = 'ametys-front-edition-page-toolbar-btn-' + type + '-' + Math.random().toString(16).slice(-8);
                            var label = items[i].hasOwnProperty('label') ? items[i].label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_ACTIONS_MENU_LABEL}}";
	                        var $btn = AmetysFrontEditionPageToolbar._getToolbarButton({iconCls: items[i].hasOwnProperty('icon') ? items[i].icon : 'fas fa-ellipsis-v', label: label, id: id});
                            $btn.appendTo($toolbar);
                            
	                        // attach menu to button
                            AmetysFrontEditionPageToolbar._createMenu('#' + id, menuItems);
                        }
                    }
               }
               else if (type == 'edition-mode')
               {
                    // insert button to activate/deactivate edition mode
                    var $btn = AmetysFrontEditionPageToolbar._getToggleEditionModeButton(config, items[i]);
                    $btn.appendTo($toolbar);
               }
               else if (type == 'add-content' && actionStatus[type])
               {
                    // insert "add content" button with menu for each available content type/zone couples
                    var $btn = AmetysFrontEditionPageToolbar._getAddContentsToolbarButton(config, items[i]);
                    if ($btn)
                    {
                        $btn.appendTo($toolbar);
                    }     
               }
               else if (actionStatus[type] && AmetysFrontEditionPageToolbar.PAGE_ACTIONS_BUTTON[type])
	           {
                    var $btn = AmetysFrontEditionPageToolbar.PAGE_ACTIONS_BUTTON[type](config, items[i]);
                    if ($btn)
                    {
                        $btn.appendTo($toolbar);
                    }
               }
               else if (!AmetysFrontEditionPageToolbar.PAGE_ACTIONS_BUTTON[type])
               {
                    console.warn('[Front edition toolbar] No page action available for type ' + type);
	           }
        }
        
        // Collapsed toolbar on scroll and hide menu
        window.addEventListener("scroll", function(event) {
	        var top = this.scrollY;
	        if(top > 100) {
	          $toolbar.addClass("collapsed");
	        } else {
	          $toolbar.removeClass("collapsed");
	        }
            
            // Hide menu if open
            $j('.' + AmetysFrontEditionPageToolbar.MENU_CLASSNAME).hide();
	      },
	      false
	    );
        
        // Expand toolbar on click
        $toolbar.on('click', function() {
           $toolbar.removeClass("collapsed");
        });
        
        // Managing focus within toolbar using a roving tabindex
        AmetysFrontEditionToolbarHelper.handleKeyBoardNavigation($toolbar.get(0));
    },
    
    /**
     * @private
     * Add a item to toolbar
     * @param {DOMElement} toolbar the toolbar
     * @param {DOMElement} item the item to add
     */
    _addToolbarItem: function(toolbar, item)
    {
        var isFirst = false;
        var $items = $j(toolbar).find(">ul");
        if ($items.length == 0)
        {
            $items = $j("<ul></ul>");
            $items.appendTo($j(toolbar));
            isFirst = true;
        }
        var $li = $j("<li></li>");
        $li.appendTo($items);
        $j(item).attr("tabindex", isFirst ? "0" : "-1");
        $j(item).appendTo($li);
    },
    
    /**
     * @private
     * Create a menu
     * @param {String} selector The jQuery selector matching the button to trigger on
     * @param {Object} menuItems The menu items
     * @param {boolean} root true if the menu is a root menu
     */
    _createMenu: function(selector, menuItems)
    {
        AmetysFrontEditionToolbarHelper.attachContextMenu({
            selector: selector,
            menuItems: menuItems,
            className: AmetysFrontEditionPageToolbar.MENU_CLASSNAME,
            // menu position at left top of bouton
            position: {
                my: 'right top',
                at: 'left top' 
            }
        });
    },
    
    /**
     * @private
     * Get the button to activate/deactivate edition mode
     * @return {Object} the button
     */
    _getToggleEditionModeButton: function()
    {
        return AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls: "fas fa-pen", 
            label: AmetysFrontEdition.EDITION_MODE ? "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_EDITION_MODE_OFF}}" : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_EDITION_MODE_ON}}", 
            handler: AmetysFrontEdition.toggleEditionMode
        });
    },
    
    /**
     * Get the menu items for a page
     * @param {Object} config The needed configuration object
     * @param {Object[]} items The items of menu
     * @return {Object} the menu items
     */
    getPageMenuItems: function (config, items)
    {
        var menuItems = {};
        var hasItems = false;
        
        var items = items || [];
        var actionStatus = config.actionStatus || {};
        
        for (var i=0; i < items.length; i++)
        {
            if (typeof items[i] == 'string')
            {
                items[i] = {type: items[i]};
            }
            
            var type = items[i].type;
            if (type == 'menu')
            {
                var subMenuItems = AmetysFrontEditionPageToolbar.getPageMenuItems(config, items[i].items);
                var subMenuSize = Object.keys(subMenuItems).length;
                if (subMenuSize == 1)
                {
                    // transform unique sub menu item to menu item
                    var key = Object.keys(subMenuItems)[0];
                    menuItems[key] = subMenuItems[key];
                    hasItems = true;
                }
                else if (subMenuSize > 1)
                {
                    menuItems['menu-' + i] = {
                        name: items[i].hasOwnProperty('label') ? items[i].label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_ACTIONS_MENU_LABEL}}", 
                        icon: items[i].icon,
                        items: subMenuItems
                    }
                    hasItems = true;
                }
            }
            else if (type == 'add-content' && actionStatus[type])
            {
                var subMenuItems = AmetysFrontEditionPageToolbar._getAddContentsMenuItems(config, items[i]);
                var subMenuSize = Object.keys(subMenuItems).length;
                if (subMenuSize == 1)
                {
                    // transform unique sub menu item to menu item
                    var key = Object.keys(subMenuItems)[0];
                    menuItems[key] = subMenuItems[key];
                    
                    hasItems = true;
                }
                else if (subMenuSize > 1)
                {
                    menuItems['menu-' + i] = {
                        name: items[i].hasOwnProperty('label') ? items[i].label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_ADD_CONTENT_LABEL}}", 
                        icon: items[i].hasOwnProperty('icon') ? items[i].icon : 'fas fa-file-alt',
                        items: subMenuItems
                    }
                    
                    hasItems = true;
                }
            }
            else if (actionStatus[type] && AmetysFrontEditionPageToolbar.PAGE_ACTIONS_MENU_ITEM[type])
            {
                var menuItem = AmetysFrontEditionPageToolbar.PAGE_ACTIONS_MENU_ITEM[type](config, items[i]);
                if (menuItem)
                {
                    menuItems[type] = menuItem;
                    hasItems = true;
                }
            }
            else if (!AmetysFrontEditionPageToolbar.PAGE_ACTIONS_MENU_ITEM[type])
            {
                console.warn('[Front edition toolbar] No item available for type ' + type);
            }
        }
        
        return hasItems ? menuItems : null; // return null if no action is available
    },
    
    /**
     * @private
     * Get menu items to add contents
     * @param {Object} config The toolbar configuration
     * @param {Object} opts the button options
     * @return {Object} the menu items
     */
    _getAddContentsMenuItems: function(config, opts)
    {
        var menuItems = {};
        if (config.addContentsConfig)
        {
            for (var i=0; i < config.addContentsConfig.length; i++)
            {
                var addCfg = config.addContentsConfig[i];
                if ($j.inArray(addCfg.contentType, config.allowedContentTypes) != -1)
                {
                    menuItems['add-content-' + i] = {
	                    name: addCfg.label,
	                    icon: addCfg.icon,
	                    callback: AmetysFrontEdition.page.addContent.bind(AmetysFrontEdition, {
	                            pageId: config.pageId,
	                            zoneName: addCfg.zoneName,
	                            contentTypes: addCfg.contentType,
	                            lang: AmetysFrontEdition.LANG,
	                            initWorkflowActionId: addCfg.initActionId || 1,
	                            workflowName: addCfg.workflowName || 'content' 
	                    })
	                }
                }
            }
        }
        return menuItems;
    },
    
    /**
     * Get button menu to add contents
     * @param {Object} config The toolbar configuration
     * @param {Object} opts the button options
     */
    _getAddContentsToolbarButton: function(config, opts)
    {
        if (config.addContentsConfig)
        {
            var menuItems = AmetysFrontEditionPageToolbar._getAddContentsMenuItems(config, opts);
            
            if (Object.keys(menuItems).length > 0)
            {
                if (Object.keys(menuItems).length == 1)
                {
                    var itemType = Object.keys(menuItems)[0];
                    var menuItem = menuItems[itemType];
                    return AmetysFrontEditionPageToolbar._getToolbarButton({iconCls: menuItem.icon, label: menuItem.name, handler: menuItem.callback, openDialog: true});
                }
                else
                {
	                var id = 'ametys-front-edition-page-toolbar-btn-add-content' + Math.random().toString(16).slice(-8);
			        var label = opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_ADD_CONTENT_LABEL}}";
			        var $btn = AmetysFrontEditionPageToolbar._getToolbarButton({iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-file-alt', label: label, id: id});
                    
                    AmetysFrontEditionPageToolbar._createMenu('#' + id, menuItems);
			        
	                return $btn;
                }
            }
        }
        
        return null;
    },
    
    /**
     * @private
     * Get a button for a action
     * @param {Object} config the button configuration
     * @param {String} config.iconCls the css class for icon
     * @param {label} config.label the button's label
     * @param {Function} [config.handler] the function to call on click
     * @param {String} [config.id] button unique id
     * @param {Boolean} [config.openDialog] true if the handler of button opens a dialog
     * @return The button
     */
    _getToolbarButton: function(config)
    {
        return AmetysFrontEditionToolbarHelper.createToolbarButton($j.extend(config, {
            className: AmetysFrontEditionPageToolbar.BUTTON_CLASSNAMES,
        }));
    },
    
    /**
     * @private
     * Get the button to tag the page
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getAddPageButton: function (config, opts)
    {
        return AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-file',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_CREATE_PAGE_BUTTON}}",
            openDialog: true,
            handler: function() {
                AmetysFrontEdition.page.createPage(config.pageId, config.addPageConfig || {})
            }
        });
    },
    
    /**
     * @private
     * Menu item to tag a page
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getAddPageMenuItem: function (config, opts)
    {   
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_CREATE_PAGE_BUTTON}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-file",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.page.createPage(config.pageId, config.addPageConfig || {})
            }
        };
    },
    
    /**
     * @private
     * Get the button to tag the page
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getTagPageButton: function (config, opts)
    {
        return AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls:opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-tag',
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TAG_PAGE_BUTTON}}",
            openDialog: true,
            handler: function() {
                AmetysFrontEdition.page.changePageTags(config.pageId)
            }
        });
    },
    
    /**
     * @private
     * Menu item to tag a page
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getTagPageMenuItem: function (config, opts)
    {   
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TAG_PAGE_BUTTON}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-tag",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.page.changePageTags(config.pageId)
            }
        };
    },
    
    /**
     * @private
     * Get the button to delete the page
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getDeletePageButton: function (config, opts)
    {
        return AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-times', 
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_DELETE_PAGE_BUTTON}}", 
            handler: function() {
                AmetysFrontEdition.page.deletePage(config.pageId, config.title);
            }
        });
    },
    
    /**
     * @private
     * Menu item to delete the page
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getDeletePageMenuItem: function(config, opts)
    {
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_ACTION_DELETE}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-times",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.page.deletePage(config.pageId, config.title);
            }
        };
    },
    
    /**
     * @private
     * Get the button to rename the page
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getRenamePageButton: function (config, opts)
    {
        return AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-pen', 
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_RENAME_PAGE_BUTTON}}", 
            openDialog: true,
            handler: function() {
             AmetysFrontEdition.page.renamePage(config.pageId);
            }
        });
    },
    
    /**
     * @private
     * Menu item to rename the page
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getRenamePageMenuItem: function(config, opts)
    {
        return {
            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_ACTION_RENAME}}", 
            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-pen",
            callback: function(itemKey, opt, e) {
                AmetysFrontEdition.page.renamePage(config.pageId);
            }
        };
    },
    
    /**
     * @private
     * Get the button to schedule the publication of the page
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getSchedulePublicationPageButton: function (config, opts)
    {
        if (AmetysFrontEdition.EDITION_MODE)
        {
            return AmetysFrontEditionPageToolbar._getToolbarButton({
                iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-calendar', 
                label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_SCHEDULE_LABEL}}", 
                handler: function() {
                    AmetysFrontEdition.page.schedulePublication(config.pageId);
                }
            });
        }
        return null;
    },
    
    /**
     * @private
     * Menu item to schedule the publication of the page
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getSchedulePublicationPageMenuItem: function(config, opts)
    {
        if (AmetysFrontEdition.EDITION_MODE)
        {
	        return {
	            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_SCHEDULE_LABEL}}", 
	            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-calendar",
	            callback: function(itemKey, opt, e) {
	                AmetysFrontEdition.page.schedulePublication(config.pageId);
	            }
	        };
        }
        return null;
    },
    
    /**
     * @private
     * Get the button to move the page, with submenu to choose direction (up or down)
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMovePageButton: function (config, opts)
    {
        var id = 'ametys-front-edition-page-toolbar-btn-move-' + Math.random().toString(16).slice(-8);
        var $btn = AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-arrows-alt', 
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_MOVE_MENU_LABEL}}",
            id: id
        });
        
        AmetysFrontEditionPageToolbar._createMenu('#' + id, AmetysFrontEditionPageToolbar._getMovePageMenuItem(config, opts).items);
                            
        return $btn;
    },
    
    /**
     * @private
     * Get the menu item to move the page, with submenu to choose direction (up or down)
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMovePageMenuItem: function(config, opts)
	{
	    return {
	            name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_MOVE_MENU_LABEL}}", 
	            icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-arrows-alt",
	            "items": {
	                "up": {
	                    name: "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_MOVE_UP_LABEL}}", 
	                    icon: "fas fa-arrow-up",
	                    callback: function(itemKey, opt, e) {
	                        AmetysFrontEdition.page.movePage(config.pageId, config.parentId, config.position - 2);
	                    },
                        disabled: !config.actionStatus['move-up'],
                        className: !config.actionStatus['move-up'] ? 'disabled' : ''
	                },
	                "down": {
	                    name: "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_MOVE_DOWN_LABEL}}", 
	                    icon: "fas fa-arrow-down",
	                    callback: function(itemKey, opt, e) {
	                       AmetysFrontEdition.page.movePage(config.pageId, config.parentId, config.position + 1);
	                    },
                        disabled: !config.actionStatus['move-down'],
                        className: !config.actionStatus['move-down'] ? 'disabled' : ''
	                }
	            }
	    }
	},
    
    /**
     * @private
     * Get the button to move up the page
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMoveUpPageButton: function (config, opts)
    {
        return AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-arrow-up', 
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_MOVE_UP_LABEL}}", 
            handler: function() {
                AmetysFrontEdition.page.movePage(config.pageId, config.parentId, config.position - 1);
            }
        });
    },
    
    /**
     * @private
     * Menu item to move up a page
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getMoveUpPageMenuItem: function(config, opts)
    {
        return {
                name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_MOVE_PAGE_UP_BUTTON}}", 
                icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-arrow-up",
                callback: function(itemKey, opt, e) {
                    AmetysFrontEdition.page.movePage(config.pageId, config.parentId, config.position - 1);
                }
        }
    },
    
    /**
     * @private
     * Get the button to move down the page
     * @param {Object} config the toolbar configuration
     * @param {Object} opts the button's options
     * @return {Object} the button
     */
    _getMoveDownPageButton: function (config, opts)
    {
        return AmetysFrontEditionPageToolbar._getToolbarButton({
            iconCls: opts.hasOwnProperty('icon') ? opts.icon : 'fas fa-arrow-down', 
            label: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_TOOLBAR_PAGE_MOVE_DOWN_LABEL}}", 
            handler: function() {
                AmetysFrontEdition.page.movePage(config.pageId, config.parentId, config.position + 1);
            }
        });
    },
    
    /**
     * @private
     * Menu item to move down a page
     * @param {Object} config 
     * @param {Object} opts item options
     * @return {Object} the menu item
     */
    _getMoveDownPageMenuItem: function(config, opts)
    {
        return {
                name: opts.hasOwnProperty('label') ? opts.label : "{{i18n PLUGINS_FRONT_EDITION_MOVE_PAGE_DOWN_BUTTON}}", 
                icon: opts.hasOwnProperty('icon') ? opts.icon : "fas fa-arrow-down",
                callback: function(itemKey, opt, e) {
                    AmetysFrontEdition.page.movePage(config.pageId, config.parentId, config.position + 1);
                }
        }
    }
}

AmetysFrontEditionPageToolbar.PAGE_ACTIONS_MENU_ITEM = {
        'add-page': AmetysFrontEditionPageToolbar._getAddPageMenuItem,
        'rename': AmetysFrontEditionPageToolbar._getRenamePageMenuItem,
        'tag': AmetysFrontEditionPageToolbar._getTagPageMenuItem,
        'move': AmetysFrontEditionPageToolbar._getMovePageMenuItem,
        'move-up': AmetysFrontEditionPageToolbar._getMoveUpPageMenuItem,
        'move-down': AmetysFrontEditionPageToolbar._getMoveDownPageMenuItem,
        'delete': AmetysFrontEditionPageToolbar._getDeletePageMenuItem,
        'schedule-publication': AmetysFrontEditionPageToolbar._getSchedulePublicationPageMenuItem
}

AmetysFrontEditionPageToolbar.PAGE_ACTIONS_BUTTON = {
        'add-page': AmetysFrontEditionPageToolbar._getAddPageButton,
        'rename': AmetysFrontEditionPageToolbar._getRenamePageButton,
        'tag': AmetysFrontEditionPageToolbar._getTagPageButton,
        'move': AmetysFrontEditionPageToolbar._getMovePageButton,
        'move-up': AmetysFrontEditionPageToolbar._getMoveUpPageButtonm,
        'move-down': AmetysFrontEditionPageToolbar._getMoveDownPageButton,
        'delete': AmetysFrontEditionPageToolbar._getDeletePageButton,
        'schedule-publication': AmetysFrontEditionPageToolbar._getSchedulePublicationPageButton
}