/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The static class for edition from front office
 */
AmetysFrontEdition = {
	/**
	 * @readonly
     * @property {String} CLASSNAME_EDITABLE The css classname to set on elements that are editable
	 */
	CLASSNAME_EDITABLE : "ametys-front-edition-possible",
    /**
     * @readonly
     * @property {String} CLASSNAME_HOVER The css classname to set on elements when the mouse is over
     */
    CLASSNAME_HOVER : "ametys-front-edition-hover",
	/**
	 * @readonly
     * @property {String} CLASSNAME_EDITBUTTON The css classname to set on the button that makes the parent element editable
	 */
	CLASSNAME_EDITBUTTON : "ametys-front-edition-button",
	/**
	 * @readonly
     * @property {String} CLASSNAME_EDITION The css classname to set on elements when there is a running edition
	 */
	CLASSNAME_EDITION : "ametys-front-edition-running",
	/**
	 * @readonly
	 * @property {String} CLASSNAME_EDITORS The css classname to set on editors
	 */
	CLASSNAME_EDITORS : "ametys-front-edition",
    /**
     * 
     * @property {String} EDITBUTTON_INNERHTML The html to insert in the edition buttons 
     */
    EDITBUTTON_INNERHTML: "<span class='ametys-front-edition-button-text'>{{i18n PLUGINS_FRONT_EDITION_EDIT_BUTTON}}</span>",
	/**
	 * @readonly
     * @property {String} CONTEXT_PATH The url to use to contact server for plugins resources
	 */
	CONTEXT_PATH : null,
	/**
	 * @readonly
	 * @property {String} SITECONTEXT_PATH The url to use to contact server
	 */
	SITECONTEXT_PATH : null,
	/**
     * @property {String} [LOCALE="en"] The code of the country for locale. Set this constant.
	 */
	LOCALE : "en",
	/**
     * @property {String} JS_HASHCODE The hashcode to load all the js file using the minimizer 
	 */
	JS_HASHCODE : null,
	/**
     * @property {String} CSS_HASHCODE The hashcode to load all the css file using the minimizer 
	 */
	CSS_HASHCODE : null,
    /**
     * @readOnly
     * @property {String} FRONT_EDITION_JS_LOADING the class used to set on the html to inform that the javascript for front-edition is loading
     */
    FRONT_EDITION_JS_LOADING : "front-edition-js-loading",
    /**
     * @readOnly
     * @property {String} FRONT_EDITION_JS_LOADING_HTML some html that will be added at the end bof the body while the javascript for front-edition is loading
     */
    FRONT_EDITION_JS_LOADING_HTML : null,
    /**
     * @readOnly
     * @property {String} FRONT_EDITION_JS_LOADING_HTML_ID the id of the div that will contain FRONT_EDITION_JS_LOADING_HTML during loading (only if FRONT_EDITION_JS_LOADING_HTML is not empty)
     */
    FRONT_EDITION_JS_LOADING_HTML_ID : "front-edition-js-loading-div",
    /**
     * 
     * @property {String} FRONT_EDITION_JS_LOADED the class used to set on the html to inform that the javascript for front-edition is loaded
     */
    FRONT_EDITION_JS_LOADED : "front-edition-js-loaded",

    /**
     * @cfg {Number} [editActionId=2] The edit workflow action id, for saving the content after edition
     */

    /**
     * @cfg {Boolean} [enableSimpleMode=true] True to enable the edition field per field, false to disable it
     */
    enableSimpleMode : true,
	/**
	 * @private
	 * @property {Ext.Component} currentField The current editor field
	 */
	currentField : null,
    /**
     * @private
     * @property {Object} fields map of edition fields ( Ext.Component )
     */
    fields : {},
    /**
     * @private
     * @property {Ext.toolbar.Toolbar} toolbar to edit elements
     */
    toolbar: null,
	/**
	 * @private
     * @property {Function} callback When the additionnal files are loaded asynchronously, the callback is stored here
	 */
	callback : null,
    /**
     * @private
     * @property {Function} startCallback Optional function to invoke after the start. Be careful, this function can be called several time, each time new contents are detected as modifiables in the page
     */
    startCallback : null,
	/**
	 * @private
     * @property {Number} initialized Is the front edition initialized.
     *  * 0 = not listening to mouse events
     *  * 1 = listening to mouse events
     *  * 2 = additionnal files loading
     *  * 3 = additionnal filed loaded
	 */
	initialized : 0,

	toggleEditionMode : function()
    {
        AmetysFrontEdition.load(function()
        {
            Ametys.data.ServerComm.callMethod({
                role: "org.ametys.plugins.frontedition.FrontEditionHelper",
                methodName: "firstAvailableParentInLivePath",
                parameters: [AmetysFrontEdition.PAGEID],
                waitMessage: "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_LOADING}}",
                callback: {
                    scope: this,
                    handler: AmetysFrontEdition._toggleEditionModeCb
                },
                errorMessage: true
            });
        });
	},
    
    _toggleEditionModeCb : function(availablePagePath)
    {
        var path = AmetysFrontEdition.CONTEXT_PATH;
        var editionPath = "edition";

        if (AmetysFrontEdition.EDITION_MODE)
        {
            if (path.endsWith(editionPath))
            {
                path = path.substr(0, path.length - editionPath.length - 1);
            }
        }
        else
        {
            path += "/" + editionPath;
        }

        path += "/" + AmetysFrontEdition.LANG + "/" + availablePagePath + ".html";
        window.location.href = path;
    },

	/**
	 * Check if the given element or one of its parents is an ametys editable one : and return the element that is editable
	 * @param {HTMLElement} element The html element to check
	 * @return {HTMLElement} The editable element or null otherwise
	 */
    getEditableElement: function(element) {
		var jElement = $j(element);

		if (jElement.is("*[data-ametys-metadata]"))
        {
			return element;
		}

		var jParent = jElement.parents("*[data-ametys-metadata]");
		if (jParent.length > 0)
        {
			return jParent[0];
		}

		return null;
	},
    /**
     * permit to edit multiple elements at once
     * @param {String} contentId Content ID to filter
     * @param {number[]} workflowIds workflow ids to execute
     */
    editElements: function(contentId, workflowIds)
    {
        AmetysFrontEdition.cancelEditElements();
        AmetysFrontEdition.load(function()
        {
            var $elements = $j("[data-ametys-metadata^='" + contentId + ";']");
            var elementsList = $elements.toArray();
            AmetysFrontEdition._removeListeners();
            AmetysFrontEdition._showEditToolbar(contentId, workflowIds);
            AmetysFrontEdition._editElements(elementsList, workflowIds);
        });
    },
    /**
     * close all edition elements and re-set removed listeners
     */
    cancelEditElements: function()
    {
        AmetysFrontEdition.closeEdition(false);
        AmetysFrontEdition._removeEditToolbar();
        AmetysFrontEdition._initListeners();
    },
    _unlockAll: function()
    {
        var allFields = AmetysFrontEdition.getFields();
        var contentIds = [];
        for (var i = 0; i < allFields.length; i++)
        {
            if (allFields[i] && allFields[i].contentId)
            {
                if (contentIds.indexOf(allFields[i].contentId) == -1)
                {
                    contentIds.push(allFields[i].contentId);
                }
            }
        }
        if (contentIds.length > 0)
        {
            AmetysFrontEdition.comm.unlockOrLock(contentIds, "unlock");
        }
    },
    /**
     * get the workflow Ids only (strip the other metadatas)
     * @private
     * @param {Object[]} workflowIds array of workflow ids
     * @return {Object} workflows
     * @return {Number[]} workflows.numericWorkflowIds array of workflow ids
     * @return {Object} workflows.overrideTexts Map of override texts
     */
    _parseWorkflowIds: function(workflowIds)
    {
        var numericWorkflowIds = [];
        var overrideTexts = {};
        for (var index = 0; index < workflowIds.length; index++)
        {
            if (Ext.isObject(workflowIds[index]))
            {
                var workflowId = workflowIds[index];
                if (workflowId.hasOwnProperty("id"))
                {
                    numericWorkflowIds.push(workflowId["id"]);
                    if (workflowId.hasOwnProperty("text"))
                    {
                        overrideTexts[workflowId["id"]] = workflowId["text"];
                    }
                }
            }
            else
            {
                numericWorkflowIds.push(workflowIds[index]);
            }
        }
        return {
            numericWorkflowIds : numericWorkflowIds,
            overrideTexts : overrideTexts
        };
    },
    /**
     * Show the edit content toolbar (to save/publish/cancel edition for a whole content)
     * @private
     * @param {} contentId content id
     * @param {} workflowIds workflow ids to execute
     */
    _showEditToolbar: function(contentId, workflowIds)
    {
        AmetysFrontEdition.load(function()
        {
            if (AmetysFrontEdition.toolbar)
            {
                AmetysFrontEdition._removeEditToolbar();
            }
            var alignTarget = $j("[data-ametys-metadata^='" + contentId + ";']").get(0);//align on the 1st matching element
            var items = [];

            var parsedWorkflowIds = AmetysFrontEdition._parseWorkflowIds(workflowIds);
            var numericWorkflowIds = parsedWorkflowIds.numericWorkflowIds;
            var overrideTexts = parsedWorkflowIds.overrideTexts;
            AmetysFrontEdition.comm.getWorkflowNames(contentId, numericWorkflowIds, function(workflows)
            {
                for (var index = 0; index < numericWorkflowIds.length; index++)
                {
                    var workflowId = numericWorkflowIds[index];
                    if (workflows.hasOwnProperty(workflowId) && workflows[workflowId] != null)
                    {
                        var text = workflows[workflowId];
                        if (overrideTexts.hasOwnProperty(workflowId))
                        {
                            text = overrideTexts[workflowId];
                        }
                        items.push({
                            text: text,
                            handler: AmetysFrontEdition._getSaveFunction(contentId, workflowId)
                        });
                    }
                }
                items.push({
                        text: 'Annuler',
                        handler: function() {AmetysFrontEdition.cancelEditElements(contentId)}
                    });
                AmetysFrontEdition.toolbar = Ext.create('Ext.toolbar.Toolbar', {
                    renderTo: document.body,
                    width   : 600,
                    floating: true,
                    alignTarget: alignTarget,
                    defaultAlign: 'b-t',
                    items: items
                });
                AmetysFrontEdition.toolbar.show();
            });
        });
    },
    _getSaveFunction: function(contentId, workflowId)
    {
        return function() {AmetysFrontEdition.save(contentId, workflowId)};
    },
    _getExecFunction: function(contentId, workflowId)
    {
        return function() {AmetysFrontEdition.execWorkflowAction(contentId, workflowId)};
    },
    _removeEditToolbar: function()
    {
        if (AmetysFrontEdition.toolbar)
        {
            AmetysFrontEdition.toolbar.hide();
        }
    },

	/**
     * Check if the element or one of its parents is an ametys editable one and toggle its css class to hover.
     * Any existing element with this class will be cleared
     * @param {HTMLElement} element The html element to check 
	 */
	toggleElementToHover : function(element)
    {
        var editableElement = AmetysFrontEdition.getEditableElement(element);
        if (editableElement)
        {
            var jElement = $j(editableElement);
            if (jElement.hasClass(AmetysFrontEdition.CLASSNAME_HOVER))
            {
                return;
            }
            else
            {
                $j("." + AmetysFrontEdition.CLASSNAME_HOVER).removeClass(AmetysFrontEdition.CLASSNAME_HOVER);
                jElement.addClass(AmetysFrontEdition.CLASSNAME_HOVER);
            }
        }
        else
        {
            $j("." + AmetysFrontEdition.CLASSNAME_HOVER).removeClass(AmetysFrontEdition.CLASSNAME_HOVER);
        }
	},

    /**
     * Find the data-ametys-metadata of a field
     * @param {HTMLElement} element
     * @return {String} data-ametys-metadata attribute of the element, or from the closest parent
     */
    findFieldId: function(element)
    {
        var $parents = $j(element).parents("[data-ametys-metadata]")
        if ($parents && $parents.length > 0)
        {
            return $parents.get(0).getAttribute("data-ametys-metadata");
        }
    },
    /**
     * Get all editable fields for a contentId (or all)
     * @param {String} [contentId] contentId for which all currently opened fields will be opened
     * @return {Object[]} list of fields
     */
    getFields: function(contentId)
    {
        var result = [];
        for (var key in AmetysFrontEdition.fields)
        {
            if (AmetysFrontEdition.fields.hasOwnProperty(key))
            {
                //filter by contentId if needed
                if (contentId)
                {
                    if (AmetysFrontEdition.fields[key].contentId && AmetysFrontEdition.fields[key].contentId == contentId)
                    {
                        result.push(AmetysFrontEdition.fields[key]);
                    }
                }
                else
                {
                    result.push(AmetysFrontEdition.fields[key]);
                }
            }
        }
        return result;
    },
    /**
     * Add an editable field in the current list
     * @param {String} id unique ID for this field (contentid;metadatapath)
     * @param {String} contentId
     * @param {String} metadataPath
     * @param {HTMLElement} field
     */
    setField: function(id, contentId, metadataPath, field)
    {
        if (!AmetysFrontEdition.fields)
        {
            AmetysFrontEdition.fields = {};
        }
        AmetysFrontEdition.fields[id] = field;
        AmetysFrontEdition.fields[id].changed = false;
        AmetysFrontEdition.fields[id].contentId = contentId;
        AmetysFrontEdition.fields[id].metadataPath = metadataPath;
    },
    /**
     * Get a list of changed fields
     * @param {String} [contentId] contentId to filter
     * @return {HTMLElement[]} Array of changed fields
     */
    getFieldsChanges: function(contentId)
    {
        var changedFields = [];
        var fields = AmetysFrontEdition.getFields(contentId);
        for (var i = 0; i < fields.length; i++)
        {
            if (fields[i].changed)
            {
                changedFields.push(fields[i]);
            }
        }
        return changedFields;
    },
	/**
	 * Starts the edition mode for one element and cancel all others
     * @param {HTMLElement} element The element to edit
	 */
	editElement : function(element)
    {
		//if (AmetysFrontEdition.currentField)
        if (!$j.isEmptyObject(AmetysFrontEdition.fields))
        {
			// Click on any extjs stuff has to be ignored
			var jElement = $j(element);
            if (jElement.is(".x-body,.x-mask,.x-field,.tox") || jElement.parents(".x-body,.x-mask,.x-field,.tox").length > 0)
            {
                // Dnd to enlarge selectContent
				return;
			}

			var changedFields = AmetysFrontEdition.getFieldsChanges();
            if (changedFields.length != 0)
            {
            	// In this case, only one field is open in edition
    			var currentField = AmetysFrontEdition.fields[Ext.Object.getAllKeys(AmetysFrontEdition.fields)[0]];
    			if (currentField.ametysNeedCloseEditionBox)
        		{
                	// Some changes, let the user decide
        		    AmetysFrontEdition.showConfirmBox();
        		    return;
        		}
            }
		}
		
		// Cancel any existing edition
		AmetysFrontEdition.closeEdition(true, AmetysFrontEdition.editActionId);

		// Only clicks on buttons has to be taken in account
		if (!$j(element).is("." + AmetysFrontEdition.CLASSNAME_EDITBUTTON + ", ." + AmetysFrontEdition.CLASSNAME_EDITBUTTON + " *"))
        {
			return;
		}

		var elementToEdit = AmetysFrontEdition.getEditableElement(element);
        AmetysFrontEdition.load(function()
        {
            AmetysFrontEdition._editElements(elementToEdit, AmetysFrontEdition.editActionId);
        });
	},
	
	/**
     * @private
     * Show the confirm box before canceling edition
     */
    showConfirmBox: function()
    {
        var box = Ext.create('Ametys.window.DialogBox', {
                title :"{{i18n PLUGINS_FRONT_EDITION_TINYMCE_CONFIRM_BOX_CANCEL_EDITION_BUTTON_LABEL}}",
                iconCls : 'ametysicon-sign-question',
                width : 450,
                
                items : [{
                            xtype: 'component',
                            html: "{{i18n PLUGINS_FRONT_EDITION_TINYMCE_CONFIRM_BOX_CANCEL_EDITION_BUTTON_DESC}}",
                            cls: 'a-text'
                        }],
                        
                defaultButton: 'validate',
                closeAction: 'destroy',
                
                buttons : [ {
                    itemId: 'quit_and_save',
                    reference: 'quit_and_save',
                    text :"{{i18n PLUGINS_FRONT_EDITION_TINYMCE_CONFIRM_BOX_CANCEL_EDITION_BUTTON_SAVE}}",
                    handler : function () { box.close(); AmetysFrontEdition.closeEdition(true, AmetysFrontEdition.editActionId);},
                    scope: this
                },
                {
                    itemId: 'quit',
                    reference: 'quit',
                    text :"{{i18n PLUGINS_FRONT_EDITION_TINYMCE_CONFIRM_BOX_CANCEL_EDITION_BUTTON_QUIT}}",
                    handler : function () { box.close(); AmetysFrontEdition.closeEdition(false, AmetysFrontEdition.editActionId);},
                    scope: this
                },
                {
                    itemId: 'cancel',
                    reference: 'cancel',
                    text :"{{i18n PLUGINS_FRONT_EDITION_TINYMCE_CONFIRM_BOX_CANCEL_EDITION_BUTTON_CANCEL}}",
                    handler : function () { box.close();},
                    scope: this
                }]
        });
        
        box.show();
    },
	
    /**
     * @private
     * Parse the metadata infos for an element
     * @param {HTMLElement} element
     * @return {Object} result Object having the informations
     * {String} result.contentId content Id
     * {String} result.metadataPath metadata path
     * {String} result.metadataInfos unique id (contentid;metadatapath)
     */
    _getElementMetadataInfos: function(element)
    {
        var metadataInfos = element.getAttribute("data-ametys-metadata");
        var i = metadataInfos.indexOf(';');
        var contentId = metadataInfos.substring(0, i);
        var metadataPath = metadataInfos.substring(i + 1);
        return {contentId : contentId, metadataPath : metadataPath, metadataInfos : metadataInfos};
    },
    /**
     * @private
     * Edit a list of elements
     * @param {HTMLElement[]} elementsToEdit list of elements that will be edited (can be only one element without array)
     */
    _editElements: function(elementsToEdit, workflowIds)
    {
        AmetysFrontEdition.load(function()
        {
            if (elementsToEdit)
            {
                if (!Ext.isArray(elementsToEdit))
                {
                    elementsToEdit = [elementsToEdit];
                }
                if (!Ext.isArray(workflowIds))
                {
                    workflowIds = [workflowIds];
                }
                var parsedWorkflowIds = AmetysFrontEdition._parseWorkflowIds(workflowIds);
                var numericWorkflowIds = parsedWorkflowIds.numericWorkflowIds;

                var contentId = null;
                var metadataPaths = [];
                for (var i = 0; i < elementsToEdit.length; i++)
                {
                    var elementToEdit = elementsToEdit[i];
                    var metadataInfos = AmetysFrontEdition._getElementMetadataInfos(elementToEdit);

                    var currentContentId = metadataInfos.contentId;
                    var metadataPath = metadataInfos.metadataPath;

                    if (contentId == null)
                    {
                        contentId = currentContentId;
                    }
                    else if (contentId != currentContentId)
                    {
                        throw new Error("{{i18n PLUGINS_FRONT_EDITION_ERROR_EDITION_OUTSIDE_CONTENT}}");
                    }
                    metadataPaths.push(metadataPath);
                }
                var callback = Ext.bind(AmetysFrontEdition._editElementsCb, AmetysFrontEdition, [elementsToEdit], 0);
                AmetysFrontEdition.comm.getServerInfos(contentId, metadataPaths, numericWorkflowIds, callback);
            }
        });
    },
    /**
     * @private
     * Handle the ajax result
     * @param {HTMLElement[]} elementsToEdit array of elements to edit
     * @param {Object} datas ajax result
     */
    _editElementsCb: function(elementsToEdit, datas)
    {
        if (datas && datas.error == undefined && datas.data && elementsToEdit)
        {
            for (var i = 0; i < elementsToEdit.length; i++)
            {
                var elementToEdit = elementsToEdit[i];
                var metadataInfos = AmetysFrontEdition._getElementMetadataInfos(elementToEdit);
                var metadataInfo = metadataInfos.metadataInfos;
                var contentId = metadataInfos.contentId;
                var metadataPath = metadataInfos.metadataPath;
                var jElementToEdit = $j(elementToEdit);

                if (datas.data[metadataPath])
                {
                    var data = datas.data[metadataPath];
                    AmetysFrontEdition._createInput(data, elementToEdit, metadataInfos);
                }
            }
        }
        else if (datas && datas.error)
        {
            if (datas.error == 'locked' && datas.locker)
            {
                var locker = datas.locker.fullName;
                var message = Ext.String.format("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR_LOCKED}}", locker);
                Ametys.Msg.alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}", message);
                AmetysFrontEdition.cancelEditElements();
                return;
            }
            else if (datas.error == 'draft')
            {
                Ametys.Msg.alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}", "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR_DRAFT}}");
                AmetysFrontEdition.cancelEditElements();
                return;
            }
            else if (datas.error == 'workflow-rights')
            {
                Ametys.Msg.alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}", "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR_WORKFLOW}}");
                AmetysFrontEdition.cancelEditElements();
                return;
            }
            else
            {
                alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}");
                AmetysFrontEdition.cancelEditElements();
                return;
            }
        }
        else
        {
            alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}");
            AmetysFrontEdition.cancelEditElements();
            return;
        }
    },

    /**
     * @private
     * Create the input field, using the ajax result
     * @param {Object} data see AmetysFrontEdition._editElementsCb
     * @param {HTMLElement[]} elementsToEdit array of elements to edit
     * @param {Object} metadataInfos see AmetysFrontEdition._getElementMetadataInfos (can be null, if so, the infos will be searched in the page)
     */
    _createInput: function(data, elementToEdit, metadataInfos)
    {
        if (!metadataInfos)
        {
            metadataInfos = AmetysFrontEdition._getElementMetadataInfos(elementToEdit);
        }
        var metadataInfo = metadataInfos.metadataInfos;
        var contentId = metadataInfos.contentId;
        var metadataPath = metadataInfos.metadataPath;
        var jElementToEdit = $j(elementToEdit);
        jElementToEdit.addClass(AmetysFrontEdition.CLASSNAME_EDITION);
        
        var widgetParams = data['widget-params'];
        var forcedWidgetParams = JSON.parse(elementToEdit.getAttribute("data-ametys-widget-params"));
        
        var widget = data.widget;
        var forcedWidget = elementToEdit.getAttribute("data-ametys-widget");
        if (forcedWidget == null)
        {
            if (widget == 'edition.file' && widgetParams && widgetParams.filter == Ametys.form.widget.File.IMAGE_FILTER)
            {
                widget = 'edition.image';
            }
        }
        else
        {
            widget = forcedWidget;
        }
        widgetParams = forcedWidgetParams || widgetParams;
        

        var input = AmetysFrontEdition.widget.createWidget(elementToEdit.parentNode, contentId, metadataPath, data.label, data.description, data.type, data.multiple, data.value, widget, widgetParams, data.enumeration, data.validation, data);

        if (input != null)
        {
            input.ametys = {
                target : elementToEdit
            };

            AmetysFrontEdition.setField(metadataInfo, contentId, metadataPath, input);
            AmetysFrontEdition.copyCSSProperties(input);

            var position = jElementToEdit.offset();
            var width = jElementToEdit.outerWidth() + input.ametysOffset[1] + input.ametysOffset[3];
            var height = jElementToEdit.outerHeight() + input.ametysOffset[0] + input.ametysOffset[2];
            input.setWidth(width);
            input.setMinHeight(height);

            if (input.ametysAlign == 'center' || !input.ametysAlign && input.getHeight() > height)
            {
                input.setPagePosition(position.left - input.ametysOffset[3], position.top - (input.getHeight() - height)/2  - input.ametysOffset[0]);
            }
            else // if (input.ametysAlign == 'top')
            {
                input.setPagePosition(position.left - input.ametysOffset[3], position.top - input.ametysOffset[0]);
            }
            input.on('change', function(cmp, newValue, oldValue)
                {
                    input.changed = this.didValueChange(this.getInitialConfig().value, newValue);
                });
            input.show();
        }
        else
        {
            alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}");
            return;
        }
    },
    
	/**
     * @private
     * Cancel any running edition
     * @param {Boolean} save Should the edition be saved before canceling
	 */
	closeEdition : function(save, editActionId)
    {
        var changedFields = AmetysFrontEdition.getFieldsChanges();
        var allFields = AmetysFrontEdition.getFields();
        if (save && changedFields.length > 0)
        {
			// Don't close when saving to keep the previous value hidden
			AmetysFrontEdition.save(null, editActionId);
		}
        else if (allFields.length > 0)
        {
            AmetysFrontEdition._unlockAll();
            for (var i = 0; i < allFields.length; i++)
            {
                if (allFields[i])
                {
                    $j(allFields[i].ametys.target).removeClass(AmetysFrontEdition.CLASSNAME_EDITION);
                    allFields[i].destroy();
                }
            }
            AmetysFrontEdition.fields = {}
		}
	},
	
	/**
     * @private
     * Save and apply the content of current element
	 */
	save : function(contentId, editionWorkflowId)
    {
        AmetysFrontEdition.load(function()
        {
            var changedFields = AmetysFrontEdition.getFieldsChanges(contentId);
            var datas = {};
            for (var i = 0; i < changedFields.length; i++)
            {
                if (!changedFields[i].isValid())
                {
                    changedFields[i].focus();
                    throw new Error("{{i18n PLUGINS_FRONT_EDITION_ERROR_EDITION_INVALID_FIELD}}");
                }
                if (contentId == null)
                {
                    contentId = changedFields[i].contentId;
                }
                else
                {
                    if (contentId != changedFields[i].contentId)
                    {
                        changedFields[i].focus();
                        throw new Error("{{i18n PLUGINS_FRONT_EDITION_ERROR_EDITION_OUTSIDE_CONTENT}}");
                    }
                }
                datas[changedFields[i].metadataPath] = changedFields[i].getJsonValue();
            }
            if (editionWorkflowId == null)
            {
                editionWorkflowId = AmetysFrontEdition.editActionId;
            }
            AmetysFrontEdition.comm.saveMulti(editionWorkflowId, contentId, datas, false, AmetysFrontEdition._saveCb);
        });
	},
    /**
     * Execute a workflow action for a content
     * @param {String} contentId content
     * @param {Number} workflowId workflow action id
     */
    execWorkflowAction: function (contentId, workflowId, callback)
    {
        if (!Ext.isFunction(callback))
        {
            callback = function(response) {
                if (response)
                {
                    AmetysFrontEdition.refresh();
                }
                else
                {
                    alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}");
                    return;
                }
            }
        }
        AmetysFrontEdition.load(function()
        {
            var parameters = {
                "contentId": contentId,
                "id": contentId,
                "comment": "",
                "quit": true,
                "values": {},
                "args": []
            };
            AmetysFrontEdition.comm.doAction(workflowId, parameters, callback);
        });
    },

	/**
     * @private
     * Callback after saving
     * @param {Object} response The server response
     * @param {Object} parameters The parameters
	 */
	_saveCb : function(response, parameters)
    {
        if (response && !Ametys.data.ServerComm.isBadResponse(response))
        {
            let me = this;
            let contentId = parameters.args.contentId;
            let additionalFns = {
                successFn: function(contentId, msg, detailedMsg, workflowMsgErrors, params) {
                    AmetysFrontEdition.refresh();
                },
                workflowValidationErrorFn: function(contentId, msg, detailedMsg, workflowMsgErrors, params) {
                    Ametys.form.SaveHelper.SaveErrorDialog.showErrorDialog (null, msg, detailedMsg);
                },
                fieldResultErrorFn: function(contentId, msg, detailedMsg, fieldsInError, params) {
                    Ametys.form.SaveHelper.SaveErrorDialog.showErrorDialog (null, msg, detailedMsg);
                },
                fieldResultWarningFn: function(contentId, msg, detailedMsg, question, fieldsWithWarning, params) {
                    if (msg) 
                    {
                        Ametys.form.SaveHelper.SaveErrorDialog.showWarningDialog (null, msg, detailedMsg, question, Ext.bind(AmetysFrontEdition._showWarningsCb, me, [params], 1));
                    }
                    else
                    {
                        AmetysFrontEdition._showWarningsCb(true, params);
                    }
                },
                fieldResultInfoFn: function(contentId, msg, detailedMsg, params) {
                    // In case in info msg, it's a success so refresh
                    AmetysFrontEdition.refresh();
                },
                messageFn: function(contentId, messages, globalLevel, params) {
                    // In case in msg, it's a success so refresh
                    AmetysFrontEdition.refresh();
                }
            }
            Ametys.cms.content.ContentErrorHelper.handleContentEditError(response, contentId, parameters, additionalFns)
        }
        else
        {
            alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}");
            return;
        }
    },
    
    /**
     * @private
     * Callback function called after user chose to ignore warnings or not
     * @param {Boolean} ignoreWarnings true if the user choose to ignore warnings and save anyway, false otherwise
     * @param {Object} params The callback arguments.
     */
    _showWarningsCb: function (ignoreWarnings, params)
    {
        if (ignoreWarnings)
        {
            // Save again ignoring warnings
            AmetysFrontEdition.comm.saveMulti(params.args.actionId, params.args.contentId, params.args.datas, true, AmetysFrontEdition._saveCb);
        }
    },
    
    /**
     * Delete a content
     * @param {String} contentId the content id
     * @param {Function} [callback] the callback function
     * @param {Boolean} [withNoConfirm] set to true to skip confirmation before deletion
     */
    deleteContent: function(contentId, callback, withNoConfirm)
    {
        function _internalDelete(contentId)
        {
            // Get page parents before removing content
            // The deletion of a content can lead to the deletion of its page (the page may be a virtual page for example). In this case go to the first available parent page
            AmetysFrontEdition.getPageParents(function() {
              AmetysFrontEdition.comm.deleteContent(contentId, function (response) {
                    if (response && response['deleted-contents'] && response['deleted-contents'].length > 0)
                    {
                        if (Ext.isFunction(callback))
                        {
                            callback(response['deleted-contents']);   
                        }
                        else
                        {
                            // Refresh the current page or go the first available parent page if current page does not exist anymore
                            AmetysFrontEdition.refresh(AmetysFrontEdition.PAGEID);
                        }
                    }
                    else
                    {
                        alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}");
                        return;
                    }
                });
            });
        }
        
        AmetysFrontEdition.load(function() {
            if (withNoConfirm)
            {
                _internalDelete(contentId);
            }
            else
            {
                Ametys.Msg.confirm("{{i18n PLUGINS_FRONT_EDITION_DELETE_CONTENT_CONFIRM_TITLE}}", 
                                "{{i18n PLUGINS_FRONT_EDITION_DELETE_CONTENT_CONFIRM}}", 
                    function(answer) {
                        if (answer == 'yes')
                        {
                            _internalDelete(contentId);
                        }
                    },
                    this
                );
            }
        });
    },
    
    /**
     * Add tags to a content
     * @param {String} contentId the content id
     * @param {String[]} tags the tags to add
     * @param {Function} callback the callback function
     * @param {Object} callbackParameters the parameters for callback function
     */
    addTags: function(contentId, tags, callback, callbackParameters)
    {
        AmetysFrontEdition.load(function() {
            if (!Ext.isFunction(callback))
            {
                callback = AmetysFrontEdition._addTagsCb;
            }
            
            AmetysFrontEdition.comm.addTags(contentId, tags, callback, callbackParameters);
        });
    },
    
    /**
     * @private
     * Callback after adding tags to a content
     * @param {Object} response The server response
     */
    _addTagsCb : function(response)
    {
        if (response && response['allright-contents'] && response['allright-contents'].length > 0)
        {
            AmetysFrontEdition.refresh();
        }
        else
        {
            alert("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_ERROR}}");
            return;
        }
    },
    
    /**
     * Remove tags on a content
     * @param {String} contentId the content id
     * @param {String[]} tags the tags to remove
     * @param {Function} callback the callback function
     * @param {Object} callbackParameters the parameters for callback function
     */
    removeTags: function(contentId, tags, callback, callbackParameters)
    {
        AmetysFrontEdition.load(function() {
            if (!Ext.isFunction(callback))
            {
                callback = AmetysFrontEdition._addTagsCb;
            }
            
            AmetysFrontEdition.comm.removeTags(contentId, tags, callback, callbackParameters);
        });
    },
    
    copyCSSProperties : function(currentField)
    {
		var jTarget = $j(currentField.ametys.target);
        
        var inputId;
        if (currentField.getInputId && (inputId = currentField.getInputId()))
        {
    		var jInput = $j("#" + inputId);
    
    		jInput.css("color", jTarget.css("color"));
    		jInput.css("font-weight", jTarget.css("font-weight"));
    		jInput.css("font-style", jTarget.css("font-style"));
    		jInput.css("font-size", jTarget.css("font-size"));
    		jInput.css("font-family", jTarget.css("font-family"));
    		jInput.css("text-decoration", jTarget.css("text-decoration"));
    		jInput.css("text-transform", jTarget.css("text-transform"));
    		jInput.css("padding-top", jTarget.css("padding-top"));
    		jInput.css("padding-right", jTarget.css("padding-right"));
    		jInput.css("padding-bottom", jTarget.css("padding-botom"));
    		jInput.css("padding-left", jTarget.css("padding-left"));
        }
	},

	/**
     * @private
     * Load all the JS and CSS files
     * @param {Function} callback This function will be called when ExtJS framework is available
     * @param {Boolean} forceLoad true when the load is made at the loading of the page, to avoid to have some display while the js is loaded
     * @param {Funtion} [startCallback] function to invoke after end of the start
	 */
	load : function(callback, forceLoad, startCallback)
    {
		AmetysFrontEdition.start(startCallback);

		if (AmetysFrontEdition.initialized == 3)
        {
			if (Ext.isFunction(callback))//Ext is loaded
            {
                // Call the callback only after EXT is ready
                Ext.onReady(function()
                    {
                        callback();
                    });
            }
		}
        else
        {
            // Yes, we will overwrite a preceding callback... we only want to edit one file
			AmetysFrontEdition.callback = callback;
            if (forceLoad !== true)
            {
                AmetysFrontEdition.showWait();
            }
            
            if (AmetysFrontEdition.initialized == 1)
            {
				AmetysFrontEdition.initialized = 2;

				AmetysFrontEdition.preInit();
                
				// Load all css files
				$j("<link/>", {
					rel : "stylesheet",
					type : "text/css",
                   href: AmetysFrontEdition.CONTEXT_PATH + "/_plugins/web/resources-minimized/" + AmetysFrontEdition.CSS_HASHCODE + ".css"
				}).appendTo("head");

				// Load all js file
                $j.ajax({
                      dataType: "script",
                      cache: true,
                      crossDomain:true,
                      url: AmetysFrontEdition.CONTEXT_PATH + "/_plugins/web/resources-minimized/" + AmetysFrontEdition.JS_HASHCODE + ".js"
                    })
                    .done(AmetysFrontEdition.postInit)
                    .fail(function(jqxhr, settings, exception) {
                        AmetysFrontEdition.hideWait();
                        console.error("An error occurred while loading js files", exception);
                     });
			}
		}
	},

	preInit : function()
    {
		Ext = {
			scopeCss : true
		};
	},

	postInit : function()
    {
		AmetysFrontEdition.initialized = 3;
        Ametys.data.ServerComm.SERVERCOMM_URL = AmetysFrontEdition.CONTEXT_PATH + "/_plugins/front-edition/servercomm/messages.xml";
		Ext.tip.QuickTipManager.init();
        
        //register default message target factory
        var mtFactory = Ext.create("Ametys.message.factory.DefaultMessageTargetFactory", {pluginName: "core", id: "*"});
        Ametys.message.MessageTargetFactory.registerTargetFactory(mtFactory);
        
        AmetysFrontEdition.hideWait();
        
        if (Ext.isFunction(AmetysFrontEdition.callback))
        {
            // Call the callback only after EXT is ready
            Ext.onReady(function()
                {
        			AmetysFrontEdition.callback();
        			AmetysFrontEdition.callback = null;
                });
        }
	},
    
    showWait : function()
    {
        if (AmetysFrontEdition.__waiting !== true)
        {
            $j("html").addClass(AmetysFrontEdition.FRONT_EDITION_JS_LOADING);
            if (AmetysFrontEdition.FRONT_EDITION_JS_LOADING_HTML != null)
            {
                $j(document.body).append("<div id='front-edition-js-loading-div'>" + AmetysFrontEdition.FRONT_EDITION_JS_LOADING_HTML + "</div>");
            }
        }
        AmetysFrontEdition.__waiting = true;
    },
    
    hideWait : function()
    {
        $j("html").removeClass(AmetysFrontEdition.FRONT_EDITION_JS_LOADING);
        $j("html").addClass(AmetysFrontEdition.FRONT_EDITION_JS_LOADED);
        $j("#front-edition-js-loading-div").remove();
        AmetysFrontEdition.__waiting = false;
    },
    
    /**
     * @private
     * Create all needed buttons on the elements that can be edited
     * This is done during the postInit
     */
    _setAllButtons : function()
    {
        var contentIds = [];
        var jElements = $j("[data-ametys-metadata]");
        for (var i = 0; i < jElements.length; i++)
        {
            var jElement = jElements[i];
            var metadataInfos = AmetysFrontEdition._getElementMetadataInfos(jElement);

            var currentContentId = metadataInfos.contentId;
            var metadataPath = metadataInfos.metadataPath;
            if (contentIds.indexOf(currentContentId) == -1)
            {
                contentIds.push(currentContentId);
            }
        }
        
        AmetysFrontEdition.updateModifiableContents(contentIds, function()
            {
                for (var i = 0; i < contentIds.length; i++)
                {
                    AmetysFrontEdition._toggleContentEditable(contentIds[i]);
                }
                
                if (typeof AmetysFrontEdition.startCallback == 'function')
                {
                    AmetysFrontEdition.startCallback();
                }
            }
        );
    },
    
    /**
     * @private
     * Get all fields for this content and add the button/class for editable fields
     * @param {String} contentId
     */
    _toggleContentEditable: function(contentId)
    {
        var isContentEditable = false;
        if (AmetysFrontEdition.modifiableContents.hasOwnProperty(contentId))
        {
            isContentEditable = AmetysFrontEdition.modifiableContents[contentId] && AmetysFrontEdition.modifiableContents[contentId].unmodifiableAttributes != null;
        }

        var jElements = $j("[data-ametys-metadata^='" + contentId + ";']");
        for (var i = 0; i < jElements.length; i++)
        {
            var jElement = $j(jElements[i]);
            var attributeInfo = AmetysFrontEdition._getElementMetadataInfos(jElements[i]);
            var isEditable = isContentEditable && $.inArray(attributeInfo.metadataPath, AmetysFrontEdition.modifiableContents[contentId].unmodifiableAttributes) == -1;
            if (isEditable)
            {
                jElement.addClass(AmetysFrontEdition.CLASSNAME_EDITABLE).prepend("<button class=\"" + AmetysFrontEdition.CLASSNAME_EDITBUTTON + "\">" + AmetysFrontEdition.EDITBUTTON_INNERHTML + "</button>");
            }
            else
            {
                jElement.removeClass(AmetysFrontEdition.CLASSNAME_EDITABLE);
            }
        }
    },
    
    /**
     * @private
     * Check if the given contents are modifiable and update AmetysFrontEdition.modifiableContents
     * @param {String[]} contentIds The ids of contents to check
     * @param {Function} callback function that will be called with the updated contents
     */
    updateModifiableContents: function(contentIds, callback)
    {
        var unknownIds = [];
        for (var i = 0; i < contentIds.length; i++)
        {
            var contentId = contentIds[i];
            if (!AmetysFrontEdition.modifiableContents.hasOwnProperty(contentId))
            {
                unknownIds.push(contentId);
            }
        }
        
        if (unknownIds.length == 0)
        {
            // all contents are already in cache
	        callback();
        }
        else
        {
            AmetysFrontEdition.load(function()
            {
                Ametys.data.ServerComm.callMethod({
                    role: "org.ametys.plugins.frontedition.FrontEditionHelper",
                    methodName: "getModifiableContents",
                    parameters: [AmetysFrontEdition.editActionId, unknownIds, false],
                    callback: {
                        scope: this,
                        handler: AmetysFrontEdition._updateModifiableContentsCb,
                        arguments: [callback]
                    },
                    errorMessage: true
                });
            });
        }
    },
    
    /**
     * @private
     * Callback after checking edition rights
     * @param {Object} result A map with content id as key and true/false as value
     * @param {Object[]} args callback arguments:
     */
    _updateModifiableContentsCb: function(result, args)
    {
        for (var contentId in result)
        {
            if (result.hasOwnProperty(contentId))
            {
                AmetysFrontEdition.modifiableContents[contentId] = result[contentId];
            }
        }
        
        var callback = args[0];
        if (Ext.isFunction(callback))
        {
            callback();
        }
    },

    /**
     * Get the parents' page
     * @param {Function} callback the callback function to call after getting parents (parents will be stored in AmetysFrontEdition.pageParents)
     * @param {String} [pageId=AmetysFrontEdition.PAGEID] The page id. Can be null to get current editing page.
     */
    getPageParents : function(callback, pageId)
    {
        pageId = pageId || AmetysFrontEdition.PAGEID;
        if (pageId)
        {
            var options = {arguments: {callback:callback}};
            Ametys.web.page.PageDAO.getPageParents([pageId], AmetysFrontEdition._getPageParentsCb, options);
        }
    },
    _getPageParentsCb : function(results, args)
    {
        AmetysFrontEdition.pageParents = results.parents;
        if (args.callback && Ext.isFunction(args.callback))
        {
            args.callback();
        }
    },

	/**
	 * Start to listen to events to make document editable
     * @param {Funtion} [callback] function to invoke after end of the start
	 */
	start : function(callback)
    {
        AmetysFrontEdition.startCallback = callback;
        
		if (AmetysFrontEdition.initialized == 0)
        {
            this._initListeners();
			AmetysFrontEdition.initialized = 1;
            $j("document").ready(AmetysFrontEdition._setAllButtons);
		}
	},
    _onMouseMoveListener : function(event)
    {
        AmetysFrontEdition.toggleElementToHover(event.target);
    },
    _onClickListener : function(event)
    {
            AmetysFrontEdition.editElement(event.target);
    },
    _initListeners : function()
    {
        if (AmetysFrontEdition.enableSimpleMode == true || AmetysFrontEdition.enableSimpleMode == 'true')
        {
            document.getElementsByTagName('html')[0].addEventListener("mousemove", AmetysFrontEdition._onMouseMoveListener, true);
            document.getElementsByTagName('html')[0].addEventListener("click", AmetysFrontEdition._onClickListener, true);
        }
    },
    _removeListeners : function()
    {
        document.getElementsByTagName('html')[0].removeEventListener("mousemove", AmetysFrontEdition._onMouseMoveListener, true);
        document.getElementsByTagName('html')[0].removeEventListener("click", AmetysFrontEdition._onClickListener, true);
    },
    addMask: function(text)
    {
        if (text == null || text == '')
        {
            text = "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_SAVING}}";
        }
        Ametys.mask.GlobalLoadMask.mask(text);
    },
    
    /**
     * Refresh the page, filtering the parents if one of them have been deleted/moved/renamed
     * @param {String} [filterPageId] id of the page deleted/moved/renamed/updated (or null) to go to it's parent if needed
     */
	refresh: function(filterPageId)
    {
        Ametys.mask.GlobalLoadMask.mask("{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_RELOADING}}");
        
		if (filterPageId
				&& AmetysFrontEdition.pageParents != undefined
				&& AmetysFrontEdition.pageParents != null)
        {
            // Check if page still exists in current workspace (default for edition mode, live for other mode)
            AmetysFrontEdition.pageExists(filterPageId, function(exists) {
                if (exists)
                {
                    Ametys.reload();
                }
                else
                {
                    // Page does not exists anymore, redirect to its first available parent page
                    var redirectionUrlPrefix = AmetysFrontEdition.CONTEXT_PATH + "/" + AmetysFrontEdition.LANG + "/";
		            for (var i = 0; i < AmetysFrontEdition.pageParents.length; i++)
		            {
		                if (filterPageId == AmetysFrontEdition.pageParents[i].id)
		                {
		                    AmetysFrontEdition._getRedirectUrl(redirectionUrlPrefix, AmetysFrontEdition.pageParents, i + 1);
		                    break;
		                }
		            }
                }
            });
		} else {
			Ametys.reload();
		}
	},
    
    _getRedirectUrl: function(urlPrefix, pageParents, index)
    {
        if (index == pageParents.length)
        {
            // no page parent was found, redirect to index page
            window.location.href = urlPrefix + "index.html";
        }
        else
        {
            var parentId = pageParents[index].id;
            
            // check parent exists
            AmetysFrontEdition.pageExists(parentId, function(exists) {
                if (exists)
                {
                    window.location.href = urlPrefix + pageParents[index].path + ".html";
                }
                else
                {
                    AmetysFrontEdition._getRedirectUrl(urlPrefix, pageParents, index+1);
                }
            })
        }
    },
    
    /**
     * Determines if a page with given id exists
     * @param {String} pageId The page id to test
     * @param {Function} callback the callback function
     * @param {Boolean} callback.exists true if page exists, false otherwise
     */
    pageExists: function(pageId, callback)
    {
        AmetysFrontEdition.load(function()
        {
            var currentPageId = Ametys.getAppParameter("pageId");
            
            // Reset pageId parameter, the current may not exist
            Ametys.setAppParameter("pageId", null);
            
            Ametys.data.ServerComm.callMethod({
                role: "org.ametys.plugins.frontedition.FrontEditionHelper",
                methodName: "pageExists",
                parameters: [pageId, AmetysFrontEdition.EDITION_MODE],
                waitMessage: "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_SAVING}}",
                callback: {
                    scope: this,
                    handler: function(exists){
                        callback(exists);
                    },
                    ignoreOnError: false
                },
                priority: Ametys.data.ServerComm.PRIORITY_LONG_REQUEST, // force long request to ensure the request is sent alone
                errorMessage: true
            });
            
            // restore the current page id
            Ametys.setAppParameter("pageId", currentPageId);
        });
    },
    
    /**
     * Edit content into a popup
     * @param {String} contentId Content ID to filter
     * @param {String} viewName the view name to edit
     * @param {Function} [callback] function to call after edition. If empty default refresh function will be called
     */
    editContent: function(contentId, workflowId, viewName, callback)
    {
        AmetysFrontEdition.load(function()
        {
            // Get page parents before editing content
            // The edition of a content can lead to the deletion of its page (the page may be a virtual page depending on a content attribute for example). In this case go to the first available parent page
            AmetysFrontEdition.getPageParents(function() {
	                Ametys.cms.content.ContentDAO.getContent(contentId, function(content) {
	                
	                Ametys.cms.uihelper.EditContent.open({
		                    content : content,
		                    editWorkflowActionId : AmetysFrontEdition.editActionId,
		                    viewName: viewName || "default-edition",
		                    width: window.innerWidth*0.9,
		                    maxHeight: window.innerHeight*0.9,
		                }, callback || function() { 
                            // Refresh the current page or go the first available parent page if current page does not exist anymore
	                        AmetysFrontEdition.refresh(AmetysFrontEdition.PAGEID)
                        }
                   );
	            });
            });
        });
    }
};
