/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * The static class for edition from front office
 */
AmetysFrontEdition.page = {
    singleton : true,
    createPage : function(pageId, config, callback)
    {
        if (config == null)
        {
            config = {};
        }
        var defaultConfig = {
                                'pagetype-template' : true,
                                'pagetype-redirection-cms' : true,
                                'pagetype-redirection-http' : false,
                                'pagetype-blank' : false,
                                'pagecontent-card-content-filter' :
                                [
                                    'org.ametys.plugins.news.Content.news',
                                    'org.ametys.web.default.Content.article'
                                ],
                                'pagecontent-service' : false
                            };
        AmetysFrontEdition.load(function()
        {
            Ext.applyIf(config, defaultConfig);
            Ametys.plugins.web.page.AddPageWizard.open(pageId, callback, config);
        });
    },
    /**
     * Delete a page
     * @param {String} pageId the id of page to delete
     * @param {String} pageName The page name
     * @param {Object} texts texts a map of texts to use instead of the default values
     * @param {Function} [callback] Optional callback to invoked after deletion
     */
    deletePage : function(pageId, pageName, texts, callback)
    {
        AmetysFrontEdition.load(function()
        {
            if (pageId != null)
            {
                var target =
                {
                    getParameters : function()
                    {
                        return {
                            title : pageName,
                            id : pageId
                        }
                    }
                };

                //get page parents, then remove page, then refresh (with id of deleted page, to go to the 1st available parent if deleted page is an ancestor of current page)
                AmetysFrontEdition.getPageParents(function()
                    {
                    Ametys.plugins.web.sitemap.SitemapActions.remove(target, function()
                        {
                            if (typeof(callback) == 'function')
                            {
                                callback(pageId);
                            }
                            else
                            {
                                // redirect to parent page
                                AmetysFrontEdition.refresh(pageId)
                            }
                        }, texts)
                    }, pageId
                );
            }
        });
    },

    renamePage : function(pageId)
    {
        AmetysFrontEdition.load(function()
        {
            if (pageId != null)
            {
                var conf =
                {
                    'update-path' : false,
                    'alias' : false
                }
                Ametys.plugins.web.sitemap.SitemapActions.renamePage(pageId, conf, function()
                    {
                        AmetysFrontEdition.refresh()
                    }
                );
            }
        });
    },

    movePage : function(pageId, parentId, posIndex, pageName)
    {
        AmetysFrontEdition.load(function()
        {
            if (pageId != null)
            {
                if (posIndex < 0)
                {
                    posIndex = 0;
                }
                Ametys.web.page.PageDAO.movePage([pageId, parentId, posIndex], function()
                    {
                        AmetysFrontEdition.refresh()
                    }, {});
            }
        });
    },

    _confirmMove : function(answer, id, parentId, posIndex)
    {
        if (answer == 'yes')
        {
            Ametys.web.page.PageDAO.movePage([id, parentId, posIndex], function()
                {
                    AmetysFrontEdition.refresh()
                }, {});
        }
    },

    changePageTags : function(pageId)
    {
        AmetysFrontEdition.load(function()
        {
            if (pageId != null)
            {
                Ametys.plugins.web.tag.AffectTagAction.affect([pageId], [], function()
                {
                    AmetysFrontEdition.refresh()
                });
            }
        });
    },

    changeContentTags : function(contentId)
    {
        AmetysFrontEdition.load(function()
        {
            if (contentId != null)
            {
                Ametys.plugins.web.tag.AffectTagAction.affect([], [contentId], function()
                {
                    AmetysFrontEdition.refresh()
                });
            }
        });
    },

    /**
     * Create a new content in a zone in a page
     * 
     * @param {Object} [config={}] The configuration for the new content
     * @param {String[]} [config.contentTypes=null] only mandatory parameter, type of the content
     * @param {String} [config.pageId=AmetysFrontEdition.PAGEID] page where the content will be added (default to current page)
     * @param {String} [config.zoneName='default'] zone name (default to 'default')
     * @param {String} [config.contentTitle=''] content title (default to '')
     * @param {int} [config.initWorkflowActionId=1] content workflow action id for creation
     * @param {String} [config.workflowName='content'] content workflow name
     * @param {String} [config.lang=AmetysFrontEdition.LANG] content language (default to current page language)
     */
    addContent : function(config)
    {
        AmetysFrontEdition.load(function()
        {
            var contentTypes = config.contentTypes.split(";");
            if (contentTypes.length <= 1)
            {
                contentTypes = config.contentTypes;
            }
            var params = {
                contentTypes         : contentTypes,
                defaultContentTitle  : AmetysFrontEdition.page._isDefinedNotNullAndNotEmpty(config.contentTitle) ? config.contentTitle : '{{i18n PLUGINS_FRONT_EDITION_DEFAULT_CONTENT_NAME}}',
                contentLanguage      : AmetysFrontEdition.page._isDefinedNotNullAndNotEmpty(config.lang)? config.lang : AmetysFrontEdition.LANG,
                initWorkflowActionId : AmetysFrontEdition.page._isDefinedNotNullAndNotEmpty(config.initWorkflowActionId) ? config.initWorkflowActionId : 1,
                workflowName         : AmetysFrontEdition.page._isDefinedNotNullAndNotEmpty(config.workflowName) ? config.workflowName : 'content',
                siteName             : AmetysFrontEdition.SITENAME,
                pageId               : AmetysFrontEdition.page._isDefinedNotNullAndNotEmpty(config.pageId) ? config.pageId : AmetysFrontEdition.PAGEID,
                zoneName             : AmetysFrontEdition.page._isDefinedNotNullAndNotEmpty(config.zoneName) ? config.zoneName : 'default'
            };
            var callback = AmetysFrontEdition.refresh;
            Ametys.plugins.web.zone.ZoneActions.addContent(params, callback, this);
        });
    },

    _isDefinedNotNullAndNotEmpty : function(element)
    {
        if (element == undefined)
        {
            return false;
        }
        else if (element == null)
        {
            return false;
        }
        else if (element == "")
        {
            return false;
        }
        return true;
    },

    /**
     * Remove a zone item
     * 
     * @param {Object} [config={}] The configuration for the zone to remove
     * @param {String} [config.pageId=AmetysFrontEdition.PAGEID] page where the content will be added (default to current page)
     * @param {String} [config.zoneName='default'] zone name (default to 'default')
     * @param {String} [config.zoneItemId] id of the item to be removed in the zone
     * @param {Function} [config.callback] Callback function to called after deletion
     */
    removeZoneItem : function(config)
    {
        AmetysFrontEdition.load(function()
        {
            var pageId = config.pageId || AmetysFrontEdition.PAGEID;
            var zoneName = config.zoneName || 'default';
            var zoneItemId = config.zoneItemId;

            Ametys.plugins.web.zone.ZoneActions.removeZoneItem(pageId, zoneName, zoneItemId, function()
                {
                    if (Ext.isFunction(config.callback))
                    {
                        config.callback();
                    }
                    else
                    {
                        AmetysFrontEdition.refresh()
                    }
                });
        });
    },

    /**
     * Move a zone item in the page
     * @param {String} zoneItemId The identifier of the zone item to move
     * @param {String} zoneName The destination zone name
     * @param {Boolean} offset negative for up, positive for down
     */
    moveZoneItem : function (zoneItemId, zoneName, offset)
    {
        AmetysFrontEdition.load(function()
        {
            Ametys.data.ServerComm.callMethod({
                role: "org.ametys.plugins.frontedition.FrontEditionHelper",
                methodName: "moveZoneItemId",
                parameters: [zoneItemId, zoneName, AmetysFrontEdition.PAGEID, offset],
                waitMessage: "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_SAVING}}",
                callback: {
                    scope: this,
                    handler: AmetysFrontEdition.page._moveZoneItemCb
                },
                errorMessage: true
            });
        });
    },
    _moveZoneItemCb : function (success)
    {
        if (success)
        {
            AmetysFrontEdition.refresh()
        }
    },
    
    schedulePublication : function(pageId)
    {
        AmetysFrontEdition.load(function()
        {
            // Simulation of a controller, to return the page id and make a server call
            var controller = {
                getAllRightPageTargets : function()
                {
                    return [{
                        getParameters : function()
                        {
                            return { id : pageId }
                        }
                    }];
                },
                serverCall : function (methodName, parameters, callback, arguments)
                {
                    Ametys.data.ServerComm.callMethod({
                        role: "org.ametys.core.ui.RibbonControlsManager",
                        id: "org.ametys.web.publication.schedule",
                        methodName: methodName,
                        parameters: parameters,
                        waitMessage: "{{i18n PLUGINS_FRONT_EDITION_FRONTCOMM_SAVING}}",
                        callback: {
                            scope: Ametys.plugins.web.page.SchedulePublication,
                            handler: callback,
                            arguments: arguments
                        },
                        errorMessage: true
                    });
                }
            };
            
            // Now, open the dialog
            Ametys.plugins.web.page.SchedulePublication.act(controller);
        });
    }
};
