/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.frontedition;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.w3c.dom.NodeList;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.clientsideelement.styles.HTMLEditorStyle;
import org.ametys.cms.content.RootContentHelper;
import org.ametys.cms.contenttype.AttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.ClientSideElement;
import org.ametys.core.ui.ClientSideElement.Script;
import org.ametys.core.ui.ClientSideElement.ScriptFile;
import org.ametys.core.ui.widgets.richtext.RichTextConfiguration;
import org.ametys.core.ui.widgets.richtext.RichTextConfigurationExtensionPoint;
import org.ametys.core.util.I18nizableSerializer;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.core.ui.minimize.HashCache;
import org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.transformation.xslt.AmetysXSLTHelper;

import com.opensymphony.workflow.loader.ActionDescriptor;
import com.opensymphony.workflow.loader.StepDescriptor;
import com.opensymphony.workflow.loader.WorkflowDescriptor;
import com.opensymphony.workflow.spi.Step;

/**
 * Helper for preparing the Ametys Edition
 */
public class AmetysFrontEditionHelper implements Serviceable, Contextualizable, LogEnabled
{
    /** The right's id from front-edition access */
    public static final String FRONT_EDITION_RIGHT_ID = "Front_Edition_Access_Right";
    
    private static AmetysObjectResolver _ametysObjectResolver;
    private static HashCache _hashCache;
    private static RichTextConfigurationExtensionPoint _richTextConfigurationExtensionPoint;
    private static HTMLEditorStyle _htmlEditorStyle;
    private static RightManager _rightManager;
    private static ContentWorkflowHelper _contentWorkflowHelper;
    private static PageHelper _pageHelper;
    private static WorkflowHelper _workflowHelper;
    private static JSONUtils _jsonUtils;
    private static SiteManager _siteManager;
    private static WorkflowProvider _workflowProvider;
    private static ContentTypesHelper _contentTypesHelper;
    private static ContentTypeExtensionPoint _contentTypesEP;
    private static RootContentHelper _rootContentHelper;
    
    private static Context _context;

    private static Logger _logger;

    public void service(ServiceManager manager) throws ServiceException
    {
        _ametysObjectResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _hashCache = (HashCache) manager.lookup(HashCache.ROLE);
        _richTextConfigurationExtensionPoint = (RichTextConfigurationExtensionPoint) manager.lookup(RichTextConfigurationExtensionPoint.ROLE);
        _htmlEditorStyle = (HTMLEditorStyle) manager.lookup(HTMLEditorStyle.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _contentTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _contentTypesEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _pageHelper = (PageHelper) manager.lookup(PageHelper.ROLE);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _rootContentHelper = (RootContentHelper) manager.lookup(RootContentHelper.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    /**
     * Prepare a hashcode for js files
     * @param locale The language code to use
     * @param theme The url for the theme
     * @return The new hashcode to read the minimified concatened file
     */
    public static String prepareJSFiles(String locale, String theme)
    {
        String themeName = StringUtils.substringAfterLast(theme, "/");
        
        Map<String, String> filesInfos = new HashMap<>();
        filesInfos.put("media", "");
        filesInfos.put("tag", "script");
        
        Map<String, Map<String, String>> files = new LinkedHashMap<>();
        files.put("/plugins/extjs7/resources/ext-all.js", filesInfos);
        files.put("/plugins/extjs7/resources/classic/locale/locale-" + locale + ".js", filesInfos);
        files.put(theme + "/" + themeName + ".js", filesInfos);

        files.put("/plugins/core-ui/resources/js/Ext.fixes." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ext.enhancements." + locale + ".js", filesInfos);
        
        files.put("/plugins/core-ui/resources/js/Ametys." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/mask/GlobalLoadMask." + locale + ".js", filesInfos);
        
        files.put("/plugins/core-ui/resources/js/Ametys/log/Logger." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/log/Logger/Entry." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/log/LoggerFactory." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/log/ErrorDialog." + locale + ".js", filesInfos);
        
        files.put("/plugins/core-ui/resources/js/Ametys/window/DialogBox." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/window/DialogBox." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/window/MessageBox." + locale + ".js", filesInfos);
        
        files.put("/plugins/core-ui/resources/js/Ametys/form/AbstractField." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/AbstractFieldsWrapper." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/AbstractQueryableComboBox." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/AbstractQueryableComboBox/SplitterTracker." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/form/widget/AbstractQueryableComboBox." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/DateTime." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/StringTime." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/Password." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/ChangePassword." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/ReferencedNumberField." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/RichText." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/form/field/RichText." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/RichText/SplitterTracker." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/TextArea." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/ColorSelector." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/SelectUserDirectory." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/SelectGroupDirectories." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/field/Code." + locale + ".js", filesInfos);
        
        files.put("/plugins/core-ui/resources/js/Ametys/helper/ChooseLocationLeaflet." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/GeoCode." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/grid/GridColumnHelper." + locale + ".js", filesInfos);
        files.put("/plugins/tiny_mce/resources/js/tinymce.js", filesInfos);
        files.put("/plugins/codemirror/resources/js/codemirror.js", filesInfos);
        files.put("/plugins/codemirror/resources/js/addon/edit/matchbrackets.js", filesInfos);
        files.put("/plugins/codemirror/resources/js/addon/selection/active-line.js", filesInfos);
        files.put("/plugins/codemirror/resources/js/mode/xml/xml.js", filesInfos);
        files.put("/plugins/codemirror/resources/js/mode/javascript/javascript.js", filesInfos);
        files.put("/plugins/codemirror/resources/js/mode/css/css.js", filesInfos);
        files.put("/plugins/codemirror/resources/js/mode/htmlmixed/htmlmixed.js", filesInfos);
        
        files.put("/plugins/core-ui/resources/js/Ametys/data/ServerCaller." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/data/ServerComm." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/data/ServerComm." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/data/ServerCommProxy." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/data/ServerComm/TimeoutDialog." + locale + ".js", filesInfos);

        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/RichText/RichTextConfigurationBase." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/RichText/RichTextConfiguration." + locale + ".js", filesInfos);
        
        files.put("/plugins/core-ui/resources/js/Ametys/plugins/coreui/system/devmode/StacktraceHelper." + locale + ".js", filesInfos);

        files.put("/plugins/front-edition/resources/js/front-comm." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/front-widget." + locale + ".js", filesInfos);
//        files.put("/plugins/front-edition/resources/js/front-page." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/helper/EnterURL." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/helper/FileUpload." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/helper/ChoosePage." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/helper/ContextToolbar." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/sitemap/SitemapTree." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/sitemap/SitemapTree/Page." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/sitemap/SitemapTree/Sitemap." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/site/SitesTree/Site." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/editor/LinkHandler." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/editor/Links." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/editor/Links/LinkHandler." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/editor/Links." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/editor/PageLinkHandler." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/editor/Images." + locale + ".js", filesInfos);
        
        files.put("/plugins/cms/resources/js/tinymce/basicstyles." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/tinymce/lists." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/tinymce/links." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/tinymce/links." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/tinymce/links." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/tinymce/styles." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/tinymce/images." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/tinymce/images." + locale + ".js", filesInfos);
        
        if (PluginsManager.getInstance().getPluginNames().contains("inlinemedia"))
        {
            
            files.put("/plugins/inlinemedia/resources/js/Ametys/plugins/inlinemedia/Media." + locale + ".js", filesInfos);
            files.put("/plugins/inlinemedia/resources/js/Ametys/plugins/inlinemedia/InsertMediaHelper." + locale + ".js", filesInfos);
            files.put("/plugins/inlinemedia/resources/js/tinymce/images." + locale + ".js", filesInfos);
        }
        
        files.put("/plugins/front-edition/resources/js/tinymce/save." + locale + ".js", filesInfos);
        
        //TinyMCE Tables
        files.put("/plugins/cms/resources/js/tinymce/tables." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/tinymce/tables." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/editor/Tables." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/editor/BasicActions." + locale + ".js", filesInfos);
        //TinyMCE Images
        files.put("/plugins/cms/resources/js/Ametys/cms/uihelper/ChooseResource." + locale + ".js", filesInfos);
        files.put("/plugins/explorer/resources/js/Ametys/explorer/tree/ExplorerTree." + locale + ".js", filesInfos);
        files.put("/plugins/explorer/resources/js/Ametys/explorer/tree/ExplorerTree/NodeEntry." + locale + ".js", filesInfos);
        files.put("/plugins/explorer/resources/js/Ametys/explorer/Resource." + locale + ".js", filesInfos);
        files.put("/plugins/explorer/resources/js/Ametys/explorer/ExplorerNodeDAO." + locale + ".js", filesInfos);
        files.put("/plugins/explorer/resources/js/Ametys/explorer/ExplorerNode." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/explorer/ExplorerNode." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/attach/AttachmentsExplorerTree." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/uihelper/ChooseAttachmentFile." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/cms/uihelper/ChooseAttachmentFile." + locale + ".js", filesInfos);
        files.put("/plugins/explorer/resources/js/Ametys/explorer/resources/actions/FileActions." + locale + ".js", filesInfos);
        files.put("/plugins/explorer/resources/js/Ametys/explorer/resources/helper/ResourceUpload." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/file/AbstractFileExplorerTree/FileNode." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/file/AbstractFileExplorerTree." + locale + ".js", filesInfos);

        //Empty Message class
        files.put("/plugins/core-ui/resources/js/Ametys/message/Message." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/message/MessageTarget." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/message/MessageTargetFactory." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/message/factory/DefaultMessageTargetFactory." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/message/MessageTargetHelper." + locale + ".js", filesInfos);
        //add page
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/plugins/web/page/AddPageWizard." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/sitemap/SitemapDAO." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/sitemap/Sitemap." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/page/PageDAO." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/web/page/PageDAO." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard/Card." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard/CreatePageCard." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard/PageContentCard." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard/ContentPropertiesCard." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/Widget." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/ConfigurableFormPanel." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/userprefs/UserPrefsDAO." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/ConfigurableFormPanel/Repeater." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/ConfigurableFormPanel/FieldCheckersManager." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/message/MessageBus." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/WidgetManager." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/DefaultWidgets." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/Date." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/DateTime." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/form/WidgetManager." + locale + ".js", filesInfos);
        
        files.put("/plugins/cms/resources/js/Ametys/cms/content/ContentDAO." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/content/ContentDAO." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/cms/content/ContentDAO." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/content/Content." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/navhistory/HistoryDAO." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/window/MessageBox." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/tool/ToolsManager." + locale + ".js", filesInfos);

        files.put("/plugins/cms/resources/js/Ametys/cms/form/widget/RichText." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/cms/form/widget/SelectContent/ContentEntry." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/form/widget/SelectContent." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/cms/form/widget/SelectContent." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/form/widget/SelectReferenceTableContent." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/cms/form/widget/SelectReferenceTableContent." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard/PageTypeCard." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/plugins/web/page/AddPageWizard/PageTypeCard." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/form/widget/SelectPage." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/web/form/widget/SelectPage." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/form/widget/SelectSite." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/helper/ChooseSite." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/site/SitesTree." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/site/SitesTree/Site." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/helper/ContextToolbar." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/site/SitesTree/Site." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/form/widget/SelectPage/PageEntry." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard/TagsCard." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/AddPageWizard/TagsCardLite." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/page/Page." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/tag/TagsTreePanel." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/tag/TagsTreePanel/TagNode." + locale + ".js", filesInfos);

        //delete page
        files.put("/plugins/web/resources/js/Ametys/plugins/web/sitemap/SitemapActions." + locale + ".js", filesInfos);

        //page tags
        files.put("/plugins/web/resources/js/Ametys/plugins/web/tag/AffectTagAction." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/uihelper/ChooseTagHelper." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/uihelper/ChooseTag." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/web/helper/ChooseTag." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/plugins/cms/tag/TagActions." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/tag/TagsTreePanel." + locale + ".js", filesInfos);
        
        // scheduled publication
        files.put("/plugins/web/resources/js/Ametys/plugins/web/page/SchedulePublication." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/plugins/web/page/SchedulePublication." + locale + ".js", filesInfos);

        // ping
        files.put("/plugins/core-ui/resources/js/Ametys/plugins/coreui/system/StartTimeChecker." + locale + ".js", filesInfos);

        // actions on content
        files.put("/plugins/cms/resources/js/Ametys/cms/uihelper/EditContent." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/zone/ZoneActions." + locale + ".js", filesInfos);
        files.put("/plugins/web/resources/js/Ametys/plugins/web/zone/ZoneDAO." + locale + ".js", filesInfos);

        // file widget (+cropping for image filter)
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/File." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/form/widget/File." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/File/FileSource." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/File/External." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/form/widget/File/Resource." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/helper/crop/CropDialog." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Jcrop/jquery.Jcrop." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/widget/Image." + locale + ".js", filesInfos);
        files.put("/plugins/front-edition/resources/js/Ametys/override/Ametys/form/widget/Image." + locale + ".js", filesInfos);
        
        // save errors and warnings
        files.put("/plugins/core-ui/resources/js/Ametys/form/SaveHelper." + locale + ".js", filesInfos);
        files.put("/plugins/core-ui/resources/js/Ametys/form/SaveHelper/SaveErrorDialog." + locale + ".js", filesInfos);
        files.put("/plugins/cms/resources/js/Ametys/cms/content/ContentErrorHelper." + locale + ".js", filesInfos);
        
        // Let's add all richtextconfiguration files
        _addRichTextConfigurationFiles(files, filesInfos, locale, true);

        return _hashCache.createHash(files, locale);
    }

    /**
     * Prepare a hashcode for css files
     * @param locale The language code to use
     * @param theme The url for the theme
     * @return The new hashcode to read the minimified concatened file
     */
    public static String prepareCSSFiles(String locale, String theme)
    {
        Map<String, String> filesInfos = new HashMap<>();
        filesInfos.put("media", "");
        filesInfos.put("tag", "link");
        
        Map<String, Map<String, String>> files = new LinkedHashMap<>();
        String themeName = StringUtils.substringAfterLast(theme, "/");
        files.put(theme + "/" + themeName + "-all.css", filesInfos);
        files.put("/plugins/front-edition/resources/css/theme-ametys-base-light.css", filesInfos);
        files.put("/plugins/core-ui/resources/font/ametys/AmetysIcon.css", filesInfos);
        files.put("/plugins/core-ui/resources/css/Ametys/form/image.css", filesInfos);
        files.put("/plugins/core-ui/resources/css/Jcrop/jquery.Jcrop.css", filesInfos);
        files.put("/plugins/web/resources/css/pages/addpage.css", filesInfos);
        files.put("/plugins/cms/resources/css/selectcontent.css", filesInfos);
        files.put("/plugins/codemirror/resources/css/codemirror.css", filesInfos);
        files.put("/plugins/codemirror/resources/js/addon/hint/show-hint.css", filesInfos);
        
        // Let's add all richtextconfiguration files
        _addRichTextConfigurationFiles(files, filesInfos, locale, false);
        
        return _hashCache.createHash(files, locale);
    }

    private static List<ScriptFile> _getRichTextConfigurationFiles(boolean scripts, Map<String, Object> contextParameters)
    {
        List<ScriptFile> files = new ArrayList<>();
        
        for (String richTextConfigurationId : _richTextConfigurationExtensionPoint.getExtensionsIds())
        {
            RichTextConfiguration richTextConfiguration = _richTextConfigurationExtensionPoint.getExtension(richTextConfigurationId);
            for (String category : richTextConfiguration.getCategories())
            {
                Set<ClientSideElement> convertors = richTextConfiguration.getConvertors(category, contextParameters);
                _addFilesForClientSideElements(scripts, files, convertors);

                Set<ClientSideElement> validators = richTextConfiguration.getValidators(category, contextParameters);
                _addFilesForClientSideElements(scripts, files, validators);
                
                if (!scripts)
                {
                    List<ScriptFile> backOfficeCSSFiles = _htmlEditorStyle.getBackOfficeCSSFiles(category, contextParameters);
                    if (backOfficeCSSFiles != null)
                    {
                        files.addAll(backOfficeCSSFiles);
                    }
                }
            }
        }

        return files;
    }

    private static void _addFilesForClientSideElements(boolean scripts, List<ScriptFile> files, Set<ClientSideElement> clientSideElmts)
    {
        if (clientSideElmts != null)
        {
            for (ClientSideElement convertor : clientSideElmts)
            {
                List<Script> scriptsToAdd = convertor.getScripts(Collections.EMPTY_MAP);
                if (scriptsToAdd != null)
                {
                    for (Script script : scriptsToAdd)
                    {
                        List<ScriptFile> scriptFiles = scripts ? script.getScriptFiles() : script.getCSSFiles();
                        if (scriptFiles != null)
                        {
                            files.addAll(scriptFiles);
                        }
                    }
                }
            }
        }
    }

    private static void _addRichTextConfigurationFiles(Map<String, Map<String, String>> files, Map<String, String> filesInfos, String locale, boolean scripts)
    {
        Map<String, Object> contextParameters = new HashMap<>();
        contextParameters.put("siteName", AmetysXSLTHelper.site());
        
        List<ScriptFile> richTextConfigurationFiles = _getRichTextConfigurationFiles(scripts, contextParameters);
        for (ScriptFile file : richTextConfigurationFiles)
        {
            String uri = null;
            if (file.isLangSpecific())
            {
                if (file.getLangPaths().containsKey(locale))
                {
                    uri = file.getLangPaths().get(locale);
                }
                else if (file.getLangPaths().containsKey(file.getDefaultLang()))
                {
                    uri = file.getLangPaths().get(file.getDefaultLang());
                }
            }
            else
            {
                uri = file.getPath();
            }
            
            if (uri != null)
            {
                files.put(uri, filesInfos);
            }
        }
    }
    /**
     * Check if we can display the front edition button for a specific right
     * Checks :
     * * Rendering context == front
     * * Edition mode
     * * Front_Edition_Access_Right on current page
     * * RightId available  on object
     * @param rightId right to check (can be null)
     * @param objectId id of page/content to check. Can be null or empty to get the current page
     * @return true if all is OK, false if at least one is false
     * @deprecated Use the version with three arguments
     */
    @Deprecated
    public static boolean hasFrontEditionRight(String rightId, String objectId)
    {
        return hasFrontEditionRight(rightId, objectId, true);
    }
    /**
     * Check if we can display the front edition button for a specific right
     * Checks :
     * * Rendering context == front
     * * Edition mode
     * * Front_Edition_Access_Right on current page
     * * RightId available on object
     * @param rightId right to check (can be null)
     * @param objectId id of page/content to check. Can be null or empty to get the current page
     * @param editionModeOnly Check if the user is in edition mode
     * @return true if all is OK, false if at least one is false
     */
    public static boolean hasFrontEditionRight(String rightId, String objectId, boolean editionModeOnly)
    {
        String aoId = objectId;
        if (StringUtils.isEmpty(aoId))
        {
            aoId = AmetysXSLTHelper.pageId();
        }
        
        if (StringUtils.isEmpty(aoId)                                 // No page or content to check
            || !RenderingContext.FRONT.toString().equals(AmetysXSLTHelper.renderingContext())     // We are no on front
            || editionModeOnly && !AmetysXSLTHelper.isEditionMode()   // Not in edition mode
            || !hasFrontEditionRight())                                 // The user has the front edition right on current page
        {
            
            return false;
        }
        
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace (rights have to be checked on default workspace)
            // The target objectId can be a node page, available of front only when a first subpage will be created
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            if (StringUtils.contains(aoId, ";"))
            {
                String[] pageIds = StringUtils.split(aoId, ";");
                for (String pid : pageIds)
                {
                    if (_rightManager.currentUserHasRight(rightId, _ametysObjectResolver.resolveById(pid)) == RightResult.RIGHT_ALLOW)   // Check right on page (If this page does not exists an exception will be thrown... this is fine)
                    {
                        return true;
                    }
                    // else check the next one
                }
                return false; // No right inside the list
            }
            else
            {
                return StringUtils.isEmpty(rightId) // There is no right to check OR the user has the right
                    || _rightManager.currentUserHasRight(rightId, _ametysObjectResolver.resolveById(aoId)) == RightResult.RIGHT_ALLOW;   // Check right on page (If this page does not exists an exception will be thrown... this is fine)
            }
        }
        finally
        {
            // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Check if current user has right on Ametys object for front edition
     * Checks :
     * * Rendering context == front
     * * Edition mode
     * * Front_Edition_Access_Right on current page
     * * RightId available on object
     * @param rightId right to check (can be null)
     * @param ao The ametys object check. Cannot be null
     * @param editionModeOnly Check if the user is in edition mode
     * @return true if all is OK, false if at least one is false
     */
    public static boolean hasFrontEditionRight(String rightId, AmetysObject ao, boolean editionModeOnly)
    {
        if (!RenderingContext.FRONT.toString().equals(AmetysXSLTHelper.renderingContext())     // We are no on front
            || editionModeOnly && !AmetysXSLTHelper.isEditionMode()   // Not in edition mode
            || !hasFrontEditionRight())                                 // The user has the front edition right on current page
        {
            return false;
        }
        
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace (rights have to be checked on default workspace)
            // The target objectId can be a node page, available of front only when a first subpage will be created
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            return StringUtils.isEmpty(rightId) // There is no right to check OR the user has the right
                    || _rightManager.currentUserHasRight(rightId, ao) == RightResult.RIGHT_ALLOW;   // Check right on page (If this page does not exists an exception will be thrown... this is fine)
        }
        finally
        {
            // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Check if the current user has the Front_Edition_Access_Right right
     * @return true if right granted
     */
    public static boolean hasFrontEditionRight()
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace (rights have to be checked on default workspace)
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            
            String pageId = AmetysXSLTHelper.pageId();
            if (StringUtils.isNotEmpty(pageId))
            {
                Page page = _ametysObjectResolver.resolveById(pageId);
                return _rightManager.currentUserHasRight(FRONT_EDITION_RIGHT_ID, page) == RightResult.RIGHT_ALLOW;
            }
            
            String lang = AmetysXSLTHelper.lang();
            if (StringUtils.isEmpty(lang))
            {
                lang = (String) request.getAttribute("lang");
            }
            String siteName = AmetysXSLTHelper.site();
            
            if (StringUtils.isNotEmpty(siteName) && StringUtils.isNoneEmpty(lang))
            {
                Site site = _siteManager.getSite(siteName);
                Sitemap sitemap = site.getSitemap(lang);
                return _rightManager.currentUserHasRight(FRONT_EDITION_RIGHT_ID, sitemap) == RightResult.RIGHT_ALLOW;
            }
            
            return false;
        }
        finally
        {
            // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
        
        
    }
    /**
     * Check if the current workflow action is available on this content
     * @param actionId action to check
     * @param contentId content to check
     * @param checkEditionMode check also if we are in edition mode
     * @return true if right is granted
     */
    public static boolean hasWorkflowRight(int actionId, String contentId, boolean checkEditionMode)
    {
        List<Integer> actionIds = new ArrayList<>();
        actionIds.add(actionId);
        return hasWorkflowRight(actionIds, contentId, checkEditionMode);
    }
    /**
     * Check if a list of workflow actions are available on this content
     * @param actionIds list of action ids
     * @param contentId id of the content to check
     * @param checkEditionMode check also if we are in edition mode
     * @return true if all actions are available
     */
    public static boolean hasWorkflowRight(List<Integer> actionIds, String contentId, boolean checkEditionMode)
    {
        Content content = _ametysObjectResolver.resolveById(contentId);
        return hasWorkflowRight(actionIds, content, checkEditionMode);
    }
    /**
     * Check if a list of workflow actions are available on this content
     * @param actionIds list of action ids
     * @param content content to check
     * @param checkEditionMode check also if we are in edition mode
     * @return true if all actions are available
     */
    public static boolean hasWorkflowRight(List<Integer> actionIds, Content content, boolean checkEditionMode)
    {
        if (checkEditionMode)
        {
            boolean editionModeOk = AmetysXSLTHelper.isEditionMode();
            if (!editionModeOk)
            {
                return false;
            }
        }
        boolean result = false;
        if (content instanceof WorkflowAwareContent)
        {
            WorkflowAwareContent wcontent = (WorkflowAwareContent) content;
            int[] availableActions = _contentWorkflowHelper.getAvailableActions(wcontent);
            for (int actionId : actionIds)
            {
                result = result || ArrayUtils.contains(availableActions, actionId);
                if (result)
                {
                    break;
                }
            }
        }
        return result;
    }
    
    /**
     * Get the toolbar configuration for page: retrieve the user rights, the parent and page position
     * @param pageId The page id
     * @param addContentsConfigAsJson The items configuration for adding contents as JSON string
     * @param editionModeOnly true to check edition mode
     * @return the user's rights, parent and page position
     * @throws UnknownAmetysObjectException if content is unknown
     * @throws AmetysRepositoryException if an error occurred
     * @throws RepositoryException if an error occurred
     */
    public static String getPageToolbarConfig(String pageId, String addContentsConfigAsJson, boolean editionModeOnly) throws UnknownAmetysObjectException, AmetysRepositoryException, RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _ametysObjectResolver.resolveById(pageId);
        SitemapElement parent = page.getParent();
        long position = _getPagePosition(page);
        
        List<Object> addContentsConfig = _jsonUtils.convertJsonToList(addContentsConfigAsJson);
        
        @SuppressWarnings("unchecked")
        Set<Object> allowedContentTypes = addContentsConfig.stream()
            .filter(Map.class::isInstance)
            .filter(cfg -> _hasContentCreationRight((Map) cfg))
            .map(cfg -> (String) ((Map) cfg).get("contentType"))
            .collect(Collectors.toSet());
        result.put("allowedContentTypes", allowedContentTypes);
        
        result.put("title", page.getTitle());
        result.put("parentId", parent.getId());
        result.put("position", position);

        // Get the active status for each action depending on user's rights, page position, FO edition mode, ...
        result.put("actionStatus", _getActionStatusOnPage(page, position, parent.getChildrenPages().getSize(), editionModeOnly));
        
        return _jsonUtils.convertObjectToJson(result);
    }
    
    private static boolean _hasContentCreationRight(Map<String, Object> config)
    {
        if (config.containsKey("contentType"))
        {
            String cTypeId = (String) config.get("contentType");
            
            ContentType contentType = _contentTypesEP.getExtension(cTypeId);
            if (contentType == null)
            {
                _logger.warn("Unknown content type '" + cTypeId + "'");
                return false;
            }
            
            String right = contentType.getRight();
            if (StringUtils.isEmpty(right))
            {
                return true;
            }
            
            return _rightManager.currentUserHasRight(right, "/cms") == RightResult.RIGHT_ALLOW || _rightManager.currentUserHasRight(right, _rootContentHelper.getRootContent()) == RightResult.RIGHT_ALLOW;
        }
        else
        {
            _logger.warn("Configuration to add content is invalid: missing 'contentType'. It will be ignored");
            return false;
        }
    }
    
    private static long _getPagePosition(Page page) throws AmetysRepositoryException
    {
        SitemapElement parent = page.getParent();
        AmetysObjectIterable< ? extends Page> children = parent.getChildrenPages();
        
        long count = 1;
        for (Page child : children)
        {
            if (child.getId().equals(page.getId()))
            {
                return count;
            }
            count++;
        }
        
        // Not found
        return -1;
    }
    
    private static Map<String, Boolean> _getActionStatusOnPage(Page page, long position, long size, boolean editionModeOnly) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        Map<String, Boolean> rights = new HashMap<>();
        
        if (page instanceof ModifiablePage)
        {
            rights.put("add-content", hasFrontEditionRight("Web_Rights_Page_AddContent", page, editionModeOnly));
            rights.put("add-page", hasFrontEditionRight("Web_Rights_Page_Create", page, editionModeOnly));
            rights.put("tag", hasFrontEditionRight("Web_Rights_Page_Tag", page, editionModeOnly));
            rights.put("rename", hasFrontEditionRight("Web_Rights_Page_Rename", page, editionModeOnly));
            rights.put("schedule-publication", hasFrontEditionRight("Web_Rights_Page_Schedule", page, editionModeOnly));
            rights.put("delete", hasFrontEditionRight("Web_Rights_Page_Delete", page, editionModeOnly));
            
            boolean canMove = hasFrontEditionRight("Web_Rights_Page_Create", page.getParent().getId(), editionModeOnly);
            boolean moveUp = canMove && position > 1;
            boolean moveDown = canMove && position < size;
            
            rights.put("move-up", moveUp);
            rights.put("move-down", moveDown);
            rights.put("move", moveDown || moveUp);
        }
        
        return rights;
    }
    
    /**
     * Get the toolbar configuration for content: retrieve user rights, available workflow actions, content current step and status history
     * @param contentId The content id
     * @param zoneItemId The zone item id
     * @param restrictedActions The actions ids to retrieve. Set to null or empty to get all available workflow actions with no restriction
     * @param jsonStepConfig The step configuration
     * @param locale the locale for translation
     * @param editionModeOnly true to check edition mode
     * @return the user's rights, available workflow actions and content current step
     * @throws UnknownAmetysObjectException if content is unknown
     * @throws AmetysRepositoryException if an error occurred
     * @throws RepositoryException if an error occurred
     */
    public static String getContentToolbarConfig(String contentId, String zoneItemId, String restrictedActions, String jsonStepConfig, String locale, boolean editionModeOnly) throws UnknownAmetysObjectException, AmetysRepositoryException, RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        
        Content content = _ametysObjectResolver.resolveById(contentId);

        // Get the active status of each action depending on user rights, FO mode, content's position, ...
        result.put("actionStatus", _getActionStatusOnContent(content, zoneItemId, editionModeOnly));
        
        if (content instanceof WorkflowAwareContent)
        {
            WorkflowDescriptor workflowDescriptor = _getWorkflowDescriptor((WorkflowAwareContent) content);
            
            WorkflowAwareContent wcontent = (WorkflowAwareContent) content;
            
            Map<String, Object> step = new HashMap<>();
            long currentStepId = wcontent.getCurrentStepId();
            
            // Current step
            StepDescriptor stepDescriptor = _getStepDescriptor(workflowDescriptor, currentStepId);
            step.put("id", currentStepId);
            String name = stepDescriptor.getName();
            step.put("label", new I18nizableText(StringUtils.substringBefore(name, ":"), StringUtils.substringAfter(name, ":")));
            result.put("step", step);
            
            Map<String, Object> stepConfig = StringUtils.isEmpty(jsonStepConfig) ? null : _jsonUtils.convertJsonToMap(jsonStepConfig);
            result.put("statusHistory", _getStatusHistory(wcontent, currentStepId, stepConfig, workflowDescriptor));
            
            List<Integer> availableActions = Arrays.stream(_contentWorkflowHelper.getAvailableActions(wcontent)).boxed().collect(Collectors.toList());
            result.put("availableWorkflowActions", availableActions);
            
            if (!editionModeOnly || AmetysXSLTHelper.isEditionMode())
            {
                List<Object> restrictedActionsAsList = StringUtils.isEmpty(restrictedActions) ? null : _jsonUtils.convertJsonToList(restrictedActions);
                
                List<Object> restrictedActionIds = restrictedActionsAsList == null ? null : restrictedActionsAsList.stream()
                    .map(o -> o instanceof Map ? ((Map) o).get("id") : o)
                    .collect(Collectors.toList());
                
                @SuppressWarnings("unchecked")
                Map<Object, Map<String, Object>> workflowActions = availableActions.stream()
                    .filter(actionId -> restrictedActionIds == null || restrictedActionIds.contains(actionId))
                    .map(actionId -> {
                        Map<String, Object> wa = new HashMap<>();
                        
                        _addDefaultWorkflowActionProperties(wa, workflowDescriptor, actionId);
                        
                        if (restrictedActionIds != null && restrictedActionsAsList != null && restrictedActionIds.contains(actionId))
                        {
                            for (Object object : restrictedActionsAsList)
                            {
                                if (object instanceof Map && ((Map) object).get("id") == actionId)
                                {
                                    // get or override properties from map sent by client side
                                    wa.putAll((Map<String, Object>) object);
                                    break;
                                }
                            }
                            
                        }
                        return Pair.of(actionId, wa);
                    })
                    .collect(Collectors.toMap(Pair::getLeft, Pair::getRight));
                
                result.put("workflowActions", workflowActions);
            }
        }
        
        Request request = ContextHelper.getRequest(_context);
        try
        {
            request.setAttribute(I18nizableSerializer.REQUEST_ATTR_LOCALE, locale);
            return _jsonUtils.convertObjectToJson(result);
        }
        finally
        {
            request.removeAttribute(I18nizableSerializer.REQUEST_ATTR_LOCALE);
        }
    }
    
    private static Map<String, Boolean> _getActionStatusOnContent(Content content, String zoneItemId, boolean editionModeOnly) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        Map<String, Boolean> rights = new HashMap<>();
        
        ZoneItem zoneItem = _ametysObjectResolver.resolveById(zoneItemId);
        Zone zone = zoneItem.getZone();
        SitemapElement sitemapElement = zone.getSitemapElement();
        
        rights.put("tag", hasFrontEditionRight("CMS_Rights_Content_Tag", content, editionModeOnly));
        rights.put("remove", sitemapElement instanceof ModifiablePage && hasFrontEditionRight("Web_Rights_Page_DeleteZoneItem", sitemapElement, editionModeOnly));
        rights.put("delete", hasFrontEditionRight("CMS_Rights_DeleteContent", content, editionModeOnly));
        
        boolean moveUp = false;
        boolean moveDown = false;
        
        long position = getZoneItemPosition(zoneItem);
        long size = getZoneSize(zone);
        
        if (position != -1 && sitemapElement instanceof ModifiablePage && hasFrontEditionRight("Web_Rights_Page_OrganizeZoneItem", sitemapElement, editionModeOnly))
        {
            if (position > 0)
            {
                moveUp = true;
            }
            
            if (position + 1 < size)
            {
                moveDown = true;
            }
        }
        rights.put("move-up", moveUp);
        rights.put("move-down", moveDown);
        rights.put("move", moveDown || moveUp);
        
        return rights;
    }
    
    private static Map<Integer, Map<String, Object>> _getStatusHistory(WorkflowAwareContent content, long currentStepId, Map<String, Object> stepsConfig, WorkflowDescriptor workflowDescriptor)
    {
        List<Step> historyStatus = new ArrayList<>();
            
        if (stepsConfig.containsKey(String.valueOf(currentStepId)))
        {
            AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(content);
            
            long workflowId = content.getWorkflowId();
            List<Step> historySteps = workflow.getHistorySteps(workflowId);

            // Sort by start date descendant
            Collections.sort(historySteps, new Comparator<Step>()
            {
                public int compare(Step s1, Step s2)
                {
                    return -s1.getStartDate().compareTo(s2.getStartDate());
                }
            });
            
            @SuppressWarnings("unchecked")
            Map<String, Object> currentStepCfg = (Map<String, Object>) stepsConfig.get(String.valueOf(currentStepId));
            String currentStepType = currentStepCfg != null ? (String) currentStepCfg.get("type") : null;
            
            boolean stop = false;
            for (Step step : historySteps)
            {
                if (stop)
                {
                    break;
                }

                long stepId = step.getStepId();
                @SuppressWarnings("unchecked")
                Map<String, Object> stepCfg = (Map<String, Object>) stepsConfig.get(String.valueOf(stepId));
                String stepType = stepCfg != null ? (String) stepCfg.get("type") : null;
                
                stop = "positive".equals(currentStepType) && !"positive".equals(stepType)
                        || ("negative".equals(currentStepType) || "neutral".equals(currentStepType)) && !"negative".equals(stepType)
                        || currentStepId == stepId;
                
                if (!stop)
                {
                    historyStatus.add(step);
                }
            }
        }
        
        Map<Integer, Map<String, Object>> jsonHistoryStatus = new LinkedHashMap<>();
        for (Step step : historyStatus)
        {
            if (!jsonHistoryStatus.containsKey(step.getStepId()))
            {
                StepDescriptor stepDescriptor = _getStepDescriptor(workflowDescriptor, step.getStepId());
                String name = stepDescriptor.getName();
                I18nizableText label = new I18nizableText(StringUtils.substringBefore(name, ":"), StringUtils.substringAfter(name, ":"));
                jsonHistoryStatus.put(step.getStepId(), Map.of("label", label));
            }
        }
        
        return jsonHistoryStatus;
    }
    
    private static void _addDefaultWorkflowActionProperties(Map<String, Object> wa, WorkflowDescriptor workflowDescriptor, int actionId)
    {
        // Add workflow action label
        ActionDescriptor wAction = workflowDescriptor.getAction(actionId);
        String name = wAction.getName();
        wa.put("label", new I18nizableText(StringUtils.substringBefore(name, ":"), StringUtils.substringAfter(name, ":")));
        
        // Default workflow icon
        switch (actionId)
        {
            case 3:
                wa.put("icon", "fas fa-hand-point-up"); // Propose
                break;
            case 4:
                wa.put("icon", "fas fa-check"); // Validate
                break;
            case 7:
                wa.put("icon", "fas fa-times"); // Refuse
                break;
            case 10:
                wa.put("icon", "fas fa-ban"); // Unpublish
                break;
            default:
                break;
        }
    }
    
    private static WorkflowDescriptor _getWorkflowDescriptor(WorkflowAwareContent content)
    {
        long workflowId = content.getWorkflowId();
        
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(content);
        
        String workflowName = workflow.getWorkflowName(workflowId);
        return workflow.getWorkflowDescriptor(workflowName);
    }
    
    private static StepDescriptor _getStepDescriptor(WorkflowDescriptor workflowDescriptor, long stepId)
    {
        if (workflowDescriptor != null)
        {
            StepDescriptor stepDescriptor = workflowDescriptor.getStep((int) stepId);
            if (stepDescriptor != null)
            {
                return stepDescriptor;
            }
        }
        
        return null;
    }
    
    /**
     * Get a string representing a json map of the contentIds with true/false if they can/can't be modified
     * @param actionId action to check
     * @param inputPageId page (can be null)
     * @param checkEditionMode check also if we are in edition mode
     * @return {contentId : true/false, ....}
     */
    public static String getModifiableContents(int actionId, String inputPageId, boolean checkEditionMode)
    {
        Map<String, Object> jsonObject = new HashMap<>();
        String pageId = inputPageId;
        if (StringUtils.isEmpty(inputPageId))
        {
            pageId = AmetysXSLTHelper.pageId();
        }
        
        if (!StringUtils.isEmpty(pageId))
        {
            AmetysObject resolveById = _ametysObjectResolver.resolveById(pageId);
            if (resolveById instanceof Page)
            {
                Page page = (Page) resolveById;
                List<Content> contents = _pageHelper.getAllContents(page);
                for (Content content : contents)
                {
                    Map<String, Object> contentInfo = new HashMap<>();
                    contentInfo.put("unmodifiableAttributes", AmetysFrontEditionHelper.getUnmodifiableAttributes(content, List.of(actionId), checkEditionMode));
                    contentInfo.put("rights", AmetysFrontEditionHelper.getRightsForContent(content));
                    
                    jsonObject.put(content.getId(), contentInfo);
                }
            }
        }

        return _jsonUtils.convertObjectToJson(jsonObject);
    }
    
    /**
     * Get path of unmodifiable attributes. Check if content can be modified and the restrictions on attributes
     * @param content the content
     * @param actionIds the workflow action ids to check
     * @param checkEditionMode to check edition mode
     * @return the list with the path of unmodifiable attributes (regarding of restrictions) or null if the content cannot be modified
     */
    public static List<String> getUnmodifiableAttributes(Content content, List<Integer> actionIds, boolean checkEditionMode)
    {
        try
        {
            boolean hasWorkflowRight = hasWorkflowRight(actionIds, content, checkEditionMode);
            if (!hasWorkflowRight)
            {
                // content is not editable
                return null;
            }
            
            Collection<? extends ModelItem> modelItems = _contentTypesHelper.getModelItems(content.getTypes());
            List<AttributeDefinition> attributes = _getAttributeDefinitionsRecursively(modelItems);
            return attributes.stream()
                .filter(mi -> !mi.canWrite(content))
                .map(mi -> mi.getPath())
                .collect(Collectors.toList());
        }
        catch (IllegalArgumentException e)
        {
            _logger.error("Fail to get restricted model items for content " + content.getId(), e);
            return List.of();
        }
    }
    
    /**
     * Get a map of chosen rights for a content
     * @param content the content
     * @return the chosen rights for a content
     */
    public static Map<String, Object> getRightsForContent(Content content)
    {
        Map<String, Object> rights = new HashMap<>();
        rights.put("hasAddFileRight", _rightManager.currentUserHasRight(ExplorerResourcesDAO.RIGHTS_RESOURCE_ADD, content) == RightResult.RIGHT_ALLOW);
        return rights;
    }
    
    private static List<AttributeDefinition> _getAttributeDefinitionsRecursively(Collection<? extends ModelItem> modelItems)
    {
        List<AttributeDefinition> attributeDefs = new ArrayList<>();
        
        for (ModelItem modelItem : modelItems)
        {
            if (modelItem instanceof AttributeDefinition)
            {
                attributeDefs.add((AttributeDefinition) modelItem);
            }
            else if (modelItem instanceof ModelItemGroup)
            {
                attributeDefs.addAll(_getAttributeDefinitionsRecursively(((ModelItemGroup) modelItem).getModelItems()));
            }
        }
        
        return attributeDefs;
    }
    
    /**
     * Get the name of a workflow action
     * @param workflowName workflow name
     * @param actionId action id in the workflow
     * @return name of the workflow action
     */
    public static String getWorkflowName(String workflowName, int actionId)
    {
        String workflowNameKey = _workflowHelper.getActionName(workflowName, actionId);
        String translatedName = org.ametys.core.util.AmetysXSLTHelper.translate(workflowNameKey);
        return translatedName;
    }
    /**
     * Get the status (validated/draft) of all contents in a page
     * @param pageId id of the page to check
     * @return "" if not a page, "draft" if all contents are drafts, "validated" if all contents are validated, and "mixed" if there are both
     */
    public static String getPageStatus(String pageId)
    {
        boolean hasDraft = false;
        boolean hasValidated = false;
        AmetysObject resolveById = _ametysObjectResolver.resolveById(pageId);
        if (resolveById instanceof Page)
        {
            Page page = (Page) resolveById;
            List<Content> contents = _pageHelper.getAllContents(page);
            for (Content content : contents)
            {
                if (isContentLive(content))
                {
                    hasValidated = true;
                }
                else
                {
                    hasDraft = true;
                }
            }
            String result = "none";
            if (hasDraft && !hasValidated)
            {
                result = "draft";
            }
            else if (!hasDraft && hasValidated)
            {
                result = "validated";
            }
            else if (hasDraft && hasValidated)
            {
                result = "mixed";
            }
            return result;
        }
        else
        {
            return "";
        }
    }

    /**
     * Get the workflow step id of a content
     * @param contentId id of the content
     * @return the step Id
     */
    public static long getContentWorkflowId(String contentId)
    {
        AmetysObject resolvedById = _ametysObjectResolver.resolveById(contentId);
        if (resolvedById instanceof Content)
        {
            Content content = (Content) resolvedById;
            return getContentWorkflowId(content);
        }
        else
        {
            return 0;
        }
    }
    /**
     * Get the workflow step id of a content
     * @param content content
     * @return the step Id
     */
    public static long getContentWorkflowId(Content content)
    {
        WorkflowAwareContent wContent = (WorkflowAwareContent) content;
        return wContent.getCurrentStepId();
    }

    /**
     * check if a content is published ot not
     * @param contentId id of the content
     * @return "validated" or "draft", "" if not found
     */
    public static String getContentStatus(String contentId)
    {
        AmetysObject resolvedById = _ametysObjectResolver.resolveById(contentId);
        if (resolvedById instanceof Content)
        {
            Content content = (Content) resolvedById;
            return getContentStatus(content);
        }
        else
        {
            return "";
        }
    }
    /**
     * check if a content is published ot not
     * @param content content
     * @return "validated" or "draft"
     */
    public static String getContentStatus(Content content)
    {
        return isContentLive(content) ? "validated" : "draft";
    }
    /**
     * check if a content is published ot not
     * @param contentId id of the content
     * @return true if validated
     */
    public static boolean isContentLive(String contentId)
    {
        AmetysObject resolvedById = _ametysObjectResolver.resolveById(contentId);
        if (resolvedById instanceof Content)
        {
            Content content = (Content) resolvedById;
            return isContentLive(content);
        }
        else
        {
            return false;
        }
    }
    /**
     * check if a content is published ot not
     * @param content content
     * @return true if validated
     */
    public static boolean isContentLive(Content content)
    {
        return Arrays.asList(((VersionableAmetysObject) content).getLabels()).contains(CmsConstants.LIVE_LABEL);
    }
    /**
     * Get the position of this zoneItem in the zone, or -1 if not found
     * @param zoneItemId zoneitem to search
     * @return zero based position, -1 if not found
     * @throws UnknownAmetysObjectException If zone item is not found
     * @throws AmetysRepositoryException If an error occurred
     */
    public static long getZoneItemPosition(String zoneItemId) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        ZoneItem zoneItem = _ametysObjectResolver.resolveById(zoneItemId);
        return getZoneItemPosition(zoneItem);
    }
    
    /**
     * Get the position of this zoneItem in its parent zone, or -1 if not found
     * @param zoneItem zoneitem
     * @return zero based position, -1 if not found
     */
    public static long getZoneItemPosition(ZoneItem zoneItem)
    {
        AmetysObject parent = zoneItem.getParent();
        if (parent instanceof DefaultTraversableAmetysObject)
        {
            return ((DefaultTraversableAmetysObject) parent).getChildPosition(zoneItem);
        }
        return -1; // virtual page
    }
    
    /**
     * Return the size of a zone
     * @param zoneName zone name
     * @param pageId page Id
     * @return size of the zone (-1 if not found)
     */
    public static long getZoneSize(String zoneName, String pageId)
    {
        Page page = _ametysObjectResolver.resolveById(pageId);
        Zone zone = page.getZone(zoneName);
        return getZoneSize(zone);
    }
    
    /**
     * Return the size of a zone
     * @param zone the zone
     * @return size of the zone or -1 is zone is null
     */
    public static long getZoneSize(Zone zone)
    {
        if (zone != null)
        {
            return zone.getZoneItems().getSize();
        }
        return -1;
    }
    /**
     * Test if a page is modifiable
     * @param pageId id of the page to check
     * @return true if the page is modifiable, false otherwise
     */
    public static boolean isPageModifiable(String pageId)
    {
        try
        {
            AmetysObject resolveById = _ametysObjectResolver.resolveById(pageId);
            return resolveById instanceof ModifiablePage;
        }
        catch (AmetysRepositoryException e)
        {
            // Nothing, this is not a page
        }
        
        return false;
    }
    
    /**
     * Returns the ids of the pages tagged with the specified tag in default workspace
     * @param tag The tag id
     * @return Array of pages ids
     */
    public static NodeList findPagesIdsByTagInDefaultWorkspace(String tag)
    {
        return findPagesIdsByTagInDefaultWorkspace(tag, false);
    }
    
    /**
     * Returns the ids of the pages tagged with the specified tag  in default workspace
     * @param tag The tag id
     * @param checkReadAccess true to return only pages with read access for current user
     * @return Array of pages ids
     */
    public static NodeList findPagesIdsByTagInDefaultWorkspace(String tag, boolean checkReadAccess)
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace
            // The tagged page can be a node page, available of front only when a first subpage will be created
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            return AmetysXSLTHelper.findPagesIdsByTag(tag, checkReadAccess);
        }
        finally
        {
         // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    /**
     * Returns the ids of the pages tagged with the specified tag in default workspace
     * @param sitename The site id
     * @param lang The language code
     * @param tag The tag id
     * @return Array of pages ids
     */
    public static NodeList findPagesIdsByTagInDefaultWorkspace(String sitename, String lang, String tag)
    {
        return findPagesIdsByTagInDefaultWorkspace(sitename, lang, tag, false);
    }
    
    /**
     * Returns the ids of the pages tagged with the specified tag in default workspace
     * @param sitename The site id
     * @param lang The language code
     * @param tag The tag id
     * @param checkReadAccess true to return only pages with read access for current user
     * @return Array of pages ids
     */
    public static NodeList findPagesIdsByTagInDefaultWorkspace(String sitename, String lang, String tag, boolean checkReadAccess)
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve the current workspace.
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace
            // The tagged page can be a node page, available of front only when a first subpage will be created
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            return AmetysXSLTHelper.findPagesIdsByTag(sitename, lang, tag, checkReadAccess);
        }
        finally
        {
         // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
}
