/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.frontedition;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.DevMode;
import org.ametys.core.DevMode.DEVMODE;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nizableSerializer;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.web.CheckNotFrontAction;
import org.ametys.web.WebAuthenticateAction;
import org.ametys.web.repository.page.Page;

/**
 * Dispatch generator for the front edition
 */
public class DispatchGenerator extends org.ametys.core.ui.dispatcher.DispatchGenerator
{
    private static final List<String> __UNPROTECTED_FOEDITION_METHODS = List.of("pageExists");
    
    private AmetysObjectResolver _resolver;
    private CurrentUserProvider _currentUserProvider;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    protected void _setContextInRequestAttributes(Map<String, Object> contextAsMap)
    {
        super._setContextInRequestAttributes(contextAsMap);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        if (contextAsMap.containsKey("pageId"))
        {
            String pageId = (String) contextAsMap.get("pageId");
            if (!StringUtils.isBlank(pageId))
            {
                try
                {
                    Page page = _resolver.resolveById(pageId);
                    request.setAttribute(Page.class.getName(), page);
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().debug("Page with id '" + pageId + "' does not exist anymore. It may have been deleted or unpublished after recent modifications");
                }
                
            }
        }
        
        // Force locale to FO edition locale
        request.setAttribute(I18nizableSerializer.REQUEST_ATTR_LOCALE, contextAsMap.get("locale"));
        
        request.setAttribute(CheckNotFrontAction.CAN_COME_FROM_FRONT_ATTRIBUTE, true);
    }
    
    @Override
    protected String _createUrl(String pluginOrWorkspace, String relativeUrl, Map<String, Object> requestParameters)
    {
        if (!_skipRightProtection(relativeUrl, requestParameters) && !AmetysFrontEditionHelper.hasFrontEditionRight())
        {
            throw new AccessDeniedException("User " + _currentUserProvider.getUser() + " is not allowed to access front edition");
        }
        
        return super._createUrl(pluginOrWorkspace, relativeUrl, requestParameters);
    }
    
    private boolean _skipRightProtection(String relativeUrl, Map<String, Object> requestParameters)
    {
        if ("client-call".equals(relativeUrl))
        {
            String methodName = (String) requestParameters.get("methodName");
            String role = (String) requestParameters.get("role");
            
            if (FrontEditionHelper.ROLE.equals(role) && __UNPROTECTED_FOEDITION_METHODS.contains(methodName))
            {
                return true;
            }
        }
        return false;
    }
    
    @Override
    protected Map<String, Object> transmitAttributes(Map<String, Object> attributes)
    {
        Map<String, Object> transmitAttributes = super.transmitAttributes(attributes);
        
        if (attributes.containsKey(WebAuthenticateAction.REQUEST_ATTRIBUTE_FRONTOFFICE_USERIDENTITY))
        {
            UserIdentity frontUserIdentity = (UserIdentity) attributes.get(WebAuthenticateAction.REQUEST_ATTRIBUTE_FRONTOFFICE_USERIDENTITY);
            transmitAttributes.put(WebAuthenticateAction.REQUEST_ATTRIBUTE_FRONTOFFICE_USERIDENTITY, frontUserIdentity);
        }
        
        if (attributes.containsKey("rendering-context"))
        {
            transmitAttributes.put("rendering-context", attributes.get("rendering-context"));
        }
        
        if (attributes.containsKey(CheckNotFrontAction.CAN_COME_FROM_FRONT_ATTRIBUTE))
        {
            transmitAttributes.put(CheckNotFrontAction.CAN_COME_FROM_FRONT_ATTRIBUTE, attributes.get(CheckNotFrontAction.CAN_COME_FROM_FRONT_ATTRIBUTE));
        }

        if (attributes.containsKey("site"))
        {
            transmitAttributes.put("site", attributes.get("site"));
        }

        return transmitAttributes;
    }
    
    @Override
    protected String _exceptionToStackTraceInformation(Throwable t)
    {
        if (DevMode.getDeveloperMode() == DEVMODE.PRODUCTION)
        {
            return "The exception is hidden for security purposes";
        }
        else
        {
            return super._exceptionToStackTraceInformation(t);
        }
    }
}
