/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.frontedition;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.Content;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneDAO;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.rights.PageRightAssignmentContext;

/**
 * Various helpers for front-edition plugin
 */
public class FrontEditionHelper extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = FrontEditionHelper.class.getName();
    
    /** The Ametys object resolver */
    private AmetysObjectResolver _ametysObjectResolver;
    private ZoneDAO _zoneDAO;
    /** Context */
    private Context _context;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _ametysObjectResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _zoneDAO = (ZoneDAO) manager.lookup(ZoneDAO.ROLE);
    }

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the first available parent for the current page, in the live workspace
     * @param pageId id of the page that need to be checked
     * @return path of the parent (can be this page if available in live)
     */
    @Callable (rights = AmetysFrontEditionHelper.FRONT_EDITION_RIGHT_ID, paramIndex = 0, rightContext = PageRightAssignmentContext.ID)
    public String firstAvailableParentInLivePath(String pageId)
    {
        AmetysObject resolveById = _ametysObjectResolver.resolveById(pageId);
        if (resolveById instanceof Page)
        {
            Page page = (Page) resolveById;
            return firstAvailableParentInLivePath(page);
        }
        else
        {
            return "";
        }
    }
    /**
     * Get the first available parent for the requested page, in the live version
     * @param page page to check
     * @return path of the parent (can be this page if available in live)
     */
    public String firstAvailableParentInLivePath(Page page)
    {
        Request request = ContextHelper.getRequest(_context);
        if (page == null)
        {
            return "";
        }
        AmetysObject available = page;
        Page availablePage = null;
        boolean found = false;
        String forcedWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        try
        {
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, "live");
    
            while (!found)
            {
                try
                {
                    AmetysObject resolveById = _ametysObjectResolver.resolveById(available.getId());
                    if (resolveById != null)
                    {
                        found = true;
                    }
                    else
                    {
                        available = available.getParent();
                    }
                }
                catch (UnknownAmetysObjectException e)
                {
                    available = available.getParent();
                }
            }

            if (available instanceof Page)
            {
                availablePage = (Page) available;
            }
        }
        finally
        {
            //reset workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, forcedWorkspace);
        }
        return availablePage == null ? "" : availablePage.getPathInSitemap();
    }
    
    /**
     * Determines if the page exists
     * @param pageId the page id
     * @param editionMode true if we are in edition mode
     * @return true if the page exists
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public boolean pageExists(String pageId, boolean editionMode)
    {
        Request request = ContextHelper.getRequest(_context);
        String currentWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        try
        {
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, editionMode ? RepositoryConstants.DEFAULT_WORKSPACE : WebConstants.LIVE_WORKSPACE);
            return _ametysObjectResolver.hasAmetysObjectForId(pageId);
        }
        finally
        {
            //reset workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWorkspace);
        }
    }
    
    /**
     * Get the name of a workflow action by it's id
     * @param contentId id of the content to check if action is available
     * @param workflowName name of the workflow checked
     * @param actionIds actions where the name needs to be retreived
     * @return workwflow action name
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<Integer, String> getWorkflowActionName(String contentId, String workflowName, List<Integer> actionIds)
    {
        Map<Integer, String> result = new HashMap<>();
        for (Integer actionId : actionIds)
        {
            boolean hasWorkflowRight = AmetysFrontEditionHelper.hasWorkflowRight(actionId, contentId, false);
            if (hasWorkflowRight)
            {
                String translatedName = AmetysFrontEditionHelper.getWorkflowName(workflowName, actionId);
                result.put(actionId, translatedName);
            }
        }
        return result;
    }
    
    /**
     * Check if contents are modifiable and check all attributes restriction
     * @param actionId the edit action id
     * @param contentIds the id of contents
     * @param checkEditionMode Check if we are in edition mode or not
     * @return A Map with content id as key. The value is null if content is unmodifiable or a map of content informations as an array with path of unmodifiable attributes otherwise
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getModifiableContents(int actionId, List<String> contentIds, boolean checkEditionMode)
    {
        Map<String, Object> result = new HashMap<>();
        for (String contentId : contentIds)
        {
            Content content = _ametysObjectResolver.resolveById(contentId);
            
            Map<String, Object> contentInfo = new HashMap<>();
            contentInfo.put("unmodifiableAttributes", AmetysFrontEditionHelper.getUnmodifiableAttributes(content, List.of(actionId), checkEditionMode));
            contentInfo.put("rights", AmetysFrontEditionHelper.getRightsForContent(content));
            
            result.put(contentId, contentInfo);
        }
        
        return result;
    }
    
    /**
     * Move a zone item of a page before/after another zone item of the same page
     * @param zoneItemId zone item to move
     * @param zoneName name of the zone
     * @param pageId page Id
     * @param offset how many item back/forward to move ? (negative for up, positive to down)
     * @return true if success
     * @throws UnknownAmetysObjectException If an error occurred
     * @throws AmetysRepositoryException If an error occurred
     * @throws RepositoryException If an error occurred
     * @throws AccessDeniedException If user is not authorized
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public boolean moveZoneItemId(String zoneItemId, String zoneName, String pageId, int offset) throws UnknownAmetysObjectException, AmetysRepositoryException, RepositoryException, AccessDeniedException
    {
        // Web_Rights_Page_OrganizeZoneItem
        ZoneItem zoneItem = _ametysObjectResolver.resolveById(zoneItemId);
        AmetysObject parent = zoneItem.getParent();
        if (parent instanceof DefaultTraversableAmetysObject)
        {
            SitemapElement page = zoneItem.getZone().getSitemapElement();
            if (!AmetysFrontEditionHelper.hasFrontEditionRight("Web_Rights_Page_OrganizeZoneItem", page, false))
            {
                throw new AccessDeniedException("User try to move zone item without sufficient right");
            }
            DefaultTraversableAmetysObject traversableParent = (DefaultTraversableAmetysObject) parent;
            long itemPosition = traversableParent.getChildPosition(zoneItem);
            long targetPosition = itemPosition + offset;
            ZoneItem targetZoneItem = traversableParent.getChildAt(targetPosition);
            return _zoneDAO.moveZoneItemTo(zoneItemId, zoneName, offset < 0, targetZoneItem.getId(), pageId);
        }
        return false;
    }
}
