/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.frontedition;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.lock.LockManager;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.cms.CmsConstants;
import org.ametys.cms.lock.LockContentManager;
import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.AvalonLoggerAdapter;
import org.ametys.plugins.core.ui.help.HelpManager;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.lock.LockHelper;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.runtime.model.DefinitionContext;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;

/**
 * Check if the content can be edited, and return the value
 */
public class GetServerValuesAction extends ServiceableAction implements Initializable
{
    /** The logger */
    protected Logger _logger;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    /** User helper */
    protected UserHelper _userHelper;
    /** Lock Content Manager */
    protected LockContentManager _lockContentManager;
    /** The help manager to get url for each property */
    protected HelpManager _helpManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _renderingContextHandler = (RenderingContextHandler) serviceManager.lookup(RenderingContextHandler.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _lockContentManager = (LockContentManager) serviceManager.lookup(LockContentManager.ROLE);
        _helpManager = (HelpManager) serviceManager.lookup(HelpManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _logger = new AvalonLoggerAdapter(getLogger());
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        String contentId = parameters.getParameter("contentId");
        String attributePathsAsString = parameters.getParameter("metadataPaths");
        String workflowIdsAsString = parameters.getParameter("workflowIds", null);

        Map<String, Object> jsonObject = new HashMap<>();
        boolean success = true;

        if (attributePathsAsString == null)
        {
            success = false;
            jsonObject.put("error", "no metadata");
            request.setAttribute(JSonReader.OBJECT_TO_READ, jsonObject);
            return EMPTY_MAP;
        }
        List<String> attributePaths = Arrays.asList(attributePathsAsString.split(";"));

        boolean validateContent = parameters.getParameterAsBoolean("validateContent", false);

        boolean isEditionMode = "true".equals(request.getParameter("_edition"));

        validateContent &= !isEditionMode; //no validation if in edition mode

        Content content = _resolver.resolveById(contentId);
        // lock validation
        UserIdentity locker = isContentLocked(content);
        if (locker != null)
        {
            success = false;
            String userFullName = _userHelper.getUserFullName(locker);
            jsonObject.put("error", "locked");
            Map<String, String> userIdentyJson = new HashMap<>();
            userIdentyJson.put("fullName", userFullName);
            jsonObject.put("locker", userIdentyJson);
        }
        else if (validateContent)
        {
            // draft/live validation
            RenderingContext context = _renderingContextHandler.getRenderingContext();
            if (context == RenderingContext.FRONT && content instanceof VersionableAmetysObject)
            {
                String[] labels = ((VersionableAmetysObject) content).getLabels();
                if (!Arrays.asList(labels).contains(CmsConstants.LIVE_LABEL))
                {
                    success = false;
                    jsonObject.put("error", "draft");
                }
            }
        }

        // workflow validation
        if (success)
        {
            if (workflowIdsAsString == null)
            {
                success = false;
                jsonObject.put("error", "no workflow Ids");
                request.setAttribute(JSonReader.OBJECT_TO_READ, jsonObject);
                return EMPTY_MAP;
            }
            List<String> workflowIdsAsStrings = Arrays.asList(workflowIdsAsString.split(";"));
            List<Integer> workflowIds = new ArrayList<>();
            for (String workflowIdAsString : workflowIdsAsStrings)
            {
                workflowIds.add(Integer.parseInt(workflowIdAsString));
            }
            boolean workflowRightsOk = AmetysFrontEditionHelper.hasWorkflowRight(workflowIds, contentId, false);
            if (!workflowRightsOk)
            {
                success = false;
                jsonObject.put("error", "workflow-rights");
                request.setAttribute(JSonReader.OBJECT_TO_READ, jsonObject);
                return EMPTY_MAP;
            }
        }

        if (success)
        {
            List<String> contentIds = new ArrayList<>(1);
            contentIds.add(contentId);
            _lockContentManager.unlockOrLock(contentIds, "lock");
            List<String> languages = getLanguages(request);
            
            Map<String, Object> attributeJsonObject = new HashMap<>();
            for (String attributePath : attributePaths)
            {
                ModelItem modelItem = getDefinition(content, attributePath);
                if (checkRestriction(content, modelItem))
                {
                    Map<String, Object> contentAttribute2Json = _contentAttribute2Json(content, modelItem, attributePath, languages);
                    attributeJsonObject.put(attributePath, contentAttribute2Json);
                }
            }
            jsonObject.put("data", attributeJsonObject);
        }

        request.setAttribute(JSonReader.OBJECT_TO_READ, jsonObject);
        return EMPTY_MAP;
    }
    
    /**
     * Check if attribute can be edited
     * @param content the content
     * @param modelItem the definition of attribute
     * @return true if the attribute can be edited
     */
    @SuppressWarnings("unchecked")
    protected boolean checkRestriction(Content content, ModelItem modelItem)
    {
        if (modelItem instanceof RestrictedModelItem)
        {
            return ((RestrictedModelItem) modelItem).canWrite(content);
        }
        return true; // no restriction
    }

    /**
     * Check if the content is locked
     * @param content The content
     * @return UserIdentity of the locker, of null if not locked
     */
    protected UserIdentity isContentLocked(Content content)
    {
        if (!(content instanceof JCRAmetysObject))
        {
            return null;
        }

        try
        {
            Node node = ((JCRAmetysObject) content).getNode();
            LockManager lockManager = node.getSession().getWorkspace().getLockManager();

            if (lockManager.isLocked(node.getPath()))
            {
                Node lockHolder = lockManager.getLock(node.getPath()).getNode();

                AmetysObject ao = _resolver.resolve(lockHolder, false);
                if (ao instanceof LockableAmetysObject)
                {
                    LockableAmetysObject lockableAO = (LockableAmetysObject) ao;
                    if (!LockHelper.isLockOwner(lockableAO, _currentUserProvider.getUser()))
                    {
                        return lockableAO.getLockOwner();
                    }
                }
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error(String.format("Repository exception during lock checking for ametys object '%s'", content.getId()), e);
            throw new AmetysRepositoryException(e);
        }
        return null;
    }
    
    /**
     * list all languages requested by the client in the request
     * @param request the request
     * @return an ordered list of all languages requested by the client (or server default locale if none requested by the client)
     */
    protected List<String> getLanguages(Request request)
    {
        Enumeration locales = request.getLocales();
        List<String> languages = new ArrayList<>();
        while (locales.hasMoreElements())
        {
            Locale locale = (Locale) locales.nextElement();
            String lang = locale.getLanguage();
            if (!languages.contains(lang))
            {
                languages.add(lang);
            }
        }
        return languages;
    }
    
    /**
     * Get the definition to the given attribute path
     * @param content the content
     * @param attributePath the path of the attribute
     * @return the model item
     * @throws ProcessingException if the attribute is not defined by the model
     * @throws AmetysRepositoryException if an error occurred
     */
    protected ModelItem getDefinition(Content content, String attributePath) throws ProcessingException, AmetysRepositoryException
    {
        if (!content.hasDefinition(attributePath))
        {
            throw new ProcessingException(String.format("Unknown attribute path '%s' for content type(s) '%s'", attributePath, StringUtils.join(content.getTypes(), ',')));
        }
        
        return content.getDefinition(attributePath);
    }
    
    /**
     * Convert the content attribute at the given path into a JSON object
     * @param content the content
     * @param modelItem the attribute definition
     * @param attributePath the path of the attribute to convert
     * @param languages all languages requested by the client
     * @return the attribute as a JSON object
     * @throws ProcessingException if an error occurs 
     */
    @SuppressWarnings("unchecked")
    protected Map<String, Object> _contentAttribute2Json(Content content, ModelItem modelItem, String attributePath, List<String> languages) throws ProcessingException
    {
        Map<String, Object> jsonObject = modelItem.toJSON(DefinitionContext.newInstance().withObject(content));
        
        String help = _getModelItemHelpLink(modelItem, languages);
        if (StringUtils.isNotBlank(help))
        {
            jsonObject.put("help", help);
        }
        
        if (modelItem instanceof ElementDefinition && (!(modelItem instanceof RestrictedModelItem) || ((RestrictedModelItem<Content>) modelItem).canRead(content)))
        {
            Object value = content.getValue(attributePath);

            if (value != null)
            {
                ElementType type = (ElementType) modelItem.getType();
                DataContext context = RepositoryDataContext.newInstance()
                                                           .withObject(content)
                                                           .withDataPath(attributePath)
                                                           .withModelItem(modelItem);
                
                Object valueAsJSON = type.valueToJSONForEdition(value, context);
                if (valueAsJSON instanceof List)
                {
                    if (!((List) valueAsJSON).isEmpty())
                    {
                        Object[] arrayValue = (Object[]) Array.newInstance(((List) valueAsJSON).get(0).getClass(), ((List) valueAsJSON).size());
                        jsonObject.put("value", ((List) valueAsJSON).toArray(arrayValue));
                    }
                    else
                    {
                        jsonObject.put("value", new Object[0]);
                    }
                }
                else
                {
                    jsonObject.put("value", valueAsJSON);
                }
            }
        }
        
        return jsonObject;
    }
    
    /**
     * Retrieves the Help link for the given model item
     * @param modelItem the model item
     * @param languages all languages requested by the client
     * @return the help link
     */
    protected String _getModelItemHelpLink(ModelItem modelItem, List<String> languages)
    {
        Model model = modelItem.getModel();
        if (model != null)
        {
            String modelId = model.getId();
            String family = model.getFamilyId();
            
            String path = modelItem.getPath();
            //If there is a path, and it does not starts with '/', we add one at the beginning
            if (StringUtils.isNotBlank(path))
            {
                path = StringUtils.prependIfMissing(path, ModelItem.ITEM_PATH_SEPARATOR);
            }
            String featureId = StringUtils.join(modelId, path);
            //Remove the starting '/' if present
            featureId = StringUtils.removeStart(featureId, ModelItem.ITEM_PATH_SEPARATOR);

            try
            {
                return _helpManager.getHelp(family, featureId, languages);
            }
            catch (Exception e)
            {
                _logger.warn("Impossible to get help for the content type '{}' on path '{}'", modelId, path, e);
            }
        }
           
        return StringUtils.EMPTY;
    }
}
