/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
 AmetysFlashInfo = {
    
    isAnonymous: true,
    
    userItems: [],
    
    _allItems:[],
    
    _nextItem: 0,
    
    _currentItemId: null,
    
    contextPath: null,
    
    /**
     * @property {String} __FLASH_INFO_ITEMS The id of flash info items for local storage
     */
    __FLASH_INFO_ITEMS: "flashInfoItems",
    
    localFlashInfos: {
        
        /**
	     * Mark a flash info item as read (local storage only)
	     * @param {String} id the item's id
	     */
	    markAsRead: function(id)
	    {
	        var items = AmetysFlashInfo.localFlashInfos.getReadItems();
	        
            id = AmetysFlashInfo._toStorageFormat(id);
            if ($j.inArray(id, items) == -1)
            {
                items.push(id);
            }
	        
	        localStorage.setItem(AmetysFlashInfo.__FLASH_INFO_ITEMS, JSON.stringify(items));
	    },
	    
	    /**
	     * Get read item stored in local storage
	     * @return {String[]} The read flash infos items
	     */
	    getReadItems: function()
	    {
	        var items = localStorage.getItem(AmetysFlashInfo.__FLASH_INFO_ITEMS) || '[]';
	        return items ? JSON.parse(items) : {}; 
	    },
        
        /**
	     * Empty the read items (local storage only)
	     */
	    removeAll: function()
	    {
	        localStorage.setItem(AmetysFlashInfo.__FLASH_INFO_ITEMS, '[]');
	    },
	    
        /**
         * Return true if item is in local storage
         * @param {String} id the item's id
         * @return {Boolean} true if present in local storage
         */
	    hasItem: function(id)
	    {
	        var items = AmetysFlashInfo.localFlashInfos.getReadItems();
	        return $j.inArray(id, items) != -1;
	    }
    },
    
    /**
     * Function to call to hide flash info banner or popup
     */
    onHide: function() {},
    
    /**
     * Function to call to show flash info banner or popup
     */
    onShow: function() {},
    
    /**
     * Initialize flash info notifications
     * @param {Function} onShow the function to call to show popup or banner
     * @param {Function} onHide the function to call to hide popup or banner
     * @param {String} ctxPath the context path
     */
    initialize: function(selector, onShow, onHide, ctxPath)
    {
        AmetysFlashInfo.onHide = onHide;
        AmetysFlashInfo.onShow = onShow;
        AmetysFlashInfo.contextPath = ctxPath;
        
        var $jbanner = $j(selector);
        var $contents = $jbanner.find("[data-flashinfo-id]");
        $contents.hide();
        
        AmetysFlashInfo._allItems = [];
        AmetysFlashInfo._nextItem = 0;
        
        $contents.each(function(index, content){
            AmetysFlashInfo._allItems.push($j(content).attr('data-flashinfo-id'));
        });
        
        if (AmetysFlashInfo._allItems.length > 0)
        {
            AmetysFlashInfo._goToNext();
        }
        else
        {
            AmetysFlashInfo.onHide();
        }
    },
    
    /**
     * Mark a item as read then go to next unread flash info
     */
    markAsRead: function()
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
            if (!me.isAnonymous)
	        {
	            me.userItems.push(me._toStorageFormat(me._currentItemId));
	            me.save(function(data){
                    me._goToNext();
                });
	        }
	        else
	        {
	            me.localFlashInfos.markAsRead(me._currentItemId);
	            me._goToNext();
	        }
        });
    },
    
    /**
     * Function to call to go to next unread flash info without marking the current as read
     */
    showLater: function()
    {
        this._goToNext();
    },
    
    /**
     * Show the next unread flash info
     */
    _goToNext: function()
    {
        var me = this;
        
        if (me._currentItemId)
        {
            $j('[data-flashinfo-id="' + me._currentItemId + '"]').hide();
            me._currentItemId = null;
        }
        
        if (me._nextItem < me._allItems.length)
        {
            me.getReadItems(function(readItems){
                var nextId = AmetysFlashInfo._allItems[me._nextItem];
	            me._nextItem++;
	            
	            if ($j.inArray(me._toStorageFormat(nextId), readItems) != -1) 
	            {
	                me._goToNext(); // already read, go to next flash
	            }
	            else
	            {
	                // Show flash
                    me._currentItemId = nextId;
	                $j('[data-flashinfo-id="' + nextId + '"]').show();
	                me.onShow();
	            }
            });
        }
        else
        {
            // No more items
            me.onHide();
        }
    },
    
    _toStorageFormat: function(id)
    {
        return id.replace(new RegExp(':', 'g'), '');
    },
    
    
    /**
     * Save the read items if a user is currently connected
     * @param {Function} [callback] the callback function
     */
    save: function(callback)
    {
        if (this.isAnonymous)
        {
            // No user connected
            return;
        }
        
        var itemIds = this.userItems;
        
	    function success(data)
	    {
            if (data && data.success)
            {
                if (typeof (callback) == 'function')
	            {
	                callback(data);
	            }
            }
	    }
	    
	    function error(jqXHR, textStatus, errorThrown)
	    {
            console.error("Unable to save read flash infos in user preferences")
	    }
	
	    $j.ajax({
	        url: AmetysFlashInfo.contextPath + "/_plugins/front-notification/read-items/save",
	        data: {
               "itemIds": itemIds.join(",")
            },
            dataType: "json",
	        type: 'post',
	        success: success,
	        error: error
	    });
    },
    
    /**
     * Get the cart's items
     * @param {String} lang The current language
     * @param {String} [view] The view to use for cart's elements. Defaults to 'cart'.
     * @param {Function} [callback] the callback function to call after getting items.
     */
    getReadItems: function(callback)
    {
        var me = this;
        
        me._internalGetItems(function(itemIds){
            
            if (typeof(callback) == "function")
            {
                callback(itemIds);
            }
        });
    },
    
    _initialized: false,
    _isRunning: false,
    _callbacks: [],
    
    _internalGetItems: function(callback)
    {
        if (this._initialized)
        {
            var itemIds = [];
            if (this.isAnonymous)
            {
                // Get items from local storage
                itemIds = this.localFlashInfos.getReadItems();
            }
            else
            {
                itemIds = this.userItems;
            }
            
            callback(itemIds);
            return;
        }
        
        this._callbacks.push(callback);
        
        var me = this;
        
        function success(data)
        {
            me._isRunning = false;
            
            if (data)
            {
                me.isAnonymous = data.isAnonymous;
                
                var itemIds = [];
                if (me.isAnonymous)
                {
                    // Get items from local storage
                    me.userItems = [];
                    itemIds = me.localFlashInfos.getReadItems();
                }
                else
                {
                    // Merge saved items with localStorage
                    me._mergeUserPrefs(data.items);
                    itemIds = me.userItems;
                }
                
                me._initialized = true;
                
                $j.each(me._callbacks, function(index, cb) {
                    cb(itemIds);
                });
                
                me._callbacks = [];
            }
        }
        
        function handleError(jqXHR, textStatus, errorThrown)
        {
            console.error("Unable to get read flash infos in user preferences")
            me._isRunning = false;
        }
        
        if (!this._isRunning)
        {
            this._isRunning = true;
            
            $j.ajax({
	            url: AmetysFlashInfo.contextPath + "/_plugins/front-notification/read-items/get",
	            dataType: "json",
	            type: 'post',
	            success: success,
	            error: handleError
	        });
        }
    },
    
    /**
     * @private
     * Merge the user preferences with the local storage
     * @param {Object[]} readItems The read items stored in user prefs
     */
    _mergeUserPrefs: function(readItems)
    {
        var localItems = this.localFlashInfos.getReadItems();
        var localSize = localItems.length;

        var me = this;
        $j.each(readItems, function(index, readItemId) {
            
            if (!me.localFlashInfos.hasItem(readItemId))
            {
                // Update local storage
                me.localFlashInfos.markAsRead(readItemId);
            }
            else
            {
                // The item is already in current cart
                localSize--;
            }
        });
        
        this.userItems = this.localFlashInfos.getReadItems();
        
        if (localSize > 0)
	    {
	        // at least there is a local element not saved on server side
	        this.save();
	    }
        
        // Empty local storage
        this.localFlashInfos.removeAll();
    }
 }