/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.io.IOException;
import java.io.InputStream;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.jackrabbit.util.ISO9075;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.Binary;
import org.ametys.cms.tag.Tag;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.linkdirectory.Link.LinkStatus;
import org.ametys.plugins.linkdirectory.Link.LinkType;
import org.ametys.plugins.linkdirectory.Link.LinkVisibility;
import org.ametys.plugins.linkdirectory.dynamic.DynamicInformationProviderExtensionPoint;
import org.ametys.plugins.linkdirectory.link.LinkDAO;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.plugins.linkdirectory.repository.DefaultLinkFactory;
import org.ametys.plugins.linkdirectory.theme.ThemeExpression;
import org.ametys.plugins.linkdirectory.theme.ThemesDAO;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinConfigurationHelper;
import org.ametys.web.skin.SkinsManager;
import org.ametys.web.userpref.FOUserPreferencesConstants;

/**
 * Link directory helper.
 */
public final class DirectoryHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The component role */
    public static final String ROLE = DirectoryHelper.class.getName();

    /** The path to the configuration file */
    private static final String __CONF_FILE_PATH = "conf/link-directory.xml";
    
    private static final String __PLUGIN_NODE_NAME = "linkdirectory";
    
    private static final String __LINKS_NODE_NAME = "ametys:directoryLinks";
    
    private static final String __USER_LINKS_NODE_NAME = "user-favorites";
    
    /** Themes DAO */
    protected ThemesDAO _themesDAO;
    
    /** The Ametys object resolver */
    private AmetysObjectResolver _ametysObjectResolver;
    
    /** The site manager */
    private SiteManager _siteManager;
    
    /** The user preferences manager */
    private UserPreferencesManager _userPreferencesManager;
    
    /** The current user provider */
    private CurrentUserProvider _currentUserProvider;
    
    /** The right manager */
    private RightManager _rightManager;
    
    /** The link DAO */
    private LinkDAO _linkDAO;
    
    /** The context */
    private Context _context;

    private DynamicInformationProviderExtensionPoint _dynamicProviderEP;

    private SkinsManager _skinsManager;

    private SkinConfigurationHelper _skinConfigurationHelper;

    private ServiceManager _smanager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _smanager = manager;
        _ametysObjectResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _userPreferencesManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE + ".FO");
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _dynamicProviderEP = (DynamicInformationProviderExtensionPoint) manager.lookup(DynamicInformationProviderExtensionPoint.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _linkDAO = (LinkDAO) manager.lookup(LinkDAO.ROLE);
        _themesDAO = (ThemesDAO) manager.lookup(ThemesDAO.ROLE);
    }
    
    private SkinsManager _getSkinManager()
    {
        if (_skinsManager == null)
        {
            try
            {
                _skinsManager = (SkinsManager) _smanager.lookup(SkinsManager.ROLE);
            }
            catch (ServiceException e)
            {
                throw new IllegalArgumentException(e);
            }
        }
        return _skinsManager;
    }
    
    private SkinConfigurationHelper _getSkinConfigurationHelper()
    {
        if (_skinConfigurationHelper == null)
        {
            try
            {
                _skinConfigurationHelper = (SkinConfigurationHelper) _smanager.lookup(SkinConfigurationHelper.ROLE);
            }
            catch (ServiceException e)
            {
                throw new IllegalArgumentException(e);
            }
        }
        return _skinConfigurationHelper;
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the root plugin storage object.
     * @param site the site.
     * @return the root plugin storage object.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public ModifiableTraversableAmetysObject getPluginNode(Site site) throws AmetysRepositoryException
    {
        try
        {
            ModifiableTraversableAmetysObject pluginsNode = site.getRootPlugins();
            
            return getOrCreateNode(pluginsNode, __PLUGIN_NODE_NAME, "ametys:unstructured");
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the link directory plugin node for site " + site.getName(), e);
        }
    }
    
    /**
     * Get the links root node.
     * @param site the site
     * @param language the language.
     * @return the links root node.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public ModifiableTraversableAmetysObject getLinksNode(Site site, String language) throws AmetysRepositoryException
    {
        try
        {
            // Get the root plugin node.
            ModifiableTraversableAmetysObject pluginNode = getPluginNode(site);
            
            // Get or create the language node.
            ModifiableTraversableAmetysObject langNode = getOrCreateNode(pluginNode, language, "ametys:unstructured");
            
            // Get or create the definitions container node in the language node and return it.
            return getOrCreateNode(langNode, __LINKS_NODE_NAME, DefaultLinkFactory.LINK_ROOT_NODE_TYPE);
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the link directory root node for site " + site.getName() + " and language " + language, e);
        }
    }
    
    /**
     * Get the links root node for the given user.
     * @param site the site
     * @param language the language.
     * @param user The user identity
     * @return the links root node for the given user.
     * @throws AmetysRepositoryException if a repository error occurs.
     */
    public ModifiableTraversableAmetysObject getLinksForUserNode(Site site, String language, UserIdentity user) throws AmetysRepositoryException
    {
        try
        {
            // Get the root plugin node.
            ModifiableTraversableAmetysObject pluginNode = getPluginNode(site);
            
            // Get or create the user links node.
            ModifiableTraversableAmetysObject userLinksNode = getOrCreateNode(pluginNode, __USER_LINKS_NODE_NAME, "ametys:unstructured");
            // Get or create the population node.
            ModifiableTraversableAmetysObject populationNode = getOrCreateNode(userLinksNode, user.getPopulationId(), "ametys:unstructured");
            // Get or create the login node.
            ModifiableTraversableAmetysObject loginNode = getOrCreateNode(populationNode, user.getLogin(), "ametys:unstructured");
            // Get or create the language node.
            ModifiableTraversableAmetysObject langNode = getOrCreateNode(loginNode, language, "ametys:unstructured");
            
            // Get or create the definitions container node in the language node and return it.
            return getOrCreateNode(langNode, __LINKS_NODE_NAME, DefaultLinkFactory.LINK_ROOT_NODE_TYPE);
        }
        catch (AmetysRepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the link directory root node for user " + user + " and for site " + site.getName() + " and language " + language, e);
        }
    }

    /**
     * Get the plugin node path
     * @param siteName the site name.
     * @return the plugin node path.
     */
    public String getPluginNodePath(String siteName)
    {
        return String.format("//element(%s, ametys:site)/ametys-internal:plugins/%s", siteName, __PLUGIN_NODE_NAME);
    }
    
    /**
     * Get the links root node path
     * @param siteName the site name.
     * @param language the language
     * @return the links root node path.
     */
    public String getLinksNodePath(String siteName, String language)
    {
        return getPluginNodePath(siteName) + "/"  + language + "/" + __LINKS_NODE_NAME;
    }
    
    /**
     * Get the links root node path for the given user 
     * @param siteName the site name.
     * @param language the language
     * @param user The user identity
     * @return the links root node path for the given user.
     */
    public String getLinksForUserNodePath(String siteName, String language, UserIdentity user)
    {
        return getPluginNodePath(siteName) + "/" + __USER_LINKS_NODE_NAME + "/" + ISO9075.encode(user.getPopulationId()) + "/" + ISO9075.encode(user.getLogin()) + "/" + language + "/" + __LINKS_NODE_NAME;
    }
    
    /**
     * Get all the links
     * @param siteName the site name.
     * @param language the language.
     * @return all the links' nodes
     */
    public String getAllLinksQuery(String siteName, String language)
    {
        return getLinksNodePath(siteName, language) + "/element(*, " + DefaultLinkFactory.LINK_NODE_TYPE + ")";
    }
    
    /**
     * Get the link query corresponding to the expression passed as a parameter
     * @param siteName the site name.
     * @param language the language.
     * @param expression the {@link Expression} of the links retrieval query
     * @return the link corresponding to the expression passed as a parameter
     */
    public String getLinksQuery(String siteName, String language, Expression expression)
    {
        return getLinksNodePath(siteName, language) + "/element(*, " + DefaultLinkFactory.LINK_NODE_TYPE + ")[" + expression.build() + "]";
    }
    
    /**
     * Get the user link query corresponding to the expression passed as a parameter
     * @param siteName the site name.
     * @param language the language.
     * @param user the user
     * @param expression the {@link Expression} of the links retrieval query. Can be null to get all user links
     * @return the user link corresponding to the expression passed as a parameter
     */
    public String getUserLinksQuery(String siteName, String language, UserIdentity user, Expression expression)
    {
        String query = getLinksForUserNodePath(siteName, language, user) + "/element(*, " + DefaultLinkFactory.LINK_NODE_TYPE + ")";
        if (expression != null)
        {
            query += "[" + expression.build() + "]";
        }
        return query;
    }
    /**
     * Get the query verifying the existence of an url
     * @param siteName the site name.
     * @param language the language.
     * @param url the url to test. 
     * @return the query verifying the existence of an url
     */
    public String getUrlExistsQuery(String siteName, String language, String url)
    {
        String lowerCaseUrl = StringUtils.replace(url, "'", "''").toLowerCase();
        return getLinksNodePath(siteName, language) + "/element(*, " + DefaultLinkFactory.LINK_NODE_TYPE + ")[fn:lower-case(@ametys-internal:url) = '" + lowerCaseUrl + "' or fn:lower-case(@ametys-internal:internal-url) = '" + lowerCaseUrl + "']";
    }
    
    /**
     * Get the query verifying the existence of an url for the given user
     * @param siteName the site name.
     * @param language the language.
     * @param url the url to test. 
     * @param user The user identity
     * @return the query verifying the existence of an url for the given user
     */
    public String getUrlExistsForUserQuery(String siteName, String language, String url, UserIdentity user)
    {
        String lowerCaseUrl = StringUtils.replace(url, "'", "''").toLowerCase();
        return getLinksForUserNodePath(siteName, language, user) + "/element(*, " + DefaultLinkFactory.LINK_NODE_TYPE + ")[fn:lower-case(@ametys-internal:url) = '" + lowerCaseUrl + "' or fn:lower-case(@ametys-internal:internal-url) = '" + lowerCaseUrl + "']";
    }
    
    /**
     * Normalizes an input string in order to capitalize it, remove accents, and replace whitespaces with underscores
     * @param s the string to normalize
     * @return the normalized string
     */
    public String normalizeString(String s)
    {
        // Strip accents
        String normalizedLabel = Normalizer.normalize(s.toUpperCase(), Normalizer.Form.NFD).replaceAll("[^\\p{ASCII}]", "");
        
        // Upper case
        String upperCaseLabel = normalizedLabel.replaceAll(" +", "_").replaceAll("[^\\w-]", "_").replaceAll("_+", "_").toUpperCase();
        
        return upperCaseLabel;
    }
    
    /**
     * Get links of a given site and language
     * @param siteName the site name
     * @param language the language
     * @return the links
     */
    public AmetysObjectIterable<DefaultLink> getLinks(String siteName, String language)
    {
        Site site = _siteManager.getSite(siteName);
        TraversableAmetysObject linksNode = getLinksNode(site, language);
        return linksNode.getChildren();
    }
    
    /**
     * Get the list of links corresponding to the given theme ids
     * @param themesIds the ids of the configured themes
     * @param siteName the site's name
     * @param language the site's language
     * @return the list of default links corresponding to the given themes
     */
    public List<DefaultLink> getLinks(List<String> themesIds, String siteName, String language)
    {
        Site site = _siteManager.getSite(siteName);
        TraversableAmetysObject linksNode = getLinksNode(site, language);
        AmetysObjectIterable<DefaultLink> links = linksNode.getChildren();
        
        return links.stream()
                .filter(l -> themesIds.isEmpty() || !Collections.disjoint(Arrays.asList(l.getThemes()), themesIds))
                .collect(Collectors.toList());
    }
    
    /**
     * Get links of a given site and language, for the given user
     * @param siteName the site name
     * @param language the language
     * @param user The user identity
     * @return the links for the given user
     */
    public AmetysObjectIterable<DefaultLink> getUserLinks(String siteName, String language, UserIdentity user)
    {
        return getUserLinks(siteName, language, user, null);
    }
    
    /**
     * Get links of a given site and language, for the given user
     * @param siteName the site name
     * @param language the language
     * @param user The user identity
     * @param themeName the theme id to filter user links. If null, return all user links
     * @return the links for the given user
     */
    public AmetysObjectIterable<DefaultLink> getUserLinks(String siteName, String language, UserIdentity user, String themeName)
    {
        ThemeExpression themeExpression = null;
        if (StringUtils.isNotBlank(themeName) && themeExists(themeName, siteName, language))
        {
            themeExpression = new ThemeExpression(themeName);
        }
        
        String linksQuery = getUserLinksQuery(siteName, language, user, themeExpression);
        return _ametysObjectResolver.query(linksQuery);
    }
    
    /**
     * Checks if the links displayed in a link directory service has access restrictions
     * @param siteName the name of the site
     * @param language the language
     * @param themesIds the list of selected theme ids
     * @return true if the links of the service have access restrictions, false otherwise
     */
    public boolean hasRestrictions(String siteName, String language, List<String> themesIds)
    {
        // No themes => we check all the links' access restrictions
        if (themesIds.isEmpty())   
        {
            String allLinksQuery = getAllLinksQuery(siteName, language);
            try (AmetysObjectIterable<AmetysObject> links = _ametysObjectResolver.query(allLinksQuery))
            {
                if (isAccessRestricted(links))
                {
                    return true;
                }
            }
            
            
        }
        // The service has themes specified => we solely check the corresponding links' access restrictions
        else
        {
            for (String themeId : themesIds)
            {
                String xPathQuery = getLinksQuery(siteName, language, new ThemeExpression(themeId));
                try (AmetysObjectIterable<AmetysObject> links = _ametysObjectResolver.query(xPathQuery))
                {
                    if (isAccessRestricted(links))
                    {
                        return true;
                    }
                }
            }
        }
        
        // All the tested links have no restricted access
        return false;
    }
    
    /**
     * Checks if the links displayed in a link directory service has internal link
     * @param siteName the name of the site
     * @param language the language
     * @param themesIds the list of selected theme ids
     * @return true if the links of the service has internal link, false otherwise
     */
    public boolean hasInternalUrl(String siteName, String language, List<String> themesIds)
    {
        Site site = _siteManager.getSite(siteName);
        String allowedIdParameter = site.getValue("allowed-ip");
        if (StringUtils.isBlank(allowedIdParameter))
        {
            return false;
        }
        
        List<DefaultLink> links = getLinks(themesIds, siteName, language);
        for (DefaultLink link : links)
        {
            if (StringUtils.isNotBlank(link.getInternalUrl()))
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Check if the links' access is restricted or not
     * @param links the links to be tested
     * @return true if the link has a restricted access, false otherwise
     */
    public boolean isAccessRestricted(AmetysObjectIterable<AmetysObject> links)
    {
        Iterator<AmetysObject> it = links.iterator();
        
        while (it.hasNext())
        {
            DefaultLink link = (DefaultLink) it.next();
            
            // If any of the links has a limited access, the service declares itself non-cacheable
            if (!_rightManager.hasAnonymousReadAccess(link))
            {
                return true;
            }
        }
        
        return false;
    }
    
    private ModifiableTraversableAmetysObject getOrCreateNode(ModifiableTraversableAmetysObject parentNode, String nodeName, String nodeType) throws AmetysRepositoryException
    {
        ModifiableTraversableAmetysObject node;
        if (parentNode.hasChild(nodeName))
        {
            node = parentNode.getChild(nodeName);
        }
        else
        {
            node = parentNode.createChild(nodeName, nodeType);
            parentNode.saveChanges();
        }
        return node;
    }
    
    /**
     * Get the configuration of links brought by skin
     * @param skinName the skin name
     * @return the skin configuration
     * @throws IOException if an error occured
     * @throws ConfigurationException if an error occured
     * @throws SAXException if an error occured
     */
    public Configuration getSkinLinksConfiguration(String skinName) throws IOException, ConfigurationException, SAXException
    {
        Skin skin = _getSkinManager().getSkin(skinName);
        try (InputStream xslIs = getClass().getResourceAsStream("link-directory-merge.xsl"))
        {
            return _getSkinConfigurationHelper().getInheritanceMergedConfiguration(skin, __CONF_FILE_PATH, xslIs);
        }
    }
    
    /**
     * Sax the directory links
     * @param siteName the site name
     * @param contentHandler the content handler
     * @param links the list of links to sax (can be null)
     * @param restrictedThemes If not empty, only link's themes among this restricted list will be saxed
     * @param userLinks the user links to sax (can be null)
     * @param storageContext the storage context, null if there is no connected user
     * @param isConfigurable true if links are configurable
     * @param contextVars the context variables
     * @param user the user
     * @throws SAXException If an error occurs while generating the SAX events
     * @throws UserPreferencesException if an exception occurs while getting the user preferences
     */
    public void saxLinks(String siteName, ContentHandler contentHandler, List<DefaultLink> links, List<DefaultLink> userLinks, List<String> restrictedThemes, boolean isConfigurable, Map<String, String> contextVars, String storageContext, UserIdentity user) throws SAXException, UserPreferencesException
    {
        // left : true if user link
        // right : the link itself
        List<Pair<Boolean, DefaultLink>> allLinks = new ArrayList<>();
        
        if (links != null)
        {
            for (DefaultLink link : links)
            {
                allLinks.add(new ImmutablePair<>(false, link));
            }
        }
        
        if (userLinks != null)
        {
            for (DefaultLink link : userLinks)
            {
                allLinks.add(new ImmutablePair<>(true, link));
            }
        }
        
        
        String[] orderedLinksPrefLinksIdsArray = null; 
        String[] hiddenLinksPrefLinksIdsArray = null; 
        if (user != null && isConfigurable)
        {
            // TODO it would be nice to change the name of this user pref but the storage is still the same, so for the moment we avoid the SQL migration
            // Cf issue LINKS-141
            // Change in org.ametys.plugins.linkdirectory.LinkDirectorySetUserPreferencesAction#act too
            
            Map<String, String> unTypedUserPrefs = _userPreferencesManager.getUnTypedUserPrefs(user, storageContext, contextVars);
            
            String orderedLinksPrefValues = unTypedUserPrefs.get("checked-links");
            orderedLinksPrefLinksIdsArray = StringUtils.split(orderedLinksPrefValues, ",");
            
            String hiddenLinksPrefValues =  unTypedUserPrefs.get("hidden-links");
            hiddenLinksPrefLinksIdsArray = StringUtils.split(hiddenLinksPrefValues, ",");
        }
        
        Site site = _siteManager.getSite(siteName);
        
        boolean hasIPRestriction = hasIPRestriction(site);
        boolean isIPAuthorized = isInternalIP(site);
        
        // Sort the list according to the orderedLinksPrefLinksIdsArray
        if (ArrayUtils.isNotEmpty(orderedLinksPrefLinksIdsArray))
        {
            DefaultLinkSorter defaultLinkSorter = new DefaultLinkSorter(allLinks, orderedLinksPrefLinksIdsArray);
            allLinks.sort(defaultLinkSorter);
        }
        
        for (Pair<Boolean, DefaultLink> linkPair : allLinks)
        {
            DefaultLink link = linkPair.getRight();
            boolean userLink = linkPair.getLeft();
            
            LinkVisibility defaultVisibility = link.getDefaultVisibility();
            
            // check the access granted if it is not a user link
            if (userLink || _isCurrentUserGrantedAccess(link))
            {
                boolean selected = isConfigurable && ArrayUtils.contains(orderedLinksPrefLinksIdsArray, link.getId()); // deprecated, only used for old views, isHidden should be used now
                boolean isHidden = isConfigurable && (ArrayUtils.contains(hiddenLinksPrefLinksIdsArray, link.getId()) || LinkVisibility.HIDDEN.equals(defaultVisibility) && !selected); 
                saxLink(siteName, contentHandler, link, restrictedThemes, selected, hasIPRestriction, isIPAuthorized, userLink, isHidden);
            }
        }
    }
    
    /**
     * SAX a directory link.
     * @param siteName the site name
     * @param contentHandler the content handler
     * @param link the link to sax.
     * @param restrictedThemes If not empty, only link's themes among this restricted list of themes will be saxed
     * @param selected true if a front end user has checked this link as a user preference, false otherwise (deprecated, only used for old views, isHidden should be used now)
     * @param hasIPRestriction true if we have IP restriction
     * @param isIPAuthorized true if the IP is authorized
     * @param userLink true if it is a user link
     * @param isHidden true if the link is hidden
     * @throws SAXException If an error occurs while generating the SAX events
     */
    public void saxLink (String siteName, ContentHandler contentHandler, DefaultLink link, List<String> restrictedThemes, boolean selected, boolean hasIPRestriction, boolean isIPAuthorized, boolean userLink, boolean isHidden) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", link.getId());
        attrs.addCDATAAttribute("lang", link.getLanguage());
        
        LinkType urlType = link.getUrlType();
        
        _addURLAttribute(link, hasIPRestriction, isIPAuthorized, attrs);
        
        attrs.addCDATAAttribute("urlType", StringUtils.defaultString(urlType.toString()));
        
        if (link.getStatus() != LinkStatus.BROKEN)
        {
            String dynInfoProviderId = StringUtils.defaultString(link.getDynamicInformationProvider());
            // Check if provider exists
            if (StringUtils.isNotEmpty(dynInfoProviderId) && _dynamicProviderEP.hasExtension(dynInfoProviderId))
            {
                attrs.addCDATAAttribute("dynamicInformationProvider", dynInfoProviderId);
            }
        }
        attrs.addCDATAAttribute("title", StringUtils.defaultString(link.getTitle()));
        attrs.addCDATAAttribute("content", StringUtils.defaultString(link.getContent()));
        
        if (urlType == LinkType.PAGE)
        {
            String pageId = link.getUrl();
            try
            {
                Page page = _ametysObjectResolver.resolveById(pageId);
                attrs.addCDATAAttribute("pageTitle", page.getTitle());
            }
            catch (UnknownAmetysObjectException e)
            {
                attrs.addCDATAAttribute("unknownPage", "true");
            }
        } 
        
        attrs.addCDATAAttribute("alternative", StringUtils.defaultString(link.getAlternative()));
        attrs.addCDATAAttribute("pictureAlternative", StringUtils.defaultString(link.getPictureAlternative()));
        
        attrs.addCDATAAttribute("user-selected", selected ? "true" : "false");
        
        attrs.addCDATAAttribute("color", _linkDAO.getLinkColor(link));
        
        String pictureType = link.getPictureType();
        attrs.addCDATAAttribute("pictureType", pictureType);
        if (pictureType.equals("resource"))
        {
            String resourceId = link.getResourcePictureId();
            try
            {
                Resource resource = _ametysObjectResolver.resolveById(resourceId);
                attrs.addCDATAAttribute("pictureId", resourceId);
                attrs.addCDATAAttribute("pictureName", resource.getName());
                attrs.addCDATAAttribute("pictureSize", Long.toString(resource.getLength()));
                attrs.addCDATAAttribute("imageType", "explorer");
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().error("The resource of id'{}' does not exist anymore. The picture for link of id '{}' will be ignored.", resourceId, link.getId(), e);
            }
            
        }
        else if (pictureType.equals("external"))
        {
            Binary picMeta = link.getExternalPicture();
            attrs.addCDATAAttribute("picturePath", DefaultLink.PROPERTY_PICTURE);
            attrs.addCDATAAttribute("pictureName", picMeta.getFilename());
            attrs.addCDATAAttribute("pictureSize", Long.toString(picMeta.getLength()));
            attrs.addCDATAAttribute("imageType", "link-data");
        }
        else if (pictureType.equals("glyph"))
        {
            attrs.addCDATAAttribute("pictureGlyph", link.getPictureGlyph());
        }
        
        attrs.addCDATAAttribute("limitedAccess", String.valueOf(!_rightManager.hasAnonymousReadAccess(link))); 
        
        attrs.addCDATAAttribute("userLink", String.valueOf(userLink));
        attrs.addCDATAAttribute("isHidden", String.valueOf(isHidden));
        
        LinkStatus status = link.getStatus();
        if (status != null)
        {
            attrs.addCDATAAttribute("status", status.name());
        }
        
        if (StringUtils.isNotBlank(link.getPage()))
        {
            attrs.addCDATAAttribute("page", link.getPage());
        }
        
        XMLUtils.startElement(contentHandler, "link", attrs);
        
        // Themes
        _saxThemes(contentHandler, link, restrictedThemes);
        
        XMLUtils.endElement(contentHandler, "link");
    }
    
    /**
     * Add the URL attribute to sax
     * @param link the link
     * @param hasIPRestriction true if we have IP restriction
     * @param isIPAuthorized true if the IP is authorized
     * @param attrs the attribute
     */
    private void _addURLAttribute(DefaultLink link, boolean hasIPRestriction, boolean isIPAuthorized, AttributesImpl attrs)
    {
        String internalUrl = link.getInternalUrl();
        String externalUrl = link.getUrl();
        
        // If we have no internal URL or no IP restriction, just sax external URL
        if (StringUtils.isBlank(internalUrl) || !hasIPRestriction)
        {
            attrs.addCDATAAttribute("url", StringUtils.defaultString(externalUrl));
        }
        else
        {
            // If the IP is authorized, sax internal URL
            if (isIPAuthorized)
            {
                attrs.addCDATAAttribute("url", StringUtils.defaultString(internalUrl));
            }
            // else if we have external URL, we sax it
            else if (StringUtils.isNotBlank(externalUrl))
            {
                attrs.addCDATAAttribute("url", StringUtils.defaultString(externalUrl));
            }
            // else link is disabled it because the IP is not authorized
            else
            {
                attrs.addCDATAAttribute("disabled", "true");
            }
        }
    }
    
    /**
     * Get the actual ids of the themes configured properly, their names if they were not 
     * @param configuredThemesNames the normalized ids of the configured themes
     * @param siteName the site's name
     * @param language the site's language
     * @return the actual ids of the configured themes
     */
    public Map<String, List<String>> getThemesMap(List<String> configuredThemesNames, String siteName, String language)
    {
        Map<String, List<String>> themesMap = new HashMap<> ();
        List<String> correctThemesList = new ArrayList<> ();
        List<String> wrongThemesList = new ArrayList<> ();
        
        for (int i = 0; i < configuredThemesNames.size(); i++)
        {
            String configuredThemeName = configuredThemesNames.get(i);

            Map<String, Object> contextualParameters = new HashMap<>();
            contextualParameters.put("language", language);
            contextualParameters.put("siteName", siteName);
            Tag theme = _themesDAO.getTag(configuredThemeName, contextualParameters);
            
            if (theme == null)
            {
                getLogger().warn("The theme '{}' was not found. It will be ignored.", configuredThemeName);
                wrongThemesList.add(configuredThemeName);
            }
            else
            {
                correctThemesList.add(configuredThemeName);
            }
        }
        
        themesMap.put("themes", correctThemesList);
        themesMap.put("unknown-themes", wrongThemesList);
        return themesMap;
    }
    
    /**
     * Verify the existence of a theme
     * @param themeName the id of the theme to verify
     * @param siteName the site's name
     * @param language the site's language
     * @return true if the theme exists, false otherwise
     */
    public boolean themeExists(String themeName, String siteName, String language)
    {
        if (StringUtils.isBlank(themeName))
        {
            return false;
        }
        Map<String, Object> contextualParameters = new HashMap<>();
        contextualParameters.put("language", language);
        contextualParameters.put("siteName", siteName);
        List<String> checkTags = _themesDAO.checkTags(List.of(themeName), false, Collections.EMPTY_MAP, contextualParameters);
        return !checkTags.isEmpty();
    }
    
    /**
     * Get theme's title from its name
     * @param themeName the theme name
     * @param siteName the site's name
     * @param language the site's language
     * @return the title of the theme. Null if the theme doesn't exist
     */
    public I18nizableText getThemeTitle(String themeName, String siteName, String language)
    {
        Map<String, Object> contextualParameters = new HashMap<>();
        contextualParameters.put("language", language);
        contextualParameters.put("siteName", siteName);
        if (themeExists(themeName, siteName, language))
        {
            Tag tag = _themesDAO.getTag(themeName, contextualParameters);
            return tag.getTitle();
        }
        else
        {
            getLogger().warn("Can't find theme with name {} for site {} and language {}", themeName, siteName, language);
        }
            
        return null;
    }

    /**
     * Get the site's name
     * @param request the request
     * @return the site's name
     */
    public String getSiteName(Request request)
    {
        return WebHelper.getSiteName(request, (Page) request.getAttribute(Page.class.getName()));
    }

    /**
     * Get the site's language
     * @param request the request
     * @return the site's language
     */
    public String getLanguage(Request request)
    {
        Page page = (Page) request.getAttribute(Page.class.getName());
        if (page != null)
        {
            return page.getSitemapName();
        }
        
        String language = (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME);
        if (StringUtils.isEmpty(language))
        {
            language = request.getParameter("language");
        }
        
        return language;
    }
    
    /**
     * Retrieve the context variables from the front
     * @param request the request
     * @return the map of context variables
     */
    public Map<String, String> getContextVars(Request request)
    {
        Map<String, String> contextVars = new HashMap<> ();
        
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, getSiteName(request));
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_LANGUAGE, getLanguage(request));
    
        return contextVars;
    }
    
    /**
     * Get the appropriate storage context from request
     * @param request the request
     * @param zoneItemId the id of the zone item if we deal with a service, null for an input data
     * @return the storage context in which the user preferences will be kept
     */
    public String getStorageContext(Request request, String zoneItemId)
    {
        String siteName = getSiteName(request);
        String language = getLanguage(request);
        
        return StringUtils.isEmpty(zoneItemId) ? siteName + "/" + language : siteName + "/" + language + "/" + zoneItemId;
    }
    
    /**
     * Get the appropriate storage context 
     * @param siteName the name of the site
     * @param language the language
     * @param zoneItemId the id of the zone item if we deal with a service, null for an input data
     * @return the storage context in which the user preferences will be kept
     */
    public String getStorageContext(String siteName, String language, String zoneItemId)
    {
        return StringUtils.isEmpty(zoneItemId) ? siteName + "/" + language : siteName + "/" + language + "/" + zoneItemId;
    }
    
    /**
     * Sax the themes
     * @param contentHandler the content handler 
     * @param link the link 
     * @param restrictedThemes If not empty, only link's themes among this restricted list will be saxed
     * @throws SAXException If an error occurs while generating the SAX events
     */
    private void _saxThemes (ContentHandler contentHandler, DefaultLink link, List<String> restrictedThemes) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "themes");
        
        Map<String, Object> contextualParameters = new HashMap<>();
        contextualParameters.put("language", link.getLanguage());
        contextualParameters.put("siteName", link.getSiteName());
        
        for (String themeId : link.getThemes())
        {
            try
            {
                Tag tag = _themesDAO.getTag(themeId, contextualParameters);
                if (tag != null)
                {
                    if (restrictedThemes.isEmpty() || restrictedThemes.contains(themeId))
                    {
                        AttributesImpl attrs = new AttributesImpl();
                        attrs.addCDATAAttribute("id", themeId);
                        attrs.addCDATAAttribute("name", tag.getName());
                        
                        XMLUtils.startElement(contentHandler, "theme", attrs);
                        tag.getTitle().toSAX(contentHandler, "label");
                        XMLUtils.endElement(contentHandler, "theme");
                    }
                }
                else
                {
                    getLogger().error("Theme '{}' in link '{}' can not be found.", themeId, link.getId());
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // Theme does not exist anymore
            }
        }
            
        
        XMLUtils.endElement(contentHandler, "themes");
    }
    
    /**
     * Determines if the current user is allowed to see the link or not
     * @param link the link 
     * @return true if the current user is allowed to see the link, false otherwise
     */
    private boolean _isCurrentUserGrantedAccess(DefaultLink link)
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        // There is no access restriction
        return _rightManager.hasReadAccess(user, link);
    }
    
    /**
     * Determines if the site has IP restriction for internal links
     * @param site the site
     * @return true if the site has IP restriction
     */
    public boolean hasIPRestriction(Site site)
    {
        return site.getValue("allowed-ip") != null;
    }
    
    /**
     * Determines if the user IP matches the configured internal IP range
     * @param site the site
     * @return true if the user IP is an authorized IP for internal links or if no IP restriction is configured
     */
    public boolean isInternalIP(Site site)
    {
        String ipRegexp = site.getValue("allowed-ip");
        if (StringUtils.isNotBlank(ipRegexp))
        {
            Pattern ipRestriction = Pattern.compile(ipRegexp);
            
            Request request = ContextHelper.getRequest(_context);
            
            // The real client IP may have been put in the non-standard "X-Forwarded-For" request header, in case of reverse proxy
            String xff = request.getHeader("X-Forwarded-For");
            String ip = null;
            
            if (xff != null)
            {
                ip = xff.split(",")[0];
            }
            else
            {
                ip = request.getRemoteAddr();
            }
            
            boolean internalIP = ipRestriction.matcher(ip).matches();
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Ip '{}' is considered {} with pattern {}", ip, internalIP ? "internal" : "external", ipRestriction.pattern());
            }
            
            return internalIP;
        }
        
        // There is no IP restriction, considered user IP is authorized
        return true;
    }
    
    /**
     * Helper class to sort links (DefaultLinkSorter implementation)
     * If both links are in the ordered links list, this order is used
     * If one of them is in it and not the other, the one in it will be before the other
     * If none of them is in the list, the initial order will be used
     */
    private class DefaultLinkSorter implements Comparator<Pair<Boolean, DefaultLink>>
    {
        private String[] _orderedLinksPrefLinksIdsArray;
        private List<String> _initialList;
        /**
         * constructor for the helper
         * @param initialList initial list to keep track of the original order if no order is found
         * @param orderedLinksPrefLinksIdsArray ordered list of link ids
         */
        public DefaultLinkSorter(List<Pair<Boolean, DefaultLink>> initialList, String[] orderedLinksPrefLinksIdsArray)
        {
            _orderedLinksPrefLinksIdsArray = orderedLinksPrefLinksIdsArray;
            _initialList = initialList.stream()
                    .map(Pair::getRight)
                    .map(DefaultLink::getId)
                    .collect(Collectors.toList());
        }
        public int compare(Pair<Boolean, DefaultLink> pair1, Pair<Boolean, DefaultLink> pair2)
        {
            DefaultLink link1 = pair1.getRight();
            DefaultLink link2 = pair2.getRight();
            if (ArrayUtils.isNotEmpty(_orderedLinksPrefLinksIdsArray))
            {
                int nbOrderedLinks = _orderedLinksPrefLinksIdsArray.length;
                int pos1 = ArrayUtils.indexOf(_orderedLinksPrefLinksIdsArray, link1.getId());
                if (pos1 == ArrayUtils.INDEX_NOT_FOUND)
                {
                    pos1 = nbOrderedLinks + _initialList.indexOf(link1.getId()); // if not found, keep orginal order after user's order
                }
                
                int pos2 = ArrayUtils.indexOf(_orderedLinksPrefLinksIdsArray, link2.getId());
                if (pos2 == ArrayUtils.INDEX_NOT_FOUND)
                {
                    pos2 = nbOrderedLinks + _initialList.indexOf(link1.getId()); // if not found, keep orginal order after user's order
                }
                
                return pos1 - pos2;
            }
            else
            {
                return 0; // No sorting if no sort array
            }
        }
    }
}
