/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.inputdata.InputData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * Input data for the link directory user preferences in thumbnails mode 
 */
public class LinkDirectoryInputData extends AbstractLogEnabled implements Contextualizable, InputData, Initializable, Serviceable
{
    /** The path to the configuration file */
    private static final String __CONF_FILE_PATH = "skin://conf/link-directory.xml";
    
    /** The wildcard */
    private static final String __WILDCARD = "*";
    
    private static final String __THEMES_CACHE = LinkDirectoryInputData.class.getName() + "$skinInputDataThemesCache";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The Avalon context */
    private Context _context;
    
    /** Excalibur source resolver */
    private SourceResolver _sourceResolver;
    
    private DirectoryHelper _directoryHelper;
    
    private AbstractCacheManager _cacheManager;
    
    private Map<String, String> _configurationError;
    
    /** The last time the file was loaded */
    private Map<String, Long> _lastConfUpdate;

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context; 
    }
    
    @Override
    public void initialize() throws Exception
    {
        _lastConfUpdate = new HashMap<>();
        _configurationError = new HashMap<>();
        _cacheManager.createMemoryCache(__THEMES_CACHE,
                new I18nizableText("plugin.link-directory", "PLUGINS_LINK_DIRECTORY_CACHE_THEMES_LABEL"),
                new I18nizableText("plugin.link-directory", "PLUGINS_LINK_DIRECTORY_CACHE_THEMES_DESCRIPTION"),
                true,
                null);
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public boolean isCacheable(Site site, Page currentPage)
    {
        Request request = ContextHelper.getRequest(_context);
        
        String template = _getTemplate(request);
        if (template == null)
        {
            return true;
        }
        
        try
        {
            String skinId = site.getSkinId();
            _updateConfigurationValues(skinId);
            if (CollectionUtils.isEmpty(getThemesCache().get(skinId)) || _configurationError.containsKey(skinId))
            {
                // No configuration file or there are errors
                return true;
            }
            
            String language = _directoryHelper.getLanguage(request);
            
            List<ThemeInputData> themeInputDatas = _getThemesForSkinAndTemplate(skinId, template);
            
            if (themeInputDatas.isEmpty())
            {
                // The current template is not configured for a link directory input data
                return true;
            }
            
            for (ThemeInputData themeInputData : themeInputDatas)
            {
                if (themeInputData.isConfigurable() || themeInputData.displayUserLinks())
                {
                    // The applications are configurable
                    return false;
                }
            }
            
            // Find the configured theme ids for this template
            List<String> configuredThemesNames = themeInputDatas.stream()
                .map(inputData -> _getConfiguredThemes(inputData, language))
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
            String siteName = _directoryHelper.getSiteName(request);
            
            return !_directoryHelper.hasRestrictions(siteName, language, configuredThemesNames) && !_directoryHelper.hasInternalUrl(siteName, language, configuredThemesNames);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred while retrieving information from the skin configuration", e);
            // Configuration file is not readable => toSAX method will not generate any xml
            return true;
        }
    }

    @Override
    public void toSAX(ContentHandler contentHandler) throws ProcessingException
    {
        Request request = ContextHelper.getRequest(_context);
        
        // Get the current user's login if he is in the front office
        UserIdentity user = _currentUserProvider.getUser();

        String template = _getTemplate(request);
        if (template == null)
        {
            getLogger().info("There is no current template");
            return; 
        }
        
        String skinId = _getSkin(request);
        if (skinId == null)
        {
            getLogger().info("There is no current skin");
            return; 
        }
        
        try
        {
            _updateConfigurationValues(skinId);
            if (CollectionUtils.isEmpty(getThemesCache().get(skinId)))
            {
                return;
            }

            contentHandler.startDocument();
            
            // Is there an error in the configuration file ?
            if (_configurationError.containsKey(skinId))
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("error", _configurationError.get(skinId));
                XMLUtils.createElement(contentHandler, "linkDirectory", attrs);
            }
            else
            {
                String language = _directoryHelper.getLanguage(request);
                String siteName = _directoryHelper.getSiteName(request);
                List<ThemeInputData> themeInputDatas = _getThemesForSkinAndTemplate(skinId, template);
                for (ThemeInputData themeInputData : themeInputDatas)
                {
                    AttributesImpl attrs = new AttributesImpl();
                    attrs.addCDATAAttribute("applicable", Boolean.TRUE.toString());
                    attrs.addCDATAAttribute("configurable", String.valueOf(themeInputData.isConfigurable()));
                    attrs.addCDATAAttribute("displayUserLinks", String.valueOf(themeInputData.displayUserLinks()));
                    attrs.addCDATAAttribute("id", themeInputData.getId());
                    
                    XMLUtils.startElement(contentHandler, "linkDirectory", attrs);
                    
                    List<String> configuredThemesNames = _getConfiguredThemes(themeInputData, language);
                    if (configuredThemesNames != null)
                    {
                        Map<String, List<String>> themesMap = _directoryHelper.getThemesMap(configuredThemesNames, siteName, language);
                        List<String> correctThemesIds = themesMap.get("themes");
                        List<String> unknownThemesNames = themesMap.get("unknown-themes");

                        _saxThemes(contentHandler, correctThemesIds, unknownThemesNames);
                        _saxLinks(contentHandler, user, request, correctThemesIds, themeInputData.displayUserLinks(), themeInputData.isConfigurable(), themeInputData.getId());
                    }
                    
                    XMLUtils.endElement(contentHandler, "linkDirectory");
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("An exception occurred during the processing of the link directory's input data" , e);
        }
    }
    
    private void _saxThemes(ContentHandler contentHandler, List<String> themeIds, List<String> unknownThemesNames) throws SAXException
    {
        if (!themeIds.isEmpty())
        {
            XMLUtils.startElement(contentHandler, "themes");
            for (String themeId : themeIds)
            {
                XMLUtils.createElement(contentHandler, "theme", themeId);
            }
            XMLUtils.endElement(contentHandler, "themes");
        }
        
        if (!unknownThemesNames.isEmpty())
        {
            AttributesImpl attr = new AttributesImpl();
            attr.addCDATAAttribute("count", Integer.toString(unknownThemesNames.size()));
            XMLUtils.createElement(contentHandler, "unknown-themes", attr, StringUtils.join(unknownThemesNames, ", "));
        }
    }

    private void _saxLinks(ContentHandler contentHandler, UserIdentity user, Request request, List<String> themeIds, boolean displayUserLinks, boolean configurable, String specificContext) throws ProcessingException
    {
        String language = _directoryHelper.getLanguage(request);
        String siteName = _directoryHelper.getSiteName(request);
        try
        {
            // SAX common links
            List<DefaultLink> links = _directoryHelper.getLinks(themeIds, siteName, language);
            
            List<DefaultLink> userLinks = null;
            if (user != null && displayUserLinks)
            {
                userLinks = _directoryHelper.getUserLinks(siteName, language, user).stream().collect(Collectors.toList());
            }
            
            
            // SAX the user own links
            XMLUtils.startElement(contentHandler, "links");
            
            try
            {
                String storageContext = siteName + "/" + language;
                if (StringUtils.isNotEmpty(specificContext))
                {
                    storageContext += "/" + specificContext;
                }
                _directoryHelper.saxLinks(siteName, contentHandler, links, userLinks, themeIds, configurable, _directoryHelper.getContextVars(request), storageContext, user);
            }
            catch (Exception e)
            {
                getLogger().error("An exception occurred while saxing the links", e);
            }
            
            XMLUtils.endElement(contentHandler, "links");
        }
        catch (Exception e)
        {
            throw new ProcessingException("An error occurred while retrieving or saxing the links", e);
        }
    }

    /**
     * Retrieve the configured themes names defined in the skin file link-directory.xml for the current input data and the current language
     * @param themeInputData Can be an empty {@link String}
     * @param lang language to filter by. Themes with lang=null will always be returned.
     * @return the list of configured themes ids, can be empty, cannot be null
     */
    private List<String> _getConfiguredThemes(ThemeInputData themeInputData, String lang)
    {
        return themeInputData.getThemes()
            .stream()
            .filter(t -> t.get("lang") == null || t.get("lang").equals(lang))
            .map(t -> t.get("id"))
            .collect(Collectors.toList());
    }
    
    private List<ThemeInputData> _getThemesForSkinAndTemplate(String skinId, String template) 
    {
        return getThemesCache().get(skinId, k -> new ArrayList<>())
            .stream()
            .filter(t -> _filterByTemplate(t, template))
            .collect(Collectors.toList());
    }
    
    private boolean _filterByTemplate(ThemeInputData theme, String template)
    {
        List<String> templates = theme.getTemplates();
        return templates.contains(template) || templates.contains(__WILDCARD); 
    }
    
    /**
     * Update the configuration values : read them if the map is empty, update them if the file was changed or simply return them
     * @param skinId The skin
     * @throws Exception if an exception occurs
     */
    private void _updateConfigurationValues(String skinId) throws Exception
    {
        Source source = null;
        try
        {
            source = _sourceResolver.resolveURI(__CONF_FILE_PATH);
            if (source.exists())
            {
                _cacheConfigurationValues(source, skinId, !getThemesCache().hasKey(skinId));
            }
            else
            {
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("There is no configuration file at path '" + __CONF_FILE_PATH + "' (no input data for link directory).");
                }
                
                _lastConfUpdate.put(skinId, (long) 0);
                getThemesCache().put(skinId, null);
            }
        }
        finally
        {
            if (_sourceResolver != null && source != null)
            {
                _sourceResolver.release(source);
            }
        }
    }
    
    /**
     * Read the configuration values and store them
     * @param source the file's source
     * @param skinId The skin
     * @param forceRead true to force reload of values even if the file was not modified
     */
    private synchronized void _cacheConfigurationValues (Source source, String skinId, boolean forceRead)
    {
        long lastModified = source.getLastModified();
        if (!forceRead && _lastConfUpdate.containsKey(skinId) && _lastConfUpdate.get(skinId) != 0 && lastModified == _lastConfUpdate.get(skinId))
        {
            // While waiting for synchronized, someone else may have updated the cache
            return;
        }

        List<ThemeInputData> themesCache = new ArrayList<>();

        getLogger().info("Caching configuration");
        
        try
        {
            Configuration configuration = _directoryHelper.getSkinLinksConfiguration(skinId);
            
            Configuration[] themesConfigurations = configuration.getChild("inputdata").getChildren("themes");
            
            for (Configuration themesConfiguration : themesConfigurations)
            {
                List<Map<String, String>> themes = new ArrayList<> ();
                
                Configuration[] themeConfigurations = themesConfiguration.getChildren();
                for (Configuration themeConfiguration : themeConfigurations)
                {
                    Map<String, String> theme = new HashMap<> ();
                    String id = themeConfiguration.getAttribute("id", null);
                    theme.put("id", id);
                    theme.put("lang", themeConfiguration.getAttribute("lang", null));
                    themes.add(theme);
                }
                
                String[] templates = StringUtils.split(themesConfiguration.getAttribute("templates", __WILDCARD), ',');
                
                ThemeInputData themeInputData = new ThemeInputData(themesConfiguration.getAttribute("inputDataId", StringUtils.EMPTY), Arrays.asList(templates), themes, themesConfiguration.getAttributeAsBoolean("configurable", false), themesConfiguration.getAttributeAsBoolean("displayUserLinks", false));
                themesCache.add(themeInputData);
            }
            
            _configurationError.remove(skinId);
            getThemesCache().put(skinId, themesCache);
            _lastConfUpdate.put(skinId, source.getLastModified());
        }
        catch (Exception e)
        {
            getLogger().warn("An error occured while getting the configuration's file values", e);
            _configurationError.put(skinId, e.getMessage());
        }
    }
       
    /**
     * Get the current template 
     * @param request the request
     * @return the current template
     */
    private String _getTemplate(Request request)
    {
        return (String) request.getAttribute("template");
    }
    
    
     /**
      * Get the current skin 
      * @param request the request
      * @return the current skin
      */
    private String _getSkin(Request request)
    {
        return (String) request.getAttribute("skin");
    }
    
    private Cache<String, List<ThemeInputData>> getThemesCache()
    {
        return _cacheManager.get(__THEMES_CACHE);
    }
    
    private static class ThemeInputData 
    {
        private String _id;
        private List<String> _templates;
        private List<Map<String, String>> _themes;
        private boolean _configurable;
        private boolean _displayUserLinks;
        
        ThemeInputData (String id, List<String> templates, List<Map<String, String>> themes, boolean configurable, boolean displayUserLinks)
        {
            _id = id;
            _templates = templates;
            _themes = themes;
            _configurable = configurable;
            _displayUserLinks = displayUserLinks;
        } 
        
        boolean isConfigurable ()
        {
            return _configurable;
        }
        
        boolean displayUserLinks()
        {
            return _displayUserLinks;
        }
        
        List<String> getTemplates ()
        {
            return _templates;
        }
        
        List<Map<String, String>> getThemes ()
        {
            return _themes;
        }
        
        String getId()
        {
            return _id;
        }
    }
}
