/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.user.status.PersonalDataPolicy;
import org.ametys.core.user.status.PersonalDataProcessingException;
import org.ametys.core.user.status.UserStatusInfo;
import org.ametys.plugins.linkdirectory.link.LinkDAO;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.config.Config;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Delete user links after a period
 */
public class UserLinksDataPolicy implements PersonalDataPolicy, Serviceable
{
    /** The link directory helper */
    protected DirectoryHelper _directoryHelper;
    /** The link DAO */
    protected LinkDAO _linkDAO;
    /** The site manager */
    protected SiteManager _siteManager;
    
    private Period _retentionPeriod;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
        _linkDAO = (LinkDAO) manager.lookup(LinkDAO.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        
        Long config = Config.getInstance().<Long>getValue("link-directory.user-links.data-policy.retention.period", false, null);
        _retentionPeriod = config != null && config >= 0 ? Period.ofMonths(config.intValue()) : null;
    }
    
    public AnonymizationResult process(UserStatusInfo userStatusInfo) throws PersonalDataProcessingException
    {
        if (_retentionPeriod == null)
        {
            return AnonymizationResult.TOO_EARLY;
        }
        else if (userStatusInfo.getMissingSinceDate().isBefore(ZonedDateTime.now().minus(_retentionPeriod)))
        {
            int handled = 0;
            UserIdentity userIdentity = userStatusInfo.getUserIdentity();
            try (AmetysObjectIterable<Site> sites = _siteManager.getSites())
            {
                for (Site site: sites)
                {
                    try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
                    {
                        for (Sitemap sitemap : sitemaps)
                        {
                            try (AmetysObjectIterable<DefaultLink> userLinks = _directoryHelper.getUserLinks(site.getName(), sitemap.getName(), userIdentity))
                            {
                                for (DefaultLink link : userLinks)
                                {
                                    _linkDAO.deleteLink(link);
                                    handled++;
                                }
                            }
                        }
                    }
                }
                
                if (handled > 0)
                {
                    ForensicLogger.info("data.policy.gdpr.remove.user.links", Map.of("handled", Integer.toString(handled), "identity", userIdentity), UserPopulationDAO.SYSTEM_USER_IDENTITY);
                    return AnonymizationResult.PROCESSED;
                }
                else
                {
                    return AnonymizationResult.NO_DATA;
                }
            }
            catch (AmetysRepositoryException e)
            {
                throw new PersonalDataProcessingException("An error prevented the processing of the links for user '" + userIdentity + "'.", e);
            }
        }
        else
        {
            return AnonymizationResult.TOO_EARLY;
        }
    }
}
