/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.link;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.plugins.linkdirectory.LinkDirectoryService;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Generate the links' list of a link directory as XML.
 */
public class LinksGenerator extends ServiceableGenerator 
{
    private static final String __DISPLAY_USER_LINKS_PARAMETER = "displayUserLinks";
    
    /** The current user provider */
    private CurrentUserProvider _currentUserProvider;
    private DirectoryHelper _directoryHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _directoryHelper = (DirectoryHelper) serviceManager.lookup(DirectoryHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        Page page = (Page) request.getAttribute(Page.class.getName());
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());

        String siteName = (String) request.getAttribute("site");
        String language = (String) request.getAttribute("renderingLanguage");
        
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        String[] themes = serviceParameters.getValue("themes", false, new String[0]);
        
        contentHandler.startDocument();
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("siteName", siteName);
        attrs.addCDATAAttribute("language", language);
        if (page != null)
        {
            attrs.addCDATAAttribute("page", page.getId());
        }
        
        UserIdentity user = _currentUserProvider.getUser();
       
        // Get the themes as a list
        List<DefaultLink> links = _directoryHelper.getLinks(Arrays.asList(themes), siteName, language);
        
        String storageContext = null;
        
        // Get the zone item if we sax for a service
        attrs.addCDATAAttribute("zoneItemId", zoneItem.getId());
        
        if (user != null)
        {
            storageContext = _directoryHelper.getStorageContext(request, zoneItem.getId());
        }
        
        String customThemeName = serviceParameters.hasValue("custom-theme") ? serviceParameters.getValue("custom-theme") : null;
        
        if (_directoryHelper.themeExists(customThemeName, siteName, language))
        {
            attrs.addCDATAAttribute("custom-theme", customThemeName);
        }
        
        // SAX the links
        XMLUtils.startElement(contentHandler, "links", attrs);
        try
        {
            // Sax the amount of deleted themes 
            _saxDeletedThemes(siteName, language, Arrays.asList(themes));
            
            // SAX user links
            List<DefaultLink> userLinks = null;
            if (user != null && serviceParameters.getValue(__DISPLAY_USER_LINKS_PARAMETER, false, false))
            {
                userLinks = _directoryHelper.getUserLinks(siteName, language, user, customThemeName).stream().collect(Collectors.toList());
            }
            
            boolean configurable = serviceParameters.getValue(LinkDirectoryService.CONFIGURABLE_LINKS_PARAMETER, false, false);
            _directoryHelper.saxLinks(siteName, contentHandler, links, userLinks, Arrays.asList(themes), configurable, _directoryHelper.getContextVars(request), storageContext, user);
        }
        catch (Exception e)
        {
            getLogger().error("An exception occurred while saxing the links", e);
        }
        
        XMLUtils.endElement(contentHandler, "links");
        contentHandler.endDocument();
    }

    /**
     * Sax the amount of deleted themes
     * @param siteName the site's name
     * @param language the language
     * @param themesNamesList the list of theme ids
     * @throws SAXException if an exception occurs while saxing
     */
    protected void _saxDeletedThemes(String siteName, String language, List<String> themesNamesList) throws SAXException
    {
        int amount = 0;
        for (String themeName : themesNamesList)
        {
            if (!_directoryHelper.themeExists(themeName, siteName, language))
            {
                // The theme was deleted but it is still referenced by the service
                getLogger().warn("The id '" + themeName + "' does not reference an existing theme.");
                amount++;
            }
        }
        
        AttributesImpl attr = new AttributesImpl();
        attr.addCDATAAttribute("count", Integer.toString(amount));
        XMLUtils.createElement(contentHandler, "unknown-themes", attr);
    }
}
