/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.repository;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.Binary;
import org.ametys.plugins.linkdirectory.Link;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.MovableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * Repository implementation of a directory link.
 */
public class DefaultLink extends DefaultTraversableAmetysObject<DefaultLinkFactory> implements Link, SiteAwareAmetysObject, MovableAmetysObject, ModifiableModelLessDataAwareAmetysObject
{
    /** Constant for URL property. */
    public static final String PROPERTY_URL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":url";

    /** Constant for id of provider of dynamic information. */
    public static final String PROPERTY_DYNAMIC_INFO_PROVIDER = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":dynamic-information";

    /** Constant for internal URL property. */
    public static final String PROPERTY_INTERNAL_URL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":internal-url";

    /** Constant for URL type property. */
    public static final String PROPERTY_URLTYPE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":url-type";

    /** Constant for title property. */
    public static final String PROPERTY_TITLE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":title";

    /** Constant for title property. */
    public static final String PROPERTY_CONTENT = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":content";

    /** Constant for URL alternative property. */
    public static final String PROPERTY_URL_ALTERNATIVE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":url-alternative";

    /** Constant for picture type property. */
    public static final String PROPERTY_PICTURE_TYPE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":picture-type";

    /** Constant for picture data property. */
    public static final String PROPERTY_PICTURE = "picture";

    /** Constant for picture id property. */
    public static final String PROPERTY_PICTURE_ID = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":picture-id";
    
    /** Constant for picture glyph property. */
    public static final String PROPERTY_PICTURE_GLYPH = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":picture-glyph";

    /** Constant for picture alternative property. */
    public static final String PROPERTY_PICTURE_ALTERNATIVE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":picture-alternative";

    /** Constant for themes property. */
    public static final String PROPERTY_THEMES = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":themes";
    
    /** Constant for color property. */
    public static final String PROPERTY_COLOR = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":color";

    /** Constant for page property. */
    public static final String PROPERTY_PAGE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":page";

    /** Constant for status property. */
    public static final String PROPERTY_STATUS = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":status";
    
    /** Constant for default visibilty property. */
    public static final String PROPERTY_DEFAULT_VISIBILITY = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":default-visibility";


    /**
     * Create a {@link DefaultLink}.
     * 
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link DefaultLinkFactory} which creates the
     *            AmetysObject.
     */
    public DefaultLink(Node node, String parentPath, DefaultLinkFactory factory)
    {
        super(node, parentPath, factory);
    }

    @Override
    public String getUrl() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_URL).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the URL property.", e);
        }
    }

    @Override
    public void setUrl(LinkType urlType, String url) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_URLTYPE, urlType.toString());
            getNode().setProperty(PROPERTY_URL, url);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the URL property.", e);
        }
    }

    @Override
    public String getInternalUrl() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_INTERNAL_URL).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the internal URL property.", e);
        }
    }

    @Override
    public void setInternalUrl(String url) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_INTERNAL_URL, url);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the internal URL property.", e);
        }
    }

    @Override
    public LinkType getUrlType() throws AmetysRepositoryException
    {
        try
        {
            return LinkType.valueOf(getNode().getProperty(PROPERTY_URLTYPE).getString());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get URL type property", e);
        }
    }

    @Override
    public String getTitle() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_TITLE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the title property.", e);
        }
    }

    @Override
    public void setTitle(String title) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_TITLE, title);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the title property.", e);
        }
    }

    @Override
    public String getContent() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_CONTENT).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the content property.", e);
        }
    }

    @Override
    public void setContent(String content) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_CONTENT, content);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the content property.", e);
        }
    }

    @Override
    public String getAlternative() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_URL_ALTERNATIVE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the alternative property.", e);
        }
    }

    @Override
    public void setAlternative(String alternative) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_URL_ALTERNATIVE, alternative);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the alternative property.", e);
        }
    }

    @Override
    public Binary getExternalPicture() throws AmetysRepositoryException
    {
        return getValue(PROPERTY_PICTURE);
    }

    @Override
    public void setExternalPicture(String mimeType, String filename, InputStream stream) throws AmetysRepositoryException
    {
        try
        {
            removePictureMetas();

            setPictureType("external");

            Binary pic = new Binary();

            pic.setLastModificationDate(ZonedDateTime.now());
            pic.setInputStream(stream);
            pic.setMimeType(mimeType);
            pic.setFilename(filename);
            
            setValue(PROPERTY_PICTURE, pic);
        }
        catch (IOException | RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the external picture property.", e);
        }
    }

    @Override
    public String getResourcePictureId() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PICTURE_ID).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the picture ID property.", e);
        }
    }

    @Override
    public void setResourcePicture(String resourceId) throws AmetysRepositoryException
    {
        try
        {
            removePictureMetas();

            setPictureType("resource");

            getNode().setProperty(PROPERTY_PICTURE_ID, resourceId);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the alternative property.", e);
        }
    }

    @Override
    public void setNoPicture() throws AmetysRepositoryException
    {
        try
        {
            setPictureType("");

            removePictureMetas();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the alternative property.", e);
        }
    }

    @Override
    public String getPictureType() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PICTURE_TYPE).getString();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the type property.", e);
        }
    }

    @Override
    public void setPictureType(String type) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_PICTURE_TYPE, type);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the type property.", e);
        }
    }
    
    public String getPictureGlyph() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PICTURE_GLYPH).getString();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the picture glyph property.", e);
        }
    }

    public void setPictureGlyph(String glyph) throws AmetysRepositoryException
    {
        try
        {
            setPictureType("glyph");

            getNode().setProperty(PROPERTY_PICTURE_GLYPH, glyph);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the picture glyph property.", e);
        }
    }

    @Override
    public String getPictureAlternative() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PICTURE_ALTERNATIVE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the alternative property.", e);
        }
    }

    @Override
    public void setPictureAlternative(String alternative) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_PICTURE_ALTERNATIVE, alternative);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the alternative property.", e);
        }
    }

    @Override
    public String[] getThemes() throws AmetysRepositoryException
    {
        return _getListFromMetadataName(PROPERTY_THEMES);
    }

    @Override
    public void setThemes(String[] themes) throws AmetysRepositoryException
    {
        _setListWithMetadataName(themes, PROPERTY_THEMES);
    }

    @Override
    public void removeTheme(String themeId) throws AmetysRepositoryException
    {
        try
        {
            String[] themes = getThemes();
            String[] updatedThemes = ArrayUtils.removeElement(themes, themeId);
            getNode().setProperty(PROPERTY_THEMES, updatedThemes);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error removing theme of id " + themeId, e);
        }
    }

    @Override
    public Site getSite() throws AmetysRepositoryException
    {
        AmetysObject parent = getParent();
        while (parent != null)
        {
            if (parent instanceof Site)
            {
                return (Site) parent;
            }
            parent = parent.getParent();
        }

        return null;
    }

    @Override
    public String getSiteName() throws AmetysRepositoryException
    {
        return getSite().getName();
    }

    /**
     * Get the theme language.
     * 
     * @return the theme language.
     */
    public String getLanguage()
    {
        return getParent().getParent().getName();
    }

    /**
     * Remove all picture metas (picture ID and picture binary).
     * 
     * @throws RepositoryException if an error occurs.
     */
    protected void removePictureMetas() throws RepositoryException
    {
        getNode().setProperty(PROPERTY_PICTURE_ID, StringUtils.EMPTY);
        removeValue(PROPERTY_PICTURE);
    }

    @Override
    public void orderBefore(AmetysObject siblingObject) throws AmetysRepositoryException
    {
        if (siblingObject instanceof DefaultLink)
        {
            DefaultLink sibling = (DefaultLink) siblingObject;
            sibling.getPath();
            Node node = getNode();
            String siblingPath = "";
            try
            {
                siblingPath = sibling.getNode().getPath();
                if (siblingPath.contains("/"))
                {
                    siblingPath = StringUtils.substringAfterLast(siblingPath, "/");
                }
                String path = node.getPath();
                if (path.contains("/"))
                {
                    path = StringUtils.substringAfterLast(path, "/");
                }
                node.getParent().orderBefore(path, siblingPath);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(String.format("Unable to order link '%s' before link '%s'", this, siblingPath), e);
            }
        }
        else
        {
            throw new AmetysRepositoryException("A link can only be moved before another link.");
        }
    }

    @Override
    public void moveTo(AmetysObject newParent, boolean renameIfExist) throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        if (getParent().equals(newParent))
        {
            try
            {
                // Just order current node to the end.
                Node node = getNode();
                String path = node.getPath();
                if (path.contains("/"))
                {
                    path = StringUtils.substringAfterLast(path, "/");
                }
                node.getParent().orderBefore(path, null);
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException(String.format("Unable to move link '%s' outside the root link node.", this), e);
            }
        }
        else
        {
            throw new AmetysRepositoryException("A link can only be moved before another link.");
        }
    }

    @Override
    public boolean canMoveTo(AmetysObject newParent) throws AmetysRepositoryException
    {
        return getParent().equals(newParent);
    }

    /**
     * Retrieves the list of values corresponding to the metadata name passed as
     * parameter
     * 
     * @param metadataName the name of the metadata to retrieve
     * @return the list corresponding to the metadata name
     */
    private String[] _getListFromMetadataName(String metadataName)
    {
        try
        {
            if (getNode().hasProperty(metadataName))
            {
                Value[] values = getNode().getProperty(metadataName).getValues();

                String[] list = new String[values.length];

                for (int i = 0; i < values.length; i++)
                {
                    list[i] = values[i].getString();
                }

                return list;
            }
            else
            {
                return new String[0];
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occurred while trying to get the property '" + metadataName + "'.", e);
        }
    }

    /**
     * Sets the list of values to the node corresponding to the metadata name
     * passed as a parameter
     * 
     * @param list the list of value to be set
     * @param metadataName the name of the metadata
     */
    private void _setListWithMetadataName(String[] list, String metadataName)
    {
        try
        {
            getNode().setProperty(metadataName, list);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occurred while trying to set the property '" + metadataName + "'.", e);
        }
    }

    @Override
    public String getDynamicInformationProvider() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_DYNAMIC_INFO_PROVIDER).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the URL property.", e);
        }
    }

    @Override
    public void setDynamicInformationProvider(String providerId) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_DYNAMIC_INFO_PROVIDER, providerId);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the URL property.", e);
        }
    }
    
    @Override
    public String getColor() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_COLOR).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the color property.", e);
        }
    }

    @Override
    public void setColor(String color) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_COLOR, color);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the color property.", e);
        }
    }
    
    @Override
    public String getPage() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PAGE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the page property.", e);
        }
    }

    @Override
    public void setPage(String page) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_PAGE, page);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the page property.", e);
        }
    }
    
    @Override
    public LinkStatus getStatus() throws AmetysRepositoryException
    {
        try
        {
            return LinkStatus.valueOf(getNode().getProperty(PROPERTY_STATUS).getString());
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the status property.", e);
        }
    }

    @Override
    public void setStatus(LinkStatus status) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_STATUS, status.name());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the status property.", e);
        }
    }
    
    @Override
    public LinkVisibility getDefaultVisibility() throws AmetysRepositoryException
    {
        try
        {
            Node node = getNode();
            if (node.hasProperty(PROPERTY_DEFAULT_VISIBILITY))
            {
                return LinkVisibility.valueOf(getNode().getProperty(PROPERTY_DEFAULT_VISIBILITY).getString());
            }
            else
            {
                return LinkVisibility.VISIBLE;
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the visibility property.", e);
        }
    }

    @Override
    public void setDefaultVisibility(LinkVisibility visibility) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_DEFAULT_VISIBILITY, visibility.name());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the visibility property.", e);
        }
    }
    
    public ModifiableModelLessDataHolder getDataHolder()
    {
        JCRRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_getFactory().getLinkDataTypeExtensionPoint(), repositoryData);
    }
}
