/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.repository;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.actions.ExplorerResourcesDAO;
import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.plugins.linkdirectory.Link.LinkType;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.site.CopyUpdater;

/**
 * Update {@link DefaultLink} attribute targeting {@link AmetysObject} related to the original site
 */
public class LinksCopyUpdater extends AbstractLogEnabled implements CopyUpdater, Serviceable
{
    /** the ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** the directory helper */
    protected DirectoryHelper _directoryHelper;
    /** the explorer DAO */
    protected ExplorerResourcesDAO _explorerDAO;

    public void service(ServiceManager manager) throws ServiceException
    {
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
        _explorerDAO = (ExplorerResourcesDAO) manager.lookup(ExplorerResourcesDAO.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void updateSite(Site initialSite, Site createdSite)
    {
        try (AmetysObjectIterable<Sitemap> sitemaps = createdSite.getSitemaps())
        {
            for (Sitemap sitemap : sitemaps)
            {
                String allLinksQuery = _directoryHelper.getAllLinksQuery(createdSite.getName(), sitemap.getName());
                try (AmetysObjectIterable<DefaultLink> links = _resolver.query(allLinksQuery))
                {
                    for (DefaultLink link : links)
                    {
                        if (link.getUrlType() == LinkType.PAGE)
                        {
                            link.setUrl(LinkType.PAGE, _getPage(link.getUrl(), initialSite, createdSite));
                        }
                        
                        String pageId = link.getPage();
                        if (StringUtils.isNotBlank(link.getPage()))
                        {
                            link.setPage(_getPage(pageId, initialSite, createdSite));
                        }
                        
                        if ("resource".equals(link.getPictureType()))
                        {
                            link.setResourcePicture(_getResource(link.getResourcePictureId(), initialSite, createdSite));
                        }
                    }
                }
            }
        }
        
        if (createdSite.needsSave())
        {
            createdSite.saveChanges();
        }
    }
    
    /**
     * Get the updated page id.
     * @param initialPageId the initial page id
     * @param initialSite the initial site
     * @param createdSite the newly created site
     * @return the updated id.
     */
    protected String _getPage(String initialPageId, Site initialSite, Site createdSite)
    {
        try
        {
            Page initialPage = _resolver.resolveById(initialPageId);
            // update only if the page was part of the initial site.
            if (initialPage.getSite().equals(initialSite))
            {
                Sitemap createdSitemap = createdSite.getSitemap(initialPage.getSitemapName());
                Page createdPage = createdSitemap.getChild(initialPage.getPathInSitemap());
                return createdPage.getId();
            }
            else
            {
                return initialPageId;
            }
        }
        catch (AmetysRepositoryException e)
        {
            getLogger().error("Failed to migrate link to page '{}' from  site '{}' to site '{}'", initialPageId, initialSite.getName(), createdSite.getName(), e);
            return initialPageId;
        }
    }
    
    /**
     * Get the updated resource id.
     * @param initialResourceId the initial resource id
     * @param initialSite the initial site
     * @param createdSite the newly created site
     * @return the updated id
     */
    protected String _getResource(String initialResourceId, Site initialSite, Site createdSite)
    {
        try
        {
            Resource initialResource = _resolver.resolveById(initialResourceId);
            // update only if the resource was part of the initial site resources
            if (initialResource.getPath().startsWith(initialSite.getRootResources().getPath()))
            {
                Resource newResource = createdSite.getRootResources().getChild(initialResource.getResourcePath().substring(1));
                return newResource.getId();
            }
            else
            {
                return initialResourceId;
            }
        }
        catch (AmetysRepositoryException e)
        {
            getLogger().error("Failed to migrate link picture '{}' from  site '{}' to site '{}'", initialResourceId, initialSite.getName(), createdSite.getName(), e);
            return initialResourceId;
        }
    }
    
    public void updatePage(Site initialSite, Site createdSite, Page page)
    {
        // Nothing to do
    }
    
    public void updateContent(Site initialSite, Site createdSite, Content initialContent, Content createdContent)
    {
        // Nothing to do
    }
}
