/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.linkdirectory.right;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractProfileStorageBasedAccessController;
import org.ametys.plugins.linkdirectory.DirectoryHelper;
import org.ametys.plugins.linkdirectory.Link;
import org.ametys.plugins.linkdirectory.repository.DefaultLink;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * {@link AccessController} for a {@link Link}
 */
public class LinkDirectoryAccessController extends AbstractProfileStorageBasedAccessController implements Contextualizable
{
    /** the link directory context category */
    public static final I18nizableText LINK_DIRECTORY_CONTEXT_CATEGORY = new I18nizableText("plugin.link-directory", "PLUGINS_LINKDIRECTORY_LINKDIRECTORY_RIGHT_ASSIGNMENT_CONTEXT_LABEL");
    
    /** The link directory helper */
    protected DirectoryHelper _directoryHelper;
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    private Context _context;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _directoryHelper = (DirectoryHelper) manager.lookup(DirectoryHelper.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Link;
    }
    
    @Override
    protected boolean ignoreOnHasAnyPermission()
    {
        return true;
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        Site site = _siteManager.getSite(siteName);
        if (site != null)
        {
            UserIdentity user = _currentUserProvider.getUser();
            Set<Object> roots = new HashSet<>();
            try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
            {
                for (Sitemap sitemap: sitemaps)
                {
                    roots.add(_directoryHelper.getLinksNode(site, sitemap.getName()));
                    if (user != null)
                    {
                        roots.add(_directoryHelper.getLinksForUserNode(site, siteName, user));
                    }
                }
            }
            return roots;
        }
        return null;
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        return new I18nizableText("plugin.link-directory", "PLUGINS_LINKDIRECTORY_LINK_ACCESS_CONTROLLER_LINK_CONTEXT_LABEL", Map.of("title", getObjectLabel(object)));
    }
    
    public I18nizableText getObjectLabel(Object object) throws RightsException
    {
        if (object instanceof DefaultLink link)
        {
            String title = link.getTitle();
            if (StringUtils.isNotBlank(title))
            {
                return new I18nizableText(link.getLanguage().toUpperCase() + " > " + title + " (" + link.getUrl() + ")");
            }
            else
            {
                return new I18nizableText(link.getLanguage().toUpperCase() + " > " + link.getUrl());
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return LINK_DIRECTORY_CONTEXT_CATEGORY;
    }
}
