/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.slf4j.Logger;

import org.ametys.cms.search.model.SystemPropertyExtensionPoint;
import org.ametys.cms.search.ui.model.SearchUIColumnHelper;
import org.ametys.cms.search.ui.model.SearchUIModel;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewHelper;
import org.ametys.runtime.model.ViewItemContainer;

/**
 * Wrapper for a SearchUIModel to force LastValidationSystemProperty to be in result fields
 */
public class SearchModelWrapper implements SearchUIModel
{
    /** The logger */
    protected Logger _logger;

    /** The service manager */
    protected ServiceManager _manager;

    /** The SystemProperty extension point */
    protected SystemPropertyExtensionPoint _systemPropertyExtensionPoint;
    
    private SearchUIModel _realModel;
    
    /**
     * Constructor with the SearchUIModel to wrap
     * @param model the SearchUIModel to wrap
     * @param manager The service manager
     * @param logger The logger
     */
    public SearchModelWrapper(SearchUIModel model, ServiceManager manager, Logger logger)
    {
        _realModel = model;
        _manager = manager;
        _logger = logger;
    }

    public ViewItemContainer getResultItems(Map<String, Object> contextualParameters)
    {
        ViewItemContainer resultItems = _realModel.getResultItems(contextualParameters);
        
        // If no item is already a LastValidationSearchField, add it
        if (!resultItems.hasModelViewItem("lastValidation"))
        {
            try
            {
                View result = new View();
                
                result.addViewItems(ViewHelper.copyViewItems(resultItems.getViewItems()));
                resultItems.addViewItem(SearchUIColumnHelper.createModelItemColumn(getSystemPropertyExtensionPoint().getExtension("lastValidation")));
                
                return result;
            }
            catch (ServiceException e)
            {
                getLogger().error("Error while adding 'lastValidation' in the set of result columns", e);
            }
        }
        
        return resultItems;
    }
    
    /**
     * Retrieves the {@link SystemPropertyExtensionPoint}
     * @return The {@link SystemPropertyExtensionPoint}
     * @throws ServiceException if an error occurs while looking to the extension point up
     */
    protected SystemPropertyExtensionPoint getSystemPropertyExtensionPoint() throws ServiceException
    {
        if (_systemPropertyExtensionPoint == null)
        {
            _systemPropertyExtensionPoint = (SystemPropertyExtensionPoint) _manager.lookup(SystemPropertyExtensionPoint.ROLE);
        }
        
        return _systemPropertyExtensionPoint;
    }
    
    public Set<String> getContentTypes(Map<String, Object> contextualParameters)
    {
        return _realModel.getContentTypes(contextualParameters);
    }

    public Set<String> getExcludedContentTypes(Map<String, Object> contextualParameters)
    {
        return _realModel.getExcludedContentTypes(contextualParameters);
    }

    public ViewItemContainer getCriteria(Map<String, Object> contextualParameters)
    {
        return _realModel.getCriteria(contextualParameters);
    }
    
    public ViewItemContainer getFacetedCriteria(Map<String, Object> contextualParameters)
    {
        return _realModel.getFacetedCriteria(contextualParameters);
    }
    
    public ViewItemContainer getAdvancedCriteria(Map<String, Object> contextualParameters)
    {
        return _realModel.getAdvancedCriteria(contextualParameters);
    }
    
    public int getPageSize(Map<String, Object> contextualParameters)
    {
        return _realModel.getPageSize(contextualParameters);
    }

    public String getWorkspace(Map<String, Object> contextualParameters)
    {
        return _realModel.getWorkspace(contextualParameters);
    }

    public String getSearchUrl(Map<String, Object> contextualParameters)
    {
        return _realModel.getSearchUrl(contextualParameters);
    }

    public String getSearchUrlPlugin(Map<String, Object> contextualParameters)
    {
        return _realModel.getSearchUrlPlugin(contextualParameters);
    }

    public String getExportCSVUrl(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportCSVUrl(contextualParameters);
    }

    public String getExportCSVUrlPlugin(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportCSVUrlPlugin(contextualParameters);
    }

    public String getExportDOCUrl(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportDOCUrl(contextualParameters);
    }

    public String getExportDOCUrlPlugin(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportDOCUrlPlugin(contextualParameters);
    }

    public String getExportXMLUrl(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportXMLUrl(contextualParameters);
    }

    public String getExportXMLUrlPlugin(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportXMLUrlPlugin(contextualParameters);
    }

    public String getExportPDFUrl(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportPDFUrl(contextualParameters);
    }

    public String getExportPDFUrlPlugin(Map<String, Object> contextualParameters)
    {
        return _realModel.getExportPDFUrlPlugin(contextualParameters);
    }

    public String getPrintUrl(Map<String, Object> contextualParameters)
    {
        return _realModel.getPrintUrl(contextualParameters);
    }

    public String getPrintUrlPlugin(Map<String, Object> contextualParameters)
    {
        return _realModel.getPrintUrlPlugin(contextualParameters);
    }

    public String getSummaryView()
    {
        return _realModel.getSummaryView();
    }

    public Map<String, Object> toJSON(Map<String, Object> contextualParameters)
    {
        return _realModel.toJSON(contextualParameters);
    }
    
    public List<Object> resultItemsToJSON(Map<String, Object> contextualParameters)
    {
        return _realModel.resultItemsToJSON(contextualParameters);
    }

    /**
     * Get the logger.
     * @return the logger.
     */
    protected final Logger getLogger()
    {
        return _logger;
    }
}
