/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;

import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.plugins.core.impl.authentication.FormCredentialProvider;
import org.ametys.plugins.mobileapp.PostConstants;

/**
 * Returns the latest version compatible between client and server
 */
public class GetPopulationsAction extends AbstractPostAction
{
    /** The helper for the associations population/context */
    protected PopulationContextHelper _populationContextHelper;
    /** The user population DAO */
    protected UserPopulationDAO _userPopulationDAO;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _populationContextHelper = (PopulationContextHelper) smanager.lookup(PopulationContextHelper.ROLE);
        _userPopulationDAO = (UserPopulationDAO) smanager.lookup(UserPopulationDAO.ROLE);
    }

    @Override
    protected Map<String, Object> doAction(Request request, Map<String, Object> jsonParams)
    {
        Map<String, Object> result = new HashMap<>();
        String siteName = (String) getParameter(PostConstants.SITE_NAME, jsonParams, request);

        if (siteName != null)
        {
            List<String> populationContexts = List.of("/sites/" + siteName, "/sites-fo/" + siteName);
            List<Map<String, Object>> userPopulationsList = new ArrayList<>();
            for (String context : populationContexts)
            {
                userPopulationsList.addAll(_populationContextHelper.getUserPopulationsOnContexts(List.of(context), false, false)
                        .stream()
                        .map(_userPopulationDAO::getUserPopulation)
                        .filter(population -> isCorrectPopulation(population))
                        .map(userPopulation -> Map.of("key", userPopulation.getId(), "value", userPopulation.getLabel()))
                        .toList());
            }
            result.put("userPopulations", userPopulationsList);
        }

        result.put("code", 200);
        return result;
    }

    private boolean isCorrectPopulation(UserPopulation population)
    {
        return population.getCredentialProviders().stream().anyMatch(FormCredentialProvider.class::isInstance);
    }

}
