/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpResponse;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.JSONUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

import com.google.gson.JsonParseException;

/**
 * Returns the theme for a site
 */
public class GetThemeAction extends ServiceableAction
{
    /** global configuration about projects enabled/disabled */
    protected static final String __PROJECT_ENABLED_CONF_ID = "plugin.mobileapp.project.enabled";

    /** Path of the base directory for mobileapp conf */
    protected static final String BASE_DIR = "context://WEB-INF/param/mobileapp/";

    /** Path of the theme.json file */
    public static final String THEME_JSON = BASE_DIR + "theme.json";

    /** Authentication Token Manager */
    protected SiteManager _siteManager;

    /** Source Resolver */
    protected org.apache.excalibur.source.SourceResolver _sourceResolver;

    /** JSON Utils */
    protected JSONUtils _jsonUtils;


    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _sourceResolver = (org.apache.excalibur.source.SourceResolver) smanager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }

    /**
     * Get the home web view if exists.
     * It first check the site configuration, then the global configuration, and return null if none is set.
     * @param site the site
     * @return the home web view if exists, null otherwise
     */
    public String getHomeWebView(Site site)
    {
        String siteWebview = site.getValue("mobileapp-webview");
        if (StringUtils.isNotBlank(siteWebview))
        {
            return siteWebview;
        }

        String globalWebview = Config.getInstance().getValue("plugin.mobileapp.home.webview");
        if (StringUtils.isNotBlank(globalWebview))
        {
            return globalWebview;
        }

        return null;
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);

        String siteName = parameters.getParameter("site");
        Site site = _siteManager.getSite(siteName);

        if (site != null)
        {
            try
            {
                Map<String, Object> jsonMap = parseJson();
                result = transformMap(jsonMap, site);
                // Icons first, this is where it can fail

                result.put("code", 200);
                result.put("name", site.getTitle());
                result.put("main_link", site.getUrl());

                boolean projectsEnabled = PluginsManager.getInstance().isPluginActive("workspaces")
                                        && Config.getInstance().getValue(__PROJECT_ENABLED_CONF_ID, false, false);
                result.put("project_enabled", projectsEnabled);

                String homeWebView = getHomeWebView(site);
                if (homeWebView != null)
                {
                    result.put("home_webview", homeWebView);
                }
            }
            catch (JsonParseException e)
            {
                getLogger().error("Invalid JSON file for " + THEME_JSON, e);
                result = new HashMap<>();
                result.put("code", 500);
                result.put("message", "invalid-icons-json");
                HttpResponse response = (HttpResponse) ObjectModelHelper.getResponse(objectModel);
                response.setStatus(500);
            }
        }
        else
        {
            result.put("code", 500);
            result.put("message", "site-not-found");
            HttpResponse response = (HttpResponse) ObjectModelHelper.getResponse(objectModel);
            response.setStatus(500);
        }

        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }

    /**
     * Get the content of the json file as a map
     * @return the map representing the json file
     * @throws IOException impossible to read the json file
     */
    protected Map<String, Object> parseJson() throws IOException
    {
        Source src = _sourceResolver.resolveURI(THEME_JSON);

        File file = ((FileSource) src).getFile();

        if (!file.exists())
        {
            throw new JsonParseException("The file " + THEME_JSON + " does not exist.");
        }

        try (FileInputStream is = FileUtils.openInputStream(file))
        {
            String body = new String(is.readAllBytes(), StandardCharsets.UTF_8);
            Map<String, Object> jsonMap = _jsonUtils.convertJsonToMap(body);
            return jsonMap;
        }
        catch (IllegalArgumentException e)
        {
            throw new JsonParseException("The file " + THEME_JSON + " must be a valid json containing a map with list of maps, at least : {\"icons\":[{\"src\":\"resources/img/logo96x96.png\"}]}.");
        }
    }

    /**
     * Parse the input json, and transform all String called src to be relative to the site and skin used.
     * @param jsonMap the map read from the json file
     * @param site site to use to get it's url and skin
     * @return the json map with src modified
     * @throws MalformedURLException impossible to find the json file
     * @throws IOException impossible to read the json file
     * @throws JsonParseException json file inexistant or invalid
     */
    protected Map<String, Object> transformMap(Map<String, Object> jsonMap, Site site) throws MalformedURLException, IOException, JsonParseException
    {
        Map<String, Object> result = new HashMap<>();

        for (String key : jsonMap.keySet())
        {
            Object object = jsonMap.get(key);
            Object transformedObject = transformObject(key, object, site);
            result.put(key, transformedObject);
        }

        return result;
    }

    /**
     * Parse the input json, and transform all String called src to be relative to the site and skin used.
     * @param jsonList a list in the json file
     * @param site site to use to get it's url and skin
     * @return the json map with src modified
     * @throws MalformedURLException impossible to find the json file
     * @throws IOException impossible to read the json file
     * @throws JsonParseException json file inexistant or invalid
     */
    protected List<Object> transformList(List<Object> jsonList, Site site) throws MalformedURLException, IOException, JsonParseException
    {
        List<Object> result = new ArrayList<>();
        for (Object object : jsonList)
        {
            Object transformedObject = transformObject(null, object, site);
            result.add(transformedObject);
        }
        return result;
    }

    /**
     * Parse the input json, and transform all String called src to be relative to the site and skin used.
     * @param key in case the item was in a map, the key associated (only src will be translated)
     * @param object the object to read from the json
     * @param site site to use to get it's url and skin
     * @return the json map with src modified
     * @throws MalformedURLException impossible to find the json file
     * @throws IOException impossible to read the json file
     * @throws JsonParseException json file inexistant or invalid
     */
    protected Object transformObject(String key, Object object, Site site) throws MalformedURLException, IOException, JsonParseException
    {
        Object result = object;
        if (object instanceof String)
        {
            String value = (String) object;
            if ("src".equalsIgnoreCase(key))
            {
                result = getImageUrl(value, site);
            }
        }
        else if (object instanceof List)
        {
            @SuppressWarnings("unchecked")
            List<Object> list = (List<Object>) object;
            result = transformList(list, site);
        }
        else if (object instanceof Map)
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> map = (Map<String, Object>) object;
            result = transformMap(map, site);
        }

        return result;
    }

    /**
     * Get thu full url of the theme image, based on the site url and used skin
     * @param relativePath path of the image in the skin
     * @param site site used
     * @return the image full url
     */
    protected String getImageUrl(String relativePath, Site site)
    {
        String url = site.getUrl()
                + "/skins/"
                + site.getSkinId()
                + "/"
                + relativePath;

        return url;
    }
}
