/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.text.ParseException;
import java.util.Map;

import org.apache.cocoon.environment.Request;

import org.ametys.core.authentication.CredentialProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.plugins.extrausermgt.authentication.oidc.OIDCBasedCredentialProvider;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.proc.BadJOSEException;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.Issuer;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;
import com.nimbusds.openid.connect.sdk.validators.IDTokenValidator;

/**
 * Action exchanging an OIDC access token for an Ametys mobile app token.
 */
public class GetTokenFromOIDCAction extends AbstractGetTokenAction
{
    @Override
    protected UserIdentity tryConnect(Map<String, Object> params, Request request, String context, UserPopulation userPopulation, CredentialProvider credentialProvider, int credentialProviderIndex)
    {
        if (credentialProvider instanceof OIDCBasedCredentialProvider oidcCredentialProvider)
        {
            try
            {
                String idToken = (String) getParameter("idToken", params, request);

                JWT token = JWTParser.parse(idToken);
                IDTokenValidator validator = new IDTokenValidator(new Issuer(oidcCredentialProvider.getIssuer()),
                                                                  new ClientID(oidcCredentialProvider.getClientId()),
                                                                  JWSAlgorithm.RS256,
                                                                  oidcCredentialProvider.getJwkSetURL());
                IDTokenClaimsSet claims = validator.validate(token, null);
                
                String login = claims.getStringClaim("preferred_username");
                
                for (UserDirectory userDirectory : userPopulation.getUserDirectories())
                {
                    if (userDirectory.getStoredUser(login) != null)
                    {
                        return new UserIdentity(login, userPopulation.getId());
                    }
                }
            }
            catch (ParseException | BadJOSEException | JOSEException e)
            {
                getLogger().error("Impossible to validate id token  on population '" + userPopulation.getId() + "' using credential provider at position '" + credentialProviderIndex + "'", e);
            }
        }
        
        return null;
    }
}
