/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.mobileapp.PostConstants;
import org.ametys.plugins.mobileapp.UserPreferencesHelper;

/**
 * Returns the list of feeds for a user
 */
public class SetNotificationsPrefsAction extends AbstractLoggedAction
{
    /** Parameter containing the notification switch */
    protected static final String _NOTIF_ENABLED = "enabled";

    /** Parameter containing a marker for all feeds for notification */
    protected static final String _NOTIF_ALL_FEEDS = "allFeeds";

    /** Parameter containing the requested feeds for notification */
    protected static final String _NOTIF_FEEDS = "feeds";

    /** Parameter containing the requested projects for notification */
    protected static final String _NOTIF_WORKSPACES = "workspaces";

    /** Parameter containing a marker for all projects names for notification */
    protected static final String _NOTIF_ALL_PROJECTS = "allProjects";
    
    /** Parameter containing the requested projects names for notification */
    protected static final String _NOTIF_PROJECTS = "projects";

    /** Parameter containing a marker for all types for notification */
    protected static final String _NOTIF_ALL_TYPES = "allTypes";

    /** Parameter containing the requested types for notification */
    protected static final String _NOTIF_TYPES = "types";

    /** User Preferences Helper */
    protected UserPreferencesHelper _userPreferencesHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _userPreferencesHelper = (UserPreferencesHelper) smanager.lookup(UserPreferencesHelper.ROLE);
    }

    @Override
    @SuppressWarnings("unchecked")
    protected Map<String, Object> doLoggedInAction(Request request, Map<String, Object> jsonParams)
    {
        Map<String, Object> result = new HashMap<>();
        String lang = (String) getParameter(PostConstants.LANG, jsonParams, request);

        String pushToken = (String) getParameter(PostConstants.NOTIF_TOKEN, jsonParams, request);

        List<String> feeds = (List<String>) getParameter(_NOTIF_FEEDS, jsonParams, request);

        Map<String, Object> workspaces = (Map<String, Object>) getParameter(_NOTIF_WORKSPACES, jsonParams, request);
        List<String> projects = workspaces != null ? (List<String>) workspaces.get(_NOTIF_PROJECTS) : null;
        List<String> types = workspaces != null ? (List<String>) workspaces.get(_NOTIF_TYPES) : null;
        
        // App Mobile < 1.5.0 only send feeds, projects and types.
        // For compatibility purposes, if theses arrays are present, we assume the all* fields are not set
        boolean defaultAllFeedsValue = feeds == null;
        boolean defaultAllProjectsValue = projects == null;
        boolean defaultAllTypesValue = types == null;
        
        // if all three arrays are present and empty, we assume the default value is to disable notifications
        boolean defaultDisabled = feeds != null && feeds.isEmpty() && projects != null && projects.isEmpty() && types != null && types.isEmpty();
        
        boolean enabled = _getValueOrDefault((Boolean) getParameter(_NOTIF_ENABLED, jsonParams, request), !defaultDisabled);
        
        boolean allFeeds = _getValueOrDefault((Boolean) getParameter(_NOTIF_ALL_FEEDS, jsonParams, request), defaultAllFeedsValue);
        boolean allProjects = _getValueOrDefault(workspaces != null ? (Boolean) workspaces.get(_NOTIF_ALL_PROJECTS) : null, defaultAllProjectsValue);
        boolean allTypes = _getValueOrDefault(workspaces != null ? (Boolean) workspaces.get(_NOTIF_ALL_TYPES) : null, defaultAllTypesValue);

        // Remove null items
        feeds = feeds != null ? feeds.stream().filter(StringUtils::isNotBlank).toList() : null;
        types = types != null ? types.stream().filter(StringUtils::isNotBlank).toList() : null;
        projects = projects != null ? projects.stream().filter(StringUtils::isNotBlank).toList() : null;

        _userPreferencesHelper.setNotificationSettings(pushToken, enabled, allFeeds, feeds, allProjects, projects, allTypes, types, lang);

        return result;
    }
    
    private boolean _getValueOrDefault(Boolean value, boolean defaultValue)
    {
        return value != null ? value : defaultValue;
    }
}
