/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.action;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;

import org.ametys.core.authentication.CredentialProvider;
import org.ametys.core.authentication.token.AuthenticationTokenManager;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.JSONUtils;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Returns the token for a user (only to be used by the mobile app on one site)
 */
public abstract class AbstractGetTokenAction extends ServiceableAction implements ThreadSafe
{
    /** Authentication Token Manager */
    protected AuthenticationTokenManager _authenticationTokenManager;

    /** The user population DAO */
    protected UserPopulationDAO _userPopulationDAO;

    /** The helper for the associations population/context */
    protected PopulationContextHelper _populationContextHelper;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** JSON Utils */
    protected JSONUtils _jsonUtils;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _authenticationTokenManager = (AuthenticationTokenManager) smanager.lookup(AuthenticationTokenManager.ROLE);

        _userPopulationDAO = (UserPopulationDAO) smanager.lookup(UserPopulationDAO.ROLE);
        _populationContextHelper = (PopulationContextHelper) smanager.lookup(PopulationContextHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);

        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        Request request = ObjectModelHelper.getRequest(objectModel);

        String body = null;
        HttpServletRequest postReq = (HttpServletRequest) objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);
        try (InputStream postBody = postReq.getInputStream())
        {
            body = new String(postBody.readAllBytes(), StandardCharsets.UTF_8);
        }

        String siteName = parameters.getParameter("site");
        List<String> populationContexts = List.of("/sites/" + siteName, "/sites-fo/" + siteName);

        Map<String, Object> jsonParams = _jsonUtils.convertJsonToMap(body);

        String population = (String) getParameter("population", jsonParams, request);

        UserIdentity user = null;
        UserPopulation userPopulation = population != null ? _userPopulationDAO.getUserPopulation(population) : null;

        if (userPopulation != null)
        {
            user = authenticate(jsonParams, request, parameters, populationContexts, userPopulation);
        }
        else
        {
            user = authenticate(jsonParams, request, parameters, populationContexts);
        }

        if (user != null)
        {
            String generateToken = _authenticationTokenManager.generateToken(user, 0, "mobileapp", "Token for the mobile app");

            result.put("code", 200);
            result.put("token", generateToken);
            
            // If the authentication process has lead to an effective session creation, it should be destroyed
            // Do not provide the redirector. We don't want the request to be redirected
            _currentUserProvider.logout(null);
        }
        else
        {
            result.put("code", 403);
            throw new AccessDeniedException();
        }

        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    /**
     * Get a parameter from the mobile app, either from the request body or a request parameter
     * @param name the parameter name
     * @param jsonParams the decoded JSON request body, if any
     * @param request the request
     * @return the parameter value, or null if not found
     */
    protected Object getParameter(String name, Map<String, Object> jsonParams, Request request)
    {
        if (jsonParams.containsKey(name))
        {
            return jsonParams.get(name);
        }
        else if (request.getParameter(name) != null)
        {
            return request.getParameter(name);
        }
        else
        {
            return request.getAttribute(name);
        }
    }

    private UserIdentity authenticate(Map<String, Object> params, Request request, Parameters parameters, List<String> populationContexts) throws ParameterException
    {
        String siteName = parameters.getParameter("site");

        for (String context : populationContexts)
        {
            List<UserPopulation> populations = _populationContextHelper.getUserPopulationsOnContexts(List.of(context), false, false).stream()
                    .map(_userPopulationDAO::getUserPopulation)
                    .toList();

            for (UserPopulation population : populations)
            {
                UserIdentity user = _tryConnect(params, request, context, population);
                if (user != null)
                {
                    return user;
                }
            }
        }

        getLogger().error("Unable to log in from the mobile application to the '" + siteName + "' site. At least one population should be configured with credential provider compatible with the mobile app.");
        return null;
    }

    private UserIdentity authenticate(Map<String, Object> params, Request request, Parameters parameters, List<String> populationContexts, UserPopulation userPopulation) throws ParameterException
    {
        String siteName = parameters.getParameter("site");

        Set<String> contextsForUserPopulation = _populationContextHelper.getContextsForUserPopulation(userPopulation.getId());
        
        for (String context : populationContexts)
        {
            if (contextsForUserPopulation.contains(context))
            {
                UserIdentity user = _tryConnect(params, request, context, userPopulation);
                if (user != null)
                {
                    return user;
                }
            }
        }

        getLogger().error("Unable to log in from the mobile application to the '" + siteName + "' site, with population '" + userPopulation.getId() + "'.");
        return null;
    }
    
    private UserIdentity _tryConnect(Map<String, Object> params, Request request, String context, UserPopulation userPopulation)
    {
        List<CredentialProvider> credentialProviders = userPopulation.getCredentialProviders();
        
        for (int i = 0; i < credentialProviders.size(); i++)
        {
            CredentialProvider credentialProvider = credentialProviders.get(i);
            UserIdentity user = tryConnect(params, request, context, userPopulation, credentialProvider, i);
            if (user != null)
            {
                return user;
            }
        }
        
        return null;
    }

    /**
     * Try to authenticate the current mobile user
     * @param params the parameters issued by the mobile app
     * @param request the authentication request
     * @param context the population context
     * @param userPopulation the population
     * @param credentialProvider the {@link CredentialProvider}
     * @param credentialProviderIndex the index of the credentialProvider in the given userPopulation
     * @return the authenticated user, if any
     */
    protected abstract UserIdentity tryConnect(Map<String, Object> params, Request request, String context, UserPopulation userPopulation, CredentialProvider credentialProvider, int credentialProviderIndex);
}
