/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.mobileapp.observer;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.mobileapp.FeedHelper;
import org.ametys.plugins.mobileapp.PushNotificationManager;
import org.ametys.plugins.mobileapp.UserPreferencesHelper;
import org.ametys.plugins.mobileapp.action.GetEventTypesAction;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
/**
 * On validation, test each query to notify impacted users
 */
public class ProjectActivityObserver extends AbstractLogEnabled implements AsyncObserver, Serviceable
{
    private static final String __PROJECT_ENABLED_CONF_ID = "plugin.mobileapp.project.enabled";

    private FeedHelper _feedHelper;
    private UserPreferencesHelper _userPreferencesHelper;
    private PushNotificationManager _pushNotificationManager;
    private ProjectManager _projectManager;
    private AmetysObjectResolver _resolver;
    private ProjectMemberManager _projectMemberHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _feedHelper = (FeedHelper) manager.lookup(FeedHelper.ROLE);
        _userPreferencesHelper = (UserPreferencesHelper) manager.lookup(UserPreferencesHelper.ROLE);
        _pushNotificationManager = (PushNotificationManager) manager.lookup(PushNotificationManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        
        if (PluginsManager.getInstance().isPluginActive("workspaces"))
        {
            _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
            _projectMemberHelper = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        }
    }

    public boolean supports(Event event)
    {
        return PluginsManager.getInstance().isPluginActive("workspaces")
                && Config.getInstance().getValue(__PROJECT_ENABLED_CONF_ID, false, false)
                && org.ametys.plugins.repository.ObservationConstants.EVENT_ACTIVITY_CREATED.contains(event.getId());
    }

    public int getPriority()
    {
        return MAX_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        String activityId = (String) event.getArguments().get(org.ametys.plugins.repository.ObservationConstants.ARGS_ACTIVITY_ID);
        Activity activity = _resolver.resolveById(activityId);
        
        // Push notifications are only sent for allowed types
        if (GetEventTypesAction.EVENT_TYPES.containsKey(activity.getEventType()))
        {
            ActivityType activityType = activity.getActivityType();
            if (activityType instanceof AbstractWorkspacesActivityType)
            {
                Project project = _projectManager.getProject(activity.getValue(AbstractWorkspacesActivityType.PROJECT_NAME));
                getLogger().info("Listing push notification to send for activity on project '{}'", project.getId());
                
                Map<String, Object> activity2json = activity.toJSONForClient();
                
                Map<String, Object> projectJson = _feedHelper.projectToMap(project);
                
                // we only look for tokens among project members
                Set<ProjectMember> members = _projectMemberHelper.getProjectMembers(project, true);
                List<UserIdentity> users = members.stream().map(member -> member.getUser())
                                                           .filter(Objects::nonNull)
                                                           .map(User::getIdentity)
                                                           .toList();
                
                // Get all map of language => tokens for all users, and generate a new big map language => tokens
                Map<String, Map<UserIdentity, Set<String>>> langAndTokens = new HashMap<>();
                for (UserIdentity user : users)
                {
                    Map<String, Set<String>> tokensForUser = _userPreferencesHelper.getUserImpactedTokens(user, project, activity.getEventType());
                    for (Entry<String, Set<String>> tokensByLang : tokensForUser.entrySet())
                    {
                        Map<UserIdentity, Set<String>> tokens = langAndTokens.computeIfAbsent(tokensByLang.getKey(), __ -> new HashMap<>());
                        tokens.put(user, tokensByLang.getValue());
                    }
                }
                
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("Push tokens " + langAndTokens.entrySet().stream().map(e -> e.getKey() + ": " + e.getValue().size()).collect(Collectors.joining(", ")));
                }
                
                // Get the activities infos for each languages
                Map<String, Map<String, Object>> translatedNotificationContent = langAndTokens.keySet().stream()
                        .distinct()
                        .collect(Collectors.toMap(Function.identity(), lang -> _feedHelper.getActivityInfos(activity2json, projectJson, lang)));
                
                for (Entry<String, Map<UserIdentity, Set<String>>> entry : langAndTokens.entrySet())
                {
                    String lang = entry.getKey();
                    Map<UserIdentity, Set<String>> tokens = entry.getValue();
                    Map<String, Object> notificationData = translatedNotificationContent.get(lang);
                    
                    try
                    {
                        @SuppressWarnings("unchecked")
                        Map<String, Object> category = (Map<String, Object>) ((Map<String, Object>) notificationData.get("project")).get("category");
                        String categoryLabel = ((org.ametys.runtime.i18n.I18nizableText) category.get("title")).getLabel();
                        category.put("title", categoryLabel);
                    }
                    catch (Exception e)
                    {
                        // Nothing, this is just a hack do pass the notification framework jsonifier
                    }
                    
                    String title = project.getTitle();
                    String message = (String) notificationData.get("short-description");
                    _pushNotificationManager.pushNotifications(title, message, tokens, null);
                }
            }
        }
    }
}
