/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
/* Constants */
const CREDENTIALS = new javax.jcr.SimpleCredentials('ametys', []);
const ARCHIVES_SESSION = repository.login(CREDENTIALS, 'archives');
const XPATH_QUERY = "//element(*, ametys:content)[@ametys-internal:contentType = 'org.ametys.plugins.newsletter.Content.newsletter']";
const STRING_REPOSITORY_DATA_TYPE = org.ametys.plugins.repository.data.repositorydata.RepositoryData.STRING_REPOSITORY_DATA_TYPE

/* Imports */
const JCRRepositoryData = Java.type('org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData');

function __migrateAllNewslettersInternalData(session)
{
    let contents = Repository.query(XPATH_QUERY, true, session);
    let totalContents = contents.getSize();
    let handledContents = 0;
    for (let content of contents)
    {
        if (content instanceof org.ametys.cms.repository.ModifiableContent)
        {
            Content.migrate(
                content,
                [__migrateNewsletterInternalData.bind(null)],
                false,
                null,
                false,
                session == null
            );
        }
        else
        {
            logger.warn("Unable to migrate rich-texts of the content " + content + " because it is not modifiable. This content is ignored");
        }
        
        handledContents++;
        if (handledContents % 50 == 0)
        {
            logger.info("[" + handledContents + "/" + totalContents + "] handled newsletters");
        }
    }
    
    logger.info(handledContents + " handled newsletters");
}

function __migrateNewsletterInternalData(newsletter)
{
    try
    {
        __migrateBooleanInternalData(newsletter, "automatic");
        __migrateBooleanInternalData(newsletter, "sent");
        __migrateStringInternalData(newsletter, "category");
        __migrateStringInternalData(newsletter, "test-unique-id");
    }
    catch (e)
    {
        logger.error("An error occurred migrating internal data on newsletter " + newsletter.getId(), e);
        throw e;
    }
}

function __migrateBooleanInternalData(newsletter, dataName)
{
    let repositoryData = new JCRRepositoryData(newsletter.getNode());
    if (repositoryData.hasValue(dataName))
    {
        let booleanValue;
        if (repositoryData.getType(dataName) == STRING_REPOSITORY_DATA_TYPE)
        {
            let stringValue = repositoryData.getString(dataName);
            booleanValue = java.lang.Boolean.valueOf(stringValue);
        }
        else
        {
            booleanValue = repositoryData.getBoolean(dataName);
        }
            
        newsletter.getInternalDataHolder().setValue(dataName, booleanValue);
        repositoryData.removeValue(dataName);
    }
}

function __migrateStringInternalData(newsletter, dataName)
{
    let repositoryData = new JCRRepositoryData(newsletter.getNode());
    if (repositoryData.hasValue(dataName))
    {
        let value = repositoryData.getString(dataName);
        newsletter.getInternalDataHolder().setValue(dataName, value);
        repositoryData.removeValue(dataName);
    }
}

logger.debug("Handling internal data on newsletters in default workspace");
__migrateAllNewslettersInternalData(null);
logger.debug("Handling internal data on newsletters in archives workspace");
__migrateAllNewslettersInternalData(ARCHIVES_SESSION);
