/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.plugins.newsletter.NewsletterDAO;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryDAO;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;


/**
 * SAX events for category providers, categories and newsletters
 *
 */
public class GetNewslettersAction extends ServiceableAction
{
    /** The String representing the type of a newsletter category provider node */
    private static final String __PROVIDER_TYPE = "newsletter-category-provider";
    /** The String representing the type of a newsletter category node */
    private static final String __CATEGORY_TYPE = "newsletter-category";
    /** The String representing the type of a newsletter node */
    private static final String __NEWSLETTER_TYPE = "newsletter";
    /** The String representing the write-mode of a node */
    private static final String __MODE_WRITE = "write";
    /** The String representing the read-mode of a node */
    private static final String __MODE_READ = "read";

    private CategoryProviderExtensionPoint _categoryProviderEP;
    private NewsletterDAO _newsletterDAO;
    private CategoryDAO _categoryDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _categoryProviderEP = (CategoryProviderExtensionPoint) serviceManager.lookup(CategoryProviderExtensionPoint.ROLE);
        _newsletterDAO = (NewsletterDAO) serviceManager.lookup(NewsletterDAO.ROLE);
        _categoryDAO = (CategoryDAO) serviceManager.lookup(CategoryDAO.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String categoryID = (String) jsParameters.get("node");
        String siteName = (String) jsParameters.get("sitename");
        String lang = (String) jsParameters.get("lang");
        boolean categoriesOnly = (boolean) jsParameters.get("categoriesOnly");
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        
        if (StringUtils.isEmpty(categoryID) || "root".equals(categoryID))
        {
            Set<String> ids = _categoryProviderEP.getExtensionsIds();
            for (String id : ids)
            {
                CategoryProvider provider = _categoryProviderEP.getExtension(id);
                nodes.add(providerToJSON(provider, siteName, lang));
            }
        }
        else if (categoryID.startsWith("provider_"))
        {
            String id = categoryID.substring("provider_".length());
            CategoryProvider provider = _categoryProviderEP.getExtension(id);
            List<Category> childCategories = provider.getCategories(siteName, lang);
            
            for (Category child : childCategories)
            {
                nodes.add(categoryToJSON(child, provider, categoriesOnly));
            }
        }
        else // it is a newsletter category
        {
            Set<String> ids = _categoryProviderEP.getExtensionsIds();
            for (String id : ids)
            {
                CategoryProvider provider = _categoryProviderEP.getExtension(id);
                if (provider.hasCategory(categoryID))
                {
                    // sub categories
                    Category category = provider.getCategory(categoryID);
                    List<Category> childCategories = provider.getCategories(category.getId());
                    for (Category child : childCategories)
                    {
                        nodes.add(categoryToJSON(child, provider, categoriesOnly));
                    }
                    
                    // newsletters
                    if (!categoriesOnly)
                    {
                        AmetysObjectIterable<Content> childNewsletters  = provider.getNewsletters(category.getId(), siteName, lang);
                        for (Content content : childNewsletters)
                        {
                            nodes.add(newsletterToJSON(content, provider));
                        }
                    }
                }
            }
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("children", nodes);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    /**
     * Gets category provider's properties to JSON format
     * @param categoryProvider The category provider
     * @param siteName The site name
     * @param lang The lang
     * @return The category provider properties
     */
    protected Map<String, Object> providerToJSON(CategoryProvider categoryProvider, String siteName, String lang)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", "provider_" + categoryProvider.getId());
        infos.put("type", __PROVIDER_TYPE);
        infos.put("hasChildren", !categoryProvider.getCategories(siteName, lang).isEmpty());
        infos.put("mode", categoryProvider.isWritable() ? __MODE_WRITE : __MODE_READ);
        infos.put("title", categoryProvider.getLabel());
        infos.put("description", categoryProvider.getDescription());
        
        return infos;
    }
    
    /**
     * Gets category's properties to JSON format
     * @param category The category
     * @param categoryProvider The category provider
     * @param categoriesOnly true if only categories are displayed
     * @return The category properties
     */
    protected Map<String, Object> categoryToJSON(Category category, CategoryProvider categoryProvider, boolean categoriesOnly)
    {
        Map<String, Object> infos = new HashMap<>();
        
        String id = category.getId();
        String siteName = category.getSiteName();
        boolean hasChildren = categoryProvider.hasChildren(id)
                           || !categoriesOnly && categoryProvider.hasNewsletters(id, siteName, category.getLang());
        
        infos.putAll(_categoryDAO.getCategory(id));
        
        infos.put("type", __CATEGORY_TYPE);
        infos.put("hasChildren", hasChildren);
        
        return infos;
    }
    
    /**
     * Gets newsletter's properties to JSON format
     * @param newsletter The newsletter
     * @param categoryProvider The category provider
     * @return The newsletter's properties
     */
    protected Map<String, Object> newsletterToJSON(Content newsletter, CategoryProvider categoryProvider)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.putAll(_newsletterDAO.getNewsletterProperties(newsletter));
        infos.put("type", __NEWSLETTER_TYPE);
        infos.put("hasChildren", false);
        
        return infos;
    }
}
