/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.analytics;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.web.analytics.WebAnalyticsHelper;
import org.ametys.web.analytics.WebAnalyticsProvider;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Helper which provides web analytics URI building.
 */
public class WebAnalyticsXsltHelper implements Serviceable
{
    private static WebAnalyticsHelper _webAnalyticsHelper;
    private static SiteManager _siteManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _webAnalyticsHelper = (WebAnalyticsHelper) manager.lookup(WebAnalyticsHelper.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    /**
     * Get an event image URI.
     * @param siteName the site name
     * @param category the event category.
     * @param action the event action.
     * @param label the event label.
     * @return the event image URI.
     */
    public static String eventImageUri(String siteName, String category, String action, String label)
    {
        return eventImageUri(siteName, category, action, label, 0);
    }
    
    /**
     * Get an event image URI.
     * @param siteName the site name
     * @param category the event category.
     * @param action the event action.
     * @param label the event label.
     * @param value the event value.
     * @return the event image URI.
     */
    public static String eventImageUri(String siteName, String category, String action, String label, int value)
    {
        Site site = _siteManager.getSite(siteName);
        if (_isNewsletterTrackingEnabled(site))
        {
            WebAnalyticsProvider provider = _webAnalyticsHelper.getSelectedProvider(site);
            return provider.getEventImageUri(site, category, action, label, value, true);
        }
        
        return null;
    }
    
    /**
     * Get an event link campaign params
     * @param siteName the site name
     * @param campaign the event campaign
     * @param medium the event medium
     * @param source the event source
     * @return the event link campaign params
     */
    public static String eventLinkCampaignParams(String siteName, String campaign, String medium, String source)
    {
        Site site = _siteManager.getSite(siteName);
        if (_isNewsletterTrackingEnabled(site))
        {
            WebAnalyticsProvider provider = _webAnalyticsHelper.getSelectedProvider(site);
            return provider.getEventLinkCampaignParams(site, campaign, medium, source);
        }
        
        return null;
    }
    
    private static boolean _isNewsletterTrackingEnabled(Site site)
    {
        return site.getValue("newsletter-enable-tracking", false, true);
    }
}
